/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
******************************************************************************
*
*	\file		CAENAllocator.h
*	\brief		Utilities to handle memory allocation in C minimizing memory leaks
*	\author		Francesco Pepe
*
******************************************************************************/

#ifndef CAEN_INCLUDE_CAENALLOCATOR_H_
#define CAEN_INCLUDE_CAENALLOCATOR_H_

/*!
* \defgroup Allocator Allocator
* \brief Utilities to handle memory allocation in C minimizing memory leaks
* It provides a malloc-like interface, but keeps an internal list of the
* allocated pointers. A function can be used to free them all at once, and
* a function to free all pointers and return a given value is provided. This
* allows a calling function to quickly allocate resources and return a value
* free-ing them.
* \author Francesco Pepe
* \date 2023
* \copyright LGPL-3.0-or-later
* \{ */

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>

#include <types/CAENAllocatorTypes.h>
#include <CAENUtility.h>

/*!
* Creates a new empty c_allocator_t
* \return			the new c_allocator_t
*/
c_nodiscard
CAEN_UTILITY_DLLAPI c_allocator_t CAEN_UTILITY_API c_allocator_create();

/*!
* Adds the allocated 'resource' to its internal list, using function 'df' to destroy
* it when needed.
* \param[in]		allocator			pointer to the allocator struct
* \param[in]		df					function used to destroy the resource
* \param[in]		resource			the allocated resource itself (for example the result 'malloc' or 'fopen', etc
* \return			return 'resource' in case of success or NULL otherwhise.
* \note				if resource is NULL, the function fails. If an internal error occurs, 'resource' is automatically cleaned using 'df'.
*/
c_nodiscard
CAEN_UTILITY_DLLAPI void* CAEN_UTILITY_API c_allocator_resource_create(c_allocator_t* allocator, c_resource_destroy_function df, void* resource);

/*!
* Malloc-s a new element of the given size and return it
* \param[in]		allocator			pointer to the allocator struct
* \param[in]		size				the size of memory to allocate
* \return			the pointer to allocated memory
*/
c_nodiscard c_declspec_allocator c_attribute_malloc c_attribute_alloc_size(2)
CAEN_UTILITY_DLLAPI void* CAEN_UTILITY_API c_allocator_malloc(c_allocator_t* allocator, size_t size);

/*!
* Calloc-s a new element of the given size, sets it to zero and return it.
* \param[in]		allocator			pointer to the allocator struct
* \param[in]		nmemb				the number of elements to allocate
* \param[in]		size				the size of a single element
* \return			the pointer to allocated memory
*/
c_nodiscard c_declspec_allocator c_attribute_malloc c_attribute_alloc_size(2, 3)
CAEN_UTILITY_DLLAPI void* CAEN_UTILITY_API c_allocator_calloc(c_allocator_t* allocator, size_t nmemb, size_t size);

// TODO: realloc: search the list for an element and realloc it if found. Manage pointer changes and case of new allocation.

/*!
* Return the number of allocation errors that happened trying to allocate memory with this allocator
* \param[in]		allocator			pointer to the allocator struct
* \return			the number of allocation errors
*/
c_nodiscard
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_allocator_error_count(const c_allocator_t* allocator);

/*!
* Removes the given pointer from the internal list. This is useful if you need to keep allocated the memory while
* you need to free other pointers.
* \note	to slightly boost performance, it would be advisable to allocate such resources after all other, because any
* newly allocated resource is added at the beginning of the internal linkedlist, and the list is iterated from
* first element to last for element find.
* \param[in]		allocator			pointer to the allocator struct
* \param[in]		ptr					pointer to the resource to release
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_allocator_release(c_allocator_t* allocator, void* ptr);

/*!
* Clears the list of pointer without free-ing them
* \param[in]		allocator			pointer to the allocator struct
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_allocator_clear(c_allocator_t* allocator);

/*!
* Free-es the given pointer using its destroy function and removes it from the internal list. This is useful if you
* want to destroy just one of the allocated resources.
* \note	to slightly boost performance, it would be advisable to allocate such resources after all other, because any
* newly allocated resource is added at the beginning of the internal linkedlist, and the list is iterated from
* first element to last for element find.
* \param[in]		allocator			pointer to the allocator struct
* \param[in]		allocator			pointer to the resource to be destroyed
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_allocator_free(c_allocator_t* allocator, void* ptr);

/*!
* Free-es the entire list of allocated pointers
* \param[in]		allocator			pointer to the allocator struct
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_allocator_freeall(c_allocator_t* allocator);

/*!
* Free-es the entire list of allocated pointers and evaluates to the given value.
* To be used with return like:
* \code{.cpp}
* return c_allocator_freeall_and_return(&allocator, ret);
* \endcode
* \param[in]		A			pointer to the allocator struct
* \param[in]		R			the value to return
*/
#define c_allocator_freeall_and_return(A, R) (c_allocator_freeall(A), R)

// TODO: free a single element: search the linkedlist for such element, free and remove it from list

/*! \} */

#ifdef __cplusplus
}
#endif

#endif // CAEN_INCLUDE_CAENALLOCATOR_H_
