/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
******************************************************************************
*
*	\file		CAENLinkedList.h
*	\brief		Utilities to implement linked lists
*	\author		Francesco Pepe
*
******************************************************************************/

#ifndef CAEN_INCLUDE_CAENLINKEDLIST_H_
#define CAEN_INCLUDE_CAENLINKEDLIST_H_

/*!
* \defgroup LinkedList Linked List
* \brief Utilities to implement linked list
* \author Francesco Pepe
* \date 2023
* \copyright LGPL-3.0-or-later
* \{ */

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>

#include <types/CAENLinkedListTypes.h>
#include <CAENUtility.h>

/*!
* Creates a new empty c_linkedlist_t
* \return			the new c_linkedlist_t
*/
c_nodiscard
CAEN_UTILITY_DLLAPI c_linkedlist_t CAEN_UTILITY_API c_linkedlist_create();

/*!
* Inserts the given newElement AFTER the given node
* \param[in]		node				the node after which insert the new element
* \param[in]		newElement			the data of the new element to be added
* \return			the new c_listnode_t in case of success, NULL otherwise.
*/
c_nodiscard
CAEN_UTILITY_DLLAPI c_listnode_t* CAEN_UTILITY_API c_linkedlist_insert_after(c_listnode_t* node, void* newElement);

/*!
* Inserts the given newElement BEFORE the given node
* \param[in]		node				the node before which insert the new element
* \param[in]		newElement			the data of the new element to be added
* \return			the new c_listnode_t in case of success, NULL otherwise.
*/
c_nodiscard
CAEN_UTILITY_DLLAPI c_listnode_t* CAEN_UTILITY_API c_linkedlist_insert_before(c_listnode_t* node, void* newElement);

/*!
* Inserts the given newHead to the end of the list
* \param[in]		link				the linked list
* \param[in]		newElement			the data of the new element to be added
* \return			c_Utility_ErrorCode_Success in case of success, error code otherwise
*/
c_nodiscard
CAEN_UTILITY_DLLAPI c_Utility_ErrorCode_t CAEN_UTILITY_API c_linkedlist_insert_last(c_linkedlist_t* list, void* newElement);

/*!
* Inserts the given newHead to the begin of the list
* \param[in]		link				the linked list
* \param[in]		newElement			the data of the new element to be added
* \return			c_Utility_ErrorCode_Success in case of success, error code otherwise
*/
c_nodiscard
CAEN_UTILITY_DLLAPI c_Utility_ErrorCode_t CAEN_UTILITY_API c_linkedlist_insert_first(c_linkedlist_t* list, void* newElement);

/*!
* Deletes the given node, setting the references of previous and next elements
* and updating head and tail of linkedlist if necessary.
* NOTE: the data pointer is never free-ed.
* \param[in]		list	the linked list
* \param[in]		node	the node to be deleted
* \return			'node->data' in case of success, NULL otherwise
*/
c_nodiscard
CAEN_UTILITY_DLLAPI void* CAEN_UTILITY_API c_linkedlist_delete(c_linkedlist_t* list, c_listnode_t* node);

/*!
* Deletes the first node starting from head whose 'data' field equals the given data.
* 'data' equality is evaluated using the given comparator.
* NOTE: references to head and tails are automatically updated.
* NOT automatically updated!
* NOTE: the data pointer is never free-ed.
* \param[in]	list	the linked list
* \param[in]	data	the data to be found and removed from list
* \param[in]	cmp		the data comparator. Should return non-zero when its two arguments are considered equal.
*						NOTE: i-th element of list is passed to cmp as first argument, given 'data' as second.
* \return		'data' in case of success, NULL otherwise
*/
c_nodiscard
CAEN_UTILITY_DLLAPI void* CAEN_UTILITY_API c_linkedlist_delete_data_compare(c_linkedlist_t* list, void* data, c_listdata_comparator_t cmp);

/*!
* Deletes the first node starting from head whose 'data' field equals the given data.
* NOTE: references to head and tails are automatically updated.
* NOT automatically updated!
* NOTE: the data pointer is never free-ed.
* \param[in]		list				the linked list
* \param[in]		data				the data to be found and removed from list
* \return			'data' in case of success, NULL otherwise
*/
c_nodiscard
CAEN_UTILITY_DLLAPI void* CAEN_UTILITY_API c_linkedlist_delete_data(c_linkedlist_t* list, void* data);

/*!
* Deletes the last element of the given c_linkedlist_t.
* NOTE: the data pointer is never free-ed.
* \param[in]		list				the linked list
* \return			'data' filed of the removed element in case of success, NULL otherwise
*/
c_nodiscard
CAEN_UTILITY_DLLAPI void* CAEN_UTILITY_API c_linkedlist_delete_last(c_linkedlist_t* list);

/*!
* Deletes the first element of the given c_linkedlist_t.
* NOTE: the data pointer is never free-ed.
* \param[in]		list				the linked list
* \return			'data' filed of the removed element in case of success, NULL otherwise
*/
c_nodiscard
CAEN_UTILITY_DLLAPI void* CAEN_UTILITY_API c_linkedlist_delete_first(c_linkedlist_t* list);

/*! \} */

#ifdef __cplusplus
}
#endif

#endif // CAEN_INCLUDE_CAENURL_H_
