/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENLogger.h
*	\brief		Logger implemented in C
*	\author
*
******************************************************************************/

#ifndef CAEN_INCLUDE_CAENLOGGER_H_
#define CAEN_INCLUDE_CAENLOGGER_H_

/*!
* \defgroup LogFunctions Logger functions
* \brief Log related functions
*/

/*!
* \defgroup LogMacros Logger
* \brief Log related macros
*/

#ifdef __cplusplus
extern "C" {
#endif

#include <types/CAENLoggerTypes.h>
#include <CAENUtility.h>
#include <stdarg.h>
#include <inttypes.h> // includes also stdint.h
#include <stdio.h>

#define LOG_ENV_FILENAME		"CAEN_LOG_FILENAME"
#define LOG_ENV_LEVEL			"CAEN_LOG_LEVEL"

/*!
* Set the start time.
* \param[in]		time			time in milliseconds since 00:00:00 UTC, January 1, 1970.
* \ingroup LogFunctions
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_lsetst(uint64_t time);

/*!
* Set a new file, already opened. The original file is not close, but its pointer stored. It's up to the caller to close the new file at the end.
* \param[in]		file			a pointer to FILE, already opened.
* \return			::c_logger_Success (0) in case of success. Error codes specified in #c_logger_ErrorCode_t.
* \ingroup LogFunctions
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_lchangefile(FILE *file);

/*!
* Resume the original file after a call to c_lchangefile().
* \return			::c_logger_Success (0) in case of success. Error codes specified in #c_logger_ErrorCode_t.
* \ingroup LogFunctions
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_lresumefile(void);

/*!
* Close and clear the logging system.
* \ingroup LogFunctions
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_ldeinit(void);

/*!
* Set the Severity Mask
* \param[in]		sevMask			the severity mask, a combination of bits specified in #c_logger_Severity
* \return			::c_logger_Success (0) in case of success. Error codes specified in #c_logger_ErrorCode_t.
* \ingroup LogFunctions
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_lsetsm(uint32_t sevMask);

/*!
* Set the current Severity Mask
* \param[out]		sevMask			the severity mask, a combination of bits specified in #c_logger_Severity
* \return			::c_logger_Success (0) in case of success. Error codes specified in #c_logger_ErrorCode_t.
* \ingroup LogFunctions
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_lgetsm(uint32_t *sevMask);

/*!
* Main logger function. If not initialized, the function will configure the logger. Variable argument list version.
* \param[in,out]	locallogger		the local logger
* \param[in]		s				the severity mask
* \param[in]		line			the line, to be filled with standard preprocessor define `__LINE__`
* \param[in]		format			string that contains a format string that follows the same specifications as format in printf
* \param[in]		args			a variable arguments list initialized with va_start
* \return			::c_logger_Success (0) in case of success. Error codes specified in #c_logger_ErrorCode_t.
* \ingroup LogFunctions
*/
c_attribute_format(4, 0)
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_lvprintf(const c_locallogger_t *locallogger, c_logger_Severity s, int32_t line, const char *__restrict format, va_list args);

/*!
* Main logger function. If not initialized, the function will configure the logger.
* \param[in,out]	locallogger		the local logger
* \param[in]		s				the severity mask
* \param[in]		line			the line, to be filled with standard preprocessor define `__LINE__`
* \param[in]		format			string that contains a format string that follows the same specifications as format in printf
* \param[in]		...				sequence of additional arguments
* \return			::c_logger_Success (0) in case of success. Error codes specified in #c_logger_ErrorCode_t.
* \ingroup LogFunctions
*/
c_attribute_format(4, 5)
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_lprintf(const c_locallogger_t *locallogger, c_logger_Severity s, int32_t line, const char *__restrict format, ...);

/*!
* Define a static instance of c_locallogger_t, setting the filename and the module tree.
* \param[in]		fname			the filename
* \param[in]		mname			the module tree
* \ingroup LogMacros
*/
#define INIT_C_LOGGER(fname, mname)		static const c_locallogger_t __locallogger = { (fname), (mname) }

/*!
* Wrapper to c_lvprintf(), with a reference to the local logger initialized by INIT_C_LOGGER() added as first parameter.
* \param[in]		s				the severity mask
* \param[in]		format			string that contains a format string that follows the same specifications as format in printf
* \param[in]		args			a variable arguments list initialized with va_start
* \ingroup LogMacros
*/
#define vlogMsg(s, format, args)		c_lvprintf(&(__locallogger), (s), (format), (args))

/*!
* Wrapper to c_lprintf(), with a reference to the local logger initialized by INIT_C_LOGGER() added as first parameter.
* \warning \p format must be included in the \p "__VA_ARGS__"  to avoid warnings in \c "gcc -std=c99 -pedantic" complaining about at least one variadic argument
* \param[in]		s				the filename
* \param[in]		...				the format, followed by an optional sequence of additional arguments.
* \ingroup LogMacros
*/
#define logMsg(s, ...)					c_lprintf(&(__locallogger), (s), __LINE__, __VA_ARGS__)

#ifdef __cplusplus
}
#endif

#endif // CAEN_INCLUDE_CAENLOGGER_H_
