/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2014 rxi
*
*	This file is part of the CAEN Utility.
*
*	This file is distributed under the MIT License.
*
*	SPDX-License-Identifier: MIT
*
***************************************************************************//*!
*
*	\file		CAENMap.h
*	\brief		Hash table. Implementation from rxi's "map" project.
*	\author		Giovanni Cerretani, rxi
*
******************************************************************************/

#ifndef CAEN_INCLUDE_CAENMAP_H_
#define CAEN_INCLUDE_CAENMAP_H_

/*! 
* \defgroup MapFunctions Maps
* \brief Hash table implementation in C
* \author rxi
* \date 2014
* \copyright MIT license
*/

#ifdef __cplusplus
extern "C" {
#endif

#include <types/CAENMapTypes.h>
#include <CAENUtility.h>
#include <stdint.h>

/*!
* Initialize a new map with pre-allocation.
* \param[in]		m					a pointer to an allocated map
* \param[in]		nreserved			an integer with the number of buckets to be pre-allocated. If is not a power of 2 or 0, no pre-allocation is done. It is not necessary. It can improve the speed but, on the other hand, could allocate useless space.
* \return			::c_Utility_ErrorCode_Success (0) in case of success. Error codes specified in #c_Utility_ErrorCode_t.
* \ingroup MapFunctions
*/
#define c_map_init(m, nreserved)					(((m) != NULL) ? _map_init(&(m)->base, (nreserved)) : c_Utility_ErrorCode_Map)

/*!
* Initialize a new map.
* \param[in]		m					a pointer to an allocated map
* \return			::c_Utility_ErrorCode_Success (0) in case of success. Error codes specified in #c_Utility_ErrorCode_t.
* \ingroup MapFunctions
*/
#define c_map_init_default(m)						c_map_init((m), 0)

/*!
* Deinitialize a map already initialized by c_map_init().
* \param[in]		m					a pointer to an initialized map
* \ingroup MapFunctions
*/
#define c_map_deinit(m)								do { if ((m) != NULL) _map_deinit(&(m)->base); } while (0)

/*!
* Get an element from map given its key
* \param[in]		m					a pointer to an initialized map
* \param[in]		key					a string with the key
* \return			a pointer to the element. NULL in case of error.
* \ingroup MapFunctions
*/
#define c_map_get(m, key)							(((m) != NULL) ? (m)->ref = _map_get(&(m)->base, (key)) : NULL)

/*!
* Put an element into map with a given key
* \param[in]		m					a pointer to an initialized map
* \param[in]		key					a string with the key
* \param[in]		value				the element
* \return			::c_Utility_ErrorCode_Success (0) in case of success. Error codes specified in #c_Utility_ErrorCode_t.
* \ingroup MapFunctions
*/
#define c_map_set(m, key, value)					(((m) != NULL) ? ((m)->tmp = (value), _map_set(&(m)->base, (key), &(m)->tmp, sizeof((m)->tmp))) : c_Utility_ErrorCode_Map)

/*!
* Remove an element from map given its key
* \param[in]		m					a pointer to an initialized map
* \param[in]		key					a string with the key
* \ingroup MapFunctions
*/
#define c_map_remove(m, key)						do { if ((m) != NULL) _map_remove(&(m)->base, (key)); } while (0)

/*!
* Get the number of elements in a map
* \param[in]		m					a pointer to an initialized map
* \return			the number of nodes
* \ingroup MapFunctions
*/
#define c_map_size(m)								(((m) != NULL) ? _map_size(&(m)->base) : 0)

/*!
* Define a new iterator
* \return			an interator
* \ingroup MapFunctions
*/
#define c_map_iter()								_map_iter()

/*!
* Get the next key of a map, given an iterator.
* \param[in]		m					a pointer to an initialized map
* \param[in,out]	iter				a pointer to an iterator initialized by c_map_iter()
* \return			the next key
* \ingroup MapFunctions
*/
#define c_map_next(m, iter)							(((m) != NULL) ? _map_next(&(m)->base, (iter)) : NULL)

/***************************
* Function declaration, not to be used directly.
***************************/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API _map_init(c_map_base_t *m, size_t nreserved);

CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API _map_deinit(c_map_base_t *m);

c_nodiscard
CAEN_UTILITY_DLLAPI void * CAEN_UTILITY_API _map_get(const c_map_base_t *m, const char *key);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API _map_set(c_map_base_t *m, const char *key, const void *value, size_t vsize);

CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API _map_remove(c_map_base_t *m, const char *key);

c_nodiscard
CAEN_UTILITY_DLLAPI size_t CAEN_UTILITY_API _map_size(const c_map_base_t *m);

c_nodiscard
CAEN_UTILITY_DLLAPI c_map_iter_t CAEN_UTILITY_API _map_iter(void);

c_nodiscard
CAEN_UTILITY_DLLAPI const char * CAEN_UTILITY_API _map_next(const c_map_base_t *m, c_map_iter_t *iter);

#ifdef __cplusplus
}
#endif

#endif // CAEN_INCLUDE_CAENMAP_H_
