/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENMultiplatform.h
*	\brief		Generic wrappers to platform-dependent functions
*	\author
*
******************************************************************************/

#ifndef CAEN_INCLUDE_CAENMULTIPLATFORM_H_
#define CAEN_INCLUDE_CAENMULTIPLATFORM_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <types/CAENMultiplatformTypes.h>
#include <CAENUtility.h>

/*!
* Macro to get the size of a C array, with error generated if usef for pointers
* \warning MSVC error works only when compiled as C++
* \param[in] A		a C array
* \return the size
*/
#ifdef _WIN32
#define c_arraysize(A)				_countof(A)
#else
#define c_arraysize(A)				(sizeof(A)/sizeof((A)[0])+sizeof(__typeof__(int[1-2*!!__builtin_types_compatible_p(__typeof__(A),__typeof__(&A[0]))]))*0)
#include <endian.h>
#endif

/*! \defgroup CAENMalloc Heap allocator wrappers, optimized for the current system
* \brief Wrappers to malloc/realloc/calloc/free
* \{ */
c_nodiscard c_declspec_allocator c_attribute_malloc c_attribute_alloc_size(1)
CAEN_UTILITY_DLLAPI void* CAEN_UTILITY_API c_malloc(size_t size);

c_nodiscard c_declspec_allocator c_attribute_alloc_size(2)
CAEN_UTILITY_DLLAPI void* CAEN_UTILITY_API c_realloc(void *ptr, size_t size);

c_nodiscard c_declspec_allocator c_attribute_malloc c_attribute_alloc_size(1, 2)
CAEN_UTILITY_DLLAPI void* CAEN_UTILITY_API c_calloc(size_t nmemb, size_t size);

CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_free(void *ptr);

c_nodiscard
CAEN_UTILITY_DLLAPI size_t CAEN_UTILITY_API c_malloc_size(void *ptr);

//! Wrapper to `strdup(str)`, with check for `NULL` arguments. To be freed with c_free(). \return a pointer to the newly allocated string. `NULL` in case of error.
c_nodiscard c_declspec_allocator c_attribute_malloc
CAEN_UTILITY_DLLAPI char* CAEN_UTILITY_API c_strdup(const char *str);

//! Wrapper to `strndup(str, size)`, with check for `NULL` arguments. To be freed with c_free(). \return a pointer to the newly allocated string. `NULL` in case of error.
c_nodiscard c_declspec_allocator c_attribute_malloc
CAEN_UTILITY_DLLAPI char* CAEN_UTILITY_API c_strndup(const char *str, size_t size);
/*! \} */

/*! \defgroup MemFunctions Memory related function
* \brief Wrappers to memset/memcpy/memmove
* \note Defined static inline to help optimization: it is fine, as they use only standard functions.
* \warning Cannot be declared as macros because arguments could be evaluated wtice
* \{ */
//! Wrapper to `memset(dest, 0, size)`, with check for `NULL` arguments. \return the value of dest
static inline void* c_zeromem(void* dest, size_t size) {
	return (dest == NULL) ? dest : memset(dest, 0, size);
}
//! Wrapper to `memset(dest, val, size)`, with check for `NULL` arguments. \return the value of dest
static inline void* c_memset(void* dest, int val, size_t size) {
	return (dest == NULL) ? dest : memset(dest, val, size);
}
//! Wrapper to `memcpy(dest, src, size)`, with check for `NULL` arguments. \return the value of dest
static inline void* c_memcpy(void* __restrict dest, const void* __restrict src, size_t size) {
	return (dest == NULL || src == NULL) ? dest : memcpy(dest, src, size);
}
//! Wrapper to `memmove(dest, src, size)`, with check for `NULL` arguments. \return the value of dest
static inline void* c_memmove(void* dest, const void* src, size_t size) {
	return (dest == NULL || src == NULL) ? dest : memmove(dest, src, size);
}
/*! \} */

/*! \defgroup ProcessesFunctions Process related functions
* \brief Platform independent functions to manage processes
* \{ */
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_getCurrentEnvironment(c_environment_t *dest);

c_nodiscard
CAEN_UTILITY_DLLAPI c_Process_t* CAEN_UTILITY_API c_newProc(void);

CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_freeProc(c_Process_t *proc);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_setProcExec(const char *exec, c_Process_t *proc);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_addProcArg(const char *arg, c_Process_t *proc);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_addProcEnv(const char *varname, const char *varvalue, c_Process_t *proc);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_startProcess(c_Process_t *proc);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_endProcess(c_Process_t *proc);

c_nodiscard
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_getNCPU(void);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_pause(void);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_killProcess(const char *procName);

/*! \} */

/*! \defgroup DLFunctions Load functions dynamically
* \brief Platform independent API to load / close functions from libraries dynamically
* \{ */
c_attribute_nonnull(1, 2)
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_dlload(const char* libName, c_libhandle_t *libHandle);

c_attribute_nonnull(2, 3)
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_dfload(c_libhandle_t hLib, const char* fcnName, c_fcnhandle_t *funcPtr);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_dlclose(c_libhandle_t hLib);
/*! \} */

/*! \defgroup BitManipulation Bit manipulation functions
* \brief Platform independent bit manipulation functions
* \{ */
/*! Count trailing zeros.
* \param[in]	value			a 32 bit unsigned integer
* \return		the number of trailing zeros. If value is 0, returns 32.
*/
c_nodiscard c_attribute_pure
CAEN_UTILITY_DLLAPI uint32_t CAEN_UTILITY_API c_ctz(uint32_t value);

/*! Count leading zeros.
* \param[in]	value			a 32 bit unsigned integer
* \return		the number of leading zeros. If value is 0, returns 32.
*/
c_nodiscard c_attribute_pure
CAEN_UTILITY_DLLAPI uint32_t CAEN_UTILITY_API c_clz(uint32_t value);

/*! Count bits set.
* \param[in]	value			a 32 bit unsigned integer
* \return		the number of bits set.
*/
c_nodiscard c_attribute_pure
CAEN_UTILITY_DLLAPI uint32_t CAEN_UTILITY_API c_popcount(uint32_t value);

/*! Count trailing zeros.
* \param[in]	value			a 64 bit unsigned integer
* \return		the number of trailing zeros. If value is 0, returns 64.
*/
c_nodiscard c_attribute_pure
CAEN_UTILITY_DLLAPI uint32_t CAEN_UTILITY_API c_ctz64(uint64_t value);

/*! Count leading zeros.
* \param[in]	value			a 64 bit unsigned integer
* \return		the number of leading zeros. If value is 0, returns 64.
*/
c_nodiscard c_attribute_pure
CAEN_UTILITY_DLLAPI uint32_t CAEN_UTILITY_API c_clz64(uint64_t value);

/*! Count bits set.
* \param[in]	value			a 64 bit unsigned integer
* \return		the number of bits set.
*/
c_nodiscard c_attribute_pure
CAEN_UTILITY_DLLAPI uint32_t CAEN_UTILITY_API c_popcount64(uint64_t value);
/*! \} */

/*! \defgroup direntWindows dirent.h implementation for Windows
* \brief Implementation of "dirent.h" for Windows, using "Henney's POSIX Directory Browsing API for Windows"
* \author Kevlin Henney
* \date 1997, 2003
* \copyright	Copyright Kevlin Henney, 1997, 2003. All rights reserved.
*				Permission to use, copy, modify, and distribute this software and its
*				documentation for any purpose is hereby granted without fee, provided
*				that this copyright and permissions notice appear in all copies and
*				derivatives.
*				This software is supplied "as is" without express or implied warranty.
*				But that said, if there are any problems please get in touch.
* \{ */
//! \warn Not marked CAEN_UTILITY_API (meaningful only on 32-bit Windows), likely an original sin more than a choice.
c_nodiscard
CAEN_UTILITY_DLLAPI DIR* c_opendir(const char* name);

CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_closedir(DIR* dir);

//! \warn Not marked CAEN_UTILITY_API (meaningful only on 32-bit Windows), likely an original sin more than a choice.
c_nodiscard
CAEN_UTILITY_DLLAPI struct dirent* c_readdir(DIR* dir);

CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_rewinddir(DIR* dir);

#ifdef _WIN32 // Windows (deprecated macros)
#define opendir(name)				c_opendir(name)
#define closedir(dir)				c_closedir(dir)
#define readdir(dir)				c_readdir(dir)
#define rewinddir(dir)				c_rewinddir(dir)
#endif
/*! \} */

/*! \defgroup conioLinux conio.h implementation for Linux
* \brief Implementation of "conio.h" for Linux
* \pre Linux only
* \{ */
c_nodiscard
CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_getch(void);

c_nodiscard
CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_kbhit(void);

#ifndef _WIN32 // Linux (deprecated macros)
#define _getch()					c_getch()
#define _kbhit()					c_kbhit()
#endif
/*! \} */

/*! \defgroup posixMacro POSIX functions on Windows
* \brief Macros to use some POSIX functions with their Windows equivalent.
* \{ */
c_nodiscard
CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_access(const char* pathname, int mode);

CAEN_UTILITY_DLLAPI char* CAEN_UTILITY_API c_getcwd(char* buf, size_t size);

CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_chdir(const char* path);

CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_rmdir(const char* path);

c_nodiscard
CAEN_UTILITY_DLLAPI int64_t CAEN_UTILITY_API c_getpid(void);

c_nodiscard
CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_strcasecmp(const char* s1, const char* s2);

c_nodiscard
CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_strncasecmp(const char* s1, const char* s2, size_t n);
/*! \} */

c_nodiscard c_declspec_allocator c_attribute_malloc
CAEN_UTILITY_DLLAPI char* CAEN_UTILITY_API c_getExpandedString(const char *filename); //! Also allocate the char*, that should be freed with c_free()

#define c_freeExpandedString(str)			c_free(str) //! Deprecated macro

c_nodiscard
CAEN_UTILITY_DLLAPI bool CAEN_UTILITY_API c_checkFileExists(const char *fname);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_getDirName(char *dest, const char *source);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_getBaseName(char *dest, const char *source);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_remAllFiles(const char *dir);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_mkdir(const char *path);

//! Same of c_mkdir() but creates also intermediate directories
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_mkdirp(const char *dir);

/*! \defgroup FloatFunctions Floating point manipulation
* \brief Platform independent floating point manipulation functions
* \{ */
/*! To send double precision floating point numbers on network (IEEE 754 compliant):
* split double into mantissa, exponent and floating-point classification to be read with c_ldexp().
* \param[in]	value			the value
* \param[out]	exponent		the exponent
* \param[out]	classification	the classification, specified in #c_Float_Classification_t
* \return		the mantissa
*/
c_nodiscard
CAEN_UTILITY_DLLAPI int64_t CAEN_UTILITY_API c_frexp(double value, int16_t *exponent, int8_t *classification);

/*! To receive double precision floating point numbers on network (IEEE 754 compliant):
* merge mantissa, exponent and floating-point classification provided by c_frexp() into double.
* \param[in]	mantissa		the mantissa
* \param[in]	exponent		the exponent
* \param[in]	classification	the classification, specified in #c_Float_Classification_t
* \return		the value
*/
c_nodiscard c_attribute_pure
CAEN_UTILITY_DLLAPI double CAEN_UTILITY_API c_ldexp(int64_t mantissa, int16_t exponent, int8_t classification);
/*! \} */

/*! \defgroup TimeFunctions Time related functions
* \brief Platform independent functions to manage time
* \{ */
c_nodiscard
CAEN_UTILITY_DLLAPI uint64_t CAEN_UTILITY_API c_get_time(void); //!< Get time in milliseconds since 00:00:00 UTC, January 1, 1970.

c_attribute_nonnull(1)
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_GetLocalTime(c_systemtime_t *t);

c_attribute_nonnull(1)
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_GetSystemTime(c_systemtime_t *t);

c_nodiscard c_attribute_nonnull(1, 2)
CAEN_UTILITY_DLLAPI double CAEN_UTILITY_API c_DiffSystemTime_sec(const c_systemtime_t *t1, const c_systemtime_t *t2);

c_nodiscard
CAEN_UTILITY_DLLAPI uint64_t CAEN_UTILITY_API c_getCurrentTimeRepresentation(void);

#define c_CopySystemTime(src, dest)			(*(dest) = *(src)) //! Deprecated macro
/*! \} */

/*! \defgroup HtoNFunctions Portable host to Network conversion functions
* \brief Platform independent functions to convert integer formats
* \note Defined static inline to help optimization: it is fine, as they use only standard functions.
* \pre On Linux, these macros require __DEFAULT_SOURCE to be defined ( DEFAULT_SOURCE is defined also by _GNU_SOURCE and `-std=gnuXY`)
* \sa https://man7.org/linux/man-pages/man3/endian.3.html
* \{ */
c_nodiscard
static inline uint16_t c_hton16(uint16_t x) {
#ifdef _WIN32
	return htons(x);
#else
	return htobe16(x);
#endif
}
c_nodiscard
static inline uint16_t c_ntoh16(uint16_t x) {
#ifdef _WIN32
	return ntohs(x);
#else
	return be16toh(x);
#endif
}
c_nodiscard
static inline uint32_t c_hton32(uint32_t x) {
#ifdef _WIN32
	return htonl(x);
#else
	return htobe32(x);
#endif
}
c_nodiscard
static inline uint32_t c_ntoh32(uint32_t x) {
#ifdef _WIN32
	return ntohl(x);
#else
	return be32toh(x);
#endif
}
c_nodiscard
static inline uint64_t c_hton64(uint64_t x) {
#ifdef _WIN32
	return htonll(x);
#else
	return htobe64(x);
#endif
}
c_nodiscard
static inline uint64_t c_ntoh64(uint64_t x) {
#ifdef _WIN32
	return ntohll(x);
#else
	return be64toh(x);
#endif
}
/*! \} */

/*! \defgroup GetOptFunctions POSIX Getopt for Windows
* \brief Platform independent version for Getopt, POSIX function defined by unistd.h
* \author Hans Dietrich
* \copyright Public domain
* \pre Windows only. On Linux they are just macros to unistd.h Getopt variables
* \{ */
#ifdef _WIN32 // Windows
CAEN_UTILITY_DLLAPI extern int c_mp_opterr;

CAEN_UTILITY_DLLAPI extern int c_mp_optopt;

CAEN_UTILITY_DLLAPI extern int c_mp_optind;

CAEN_UTILITY_DLLAPI extern char *c_mp_optarg;

c_nodiscard
CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_mp_getopt(int argc, char *const *argv, const char *options);

#else // Linux
#define c_mp_opterr opterr
#define c_mp_optopt optopt
#define c_mp_optind optind
#define c_mp_optarg optarg
#define c_mp_getopt getopt
#endif

#define c_opterr c_mp_opterr
#define c_optopt c_mp_optopt
#define c_optind c_mp_optind
#define c_optarg c_mp_optarg
#define c_getopt c_mp_getopt
/*! \} */

#ifdef __cplusplus
}
#endif

#endif // CAEN_INCLUDE_CAENMULTIPLATFORM_H_
