/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENRandom.h
*	\brief		Pseudo-random number generator implemented on MT19937.
*	\author		Giovanni Cerretani
*
******************************************************************************/

#ifndef CAEN_INCLUDE_CAENRANDOM_H_
#define CAEN_INCLUDE_CAENRANDOM_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <CAENUtility.h>
#include <stdint.h>

/*! \defgroup MT19937Functions32 Pseudo-random generator MT19937
* \brief Pseudo-random number generator functions using 32-bit MT19937 by Matsumoto and Nishimura (1998). Equivalent to C++ `std::mt19937` implementation.
* \pre When seeded with `5489`, the 10000th consecutive invocation produces the value `4123659995`, as required by the C++11 standard.
* \{ */

/*!
* Initialize memory for 32-bit MT19937. If already initialized, nothing is done.
* \return ::c_Utility_ErrorCode_Success (0) in case of success. Error codes specified in #c_Utility_ErrorCode_t.
* \warning No need to call it manually, as done by this library at load.
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_rand32_init(void);

/*!
* Free memory for 32-bit MT19937. If already deinitialized, nothing is done.
* \warning No need to call it manually, as done by this library at close.
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_rand32_deinit(void);

/*!
* Re-initializes the internal state sequence to pseudo-random values.
* \pre If not initialized, it calls c_rand32_init().
* \param[in]		seed		a seed
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_rand32_seed(uint32_t seed);

/*!
* Get a randon integer number in range [0, UINT32_MAX]
* \pre If not seeded, it calls c_rand32_seed() with the default seed (5489).
* \return			the generated number
*/
c_nodiscard
CAEN_UTILITY_DLLAPI uint32_t CAEN_UTILITY_API c_rand32_int(void);

/*!
* Get a pseudo-random number in range [0, 1) as double
* Wrapper to c_rand32_int() * 2<sup>-32</sup>
* \return			the generated number
*/
c_nodiscard
CAEN_UTILITY_DLLAPI double CAEN_UTILITY_API c_rand(void);

/*! \} */

/*! \defgroup MT19937Functions64 Pseudo-random generator MT19937-64
* \brief Pseudo-random number generator functions using 64-bit MT19937 by Matsumoto and Nishimura (2000). Equivalent to C++ `std::mt19937_64` implementation.
* \pre When seeded with `5489`, the 10000th consecutive invocation produces the value `9981545732273789042`, as required by the C++11 standard.
* \{ */

/*!
* Initialize memory for 64-bit MT19937. If already initialized, nothing is done.
* \return ::c_Utility_ErrorCode_Success (0) in case of success. Error codes specified in #c_Utility_ErrorCode_t.
* \warning No need to call it manually, as done by this library at load.
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_rand64_init(void);

/*!
* Free memory for 64-bit MT19937. If already deinitialized, nothing is done.
* \warning No need to call it manually, as done by this library at close.
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_rand64_deinit(void);

/*!
* Re-initializes the internal state sequence to pseudo-random values.
* \pre If not initialized, it calls c_rand64_init().
* \param[in]		seed		a seed
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_rand64_seed(uint64_t seed);

/*!
* Get a randon integer number in range [0, UINT64_MAX]
* \pre If not seeded, it calls c_rand64_seed() with the default seed (5489).
* \return			the generated number
*/
c_nodiscard
CAEN_UTILITY_DLLAPI uint64_t CAEN_UTILITY_API c_rand64_int(void);

/*!
* Get a pseudo-random number in range [0, 1] as double
* Wrapper to c_rand64_int() * 2<sup>-64</sup>
* \return			the generated number
*/
c_nodiscard
CAEN_UTILITY_DLLAPI double CAEN_UTILITY_API c_rand64(void);

/*!
* Get a pseudo-random number normally distributed (zero mean, unit variance),
* Implements the Box-Muller transform with numbers from c_rand64_int().
* \return			the generated number
*/
c_nodiscard
CAEN_UTILITY_DLLAPI double CAEN_UTILITY_API c_rand64_normal(void);

/*! \} */

#ifdef __cplusplus
}
#endif

#endif // CAEN_INCLUDE_CAENRANDOM_H_
