/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENSocket.h
*	\brief		TCP/IP functions.
*	\author
*
******************************************************************************/

#ifndef CAEN_INCLUDE_CAENSOCKET_H_
#define CAEN_INCLUDE_CAENSOCKET_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <types/CAENSocketTypes.h>
#include <CAENUtility.h>

/*! \defgroup SocketFunctions Network protocol functions
* \brief TCP/IP protocol
*/

/*! Wrapper to WSAStartup()
* \warning No need to use it, as called by c_socket_new()
* \pre No-op on Linux.
* \return			::c_Socket_ErrorCode_Success (0) in case of success. Error codes specified in #c_Socket_ErrorCode_t.
* \ingroup SocketFunctions
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_socket_init(void);

/*! Wrapper to WSACleanup()
* \warning No need to use it, as called by c_socket_delete()
* \pre Each call of c_socket_init() must match a call to c_socket_cleanup(). No-op on Linux.
* \return			::c_Socket_ErrorCode_Success (0) in case of success. Error codes specified in #c_Socket_ErrorCode_t.
* \ingroup SocketFunctions
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_socket_cleanup(void);

/*! Call c_socket_init() and create a new c_socket_t structure. c_socket_t::socket left uninitialized
* \return			a pointer to a new c_socket_t to be deleted with c_socket_delete(). NULL in case of error.
* \ingroup SocketFunctions
*/
c_nodiscard c_attribute_malloc
CAEN_UTILITY_DLLAPI c_socket_t * CAEN_UTILITY_API c_socket_new(void);

/*! Call c_socket_reset(), c_socket_cleanup() and free memory
* \param[in]		socket				the socket to clean
* \ingroup SocketFunctions
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_socket_delete(c_socket_t *socket);

/*! Close the socket and set socket->socket to #c_socket_invalid
* \param[in]		socket				the socket to reset
* \ingroup SocketFunctions
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_socket_reset(c_socket_t *socket);

/*! \defgroup SendRecv Safe versions of recv() and send()
* \ingroup SocketFunctions
* \{ */
CAEN_UTILITY_DLLAPI c_ssize_t CAEN_UTILITY_API c_recv(const c_socket_t *sckt, void *buffer, size_t totSize);
CAEN_UTILITY_DLLAPI c_ssize_t CAEN_UTILITY_API c_send(const c_socket_t *sckt, const void *buffer, size_t totSize);
/*! \} */

/*! \defgroup MutexSendRecv Safe versions of recv() and send() with mutex locking
* \ingroup SocketFunctions
* \{ */
CAEN_UTILITY_DLLAPI c_ssize_t CAEN_UTILITY_API c_recv_unlock(c_socket_t *sckt, void *buffer, size_t totSize);
CAEN_UTILITY_DLLAPI c_ssize_t CAEN_UTILITY_API c_send_lock(c_socket_t *sckt, const void *buffer, size_t totSize);
/*! \} */

/*! \defgroup GenericSocket Platform independent wrappers of socket/accept/bind/listen/connect
* \ingroup SocketFunctions
* \{ */

/*! Create a new c_socket_t where the socket has been initialized to socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)
* \ingroup SocketFunctions
* \return			a pointer to a new c_socket_t to be deleted with c_socket_delete(). NULL in case of error.
*/
c_nodiscard c_attribute_malloc
CAEN_UTILITY_DLLAPI c_socket_t * CAEN_UTILITY_API c_tcp_socket(void);

/*! Create a new c_socket_t where the socket has been initialized to accept(socket, addr, addrlen)
* \ingroup SocketFunctions
* \param[in]		sckt				a socket created with c_tcp_socket(), bound to a local address with c_bind(), and is listening for connections after a c_listen().
* \param[out]		addr				a pointer to a sockaddr structure, filled in with the address of the peer socket
* \param[out]		addrlen				the actual size of the peer address
* \return			a pointer to a new c_socket_t to be deleted with c_socket_delete(). NULL in case of error.
*/
c_nodiscard c_attribute_malloc
CAEN_UTILITY_DLLAPI c_socket_t * CAEN_UTILITY_API c_accept(const c_socket_t *sckt, struct sockaddr *addr, c_socklen_t *addrlen);

/*! Bind a name to a socket
* \ingroup SocketFunctions
* \param[in]		sckt				a socket created with c_tcp_socket()
* \param[in]		addr				the address to be assigned
* \param[in]		addrlen				the size, in bytes, of the address structure pointed to by addr
* \return			on success, zero is returned.  On error, -1 is returned, and ::c_socket_errno is set appropriately.
*/
CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_bind(const c_socket_t *sckt, const struct sockaddr *addr, c_socklen_t addrlen);

/*! Marks the socket referred to by \p sckt as a passive socket
* \ingroup SocketFunctions
* \param[in]		sckt				a socket created with c_tcp_socket()
* \param[in]		backlog				the maximum length to which the queue of pending connections for sckt may grow
* \return			on success, zero is returned.  On error, -1 is returned, and ::c_socket_errno is set appropriately.
*/
CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_listen(const c_socket_t *sckt, int backlog);

/*! Initiate a connection on a socket
* \ingroup SocketFunctions
* \param[in]		sckt				a socket created with c_tcp_socket()
* \param[in]		addr				the address to which connect to
* \param[in]		addrlen				the size, in bytes, of the address structure pointed to by addr
* \return			on success, zero is returned.  On error, -1 is returned, and ::c_socket_errno is set appropriately.
*/
CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_connect(const c_socket_t *sckt, const struct sockaddr *addr, c_socklen_t addrlen);
/*! \} */

/*! \defgroup ClientServer Simplified client/server functions
* \ingroup SocketFunctions
* \{ */
/*! Server function. Init the socket to receive connection from inaddr:port
* \param[in,out]	server				a location where to put the address of a socket initialized with c_socket_new and to be freed
* \param[out]		inaddr				the address we agree to accept. INADDR_ANY (0x00000000) to accept any address.
* \param[out]		_port				pointer to a uint16_t with the port. If value is 0, a port is automatically choosen and the value is set accordingly
* \return			::c_Socket_ErrorCode_Success (0) in case of success. Error codes specified in #c_Socket_ErrorCode_t.
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_socket_server_init(c_socket_t **server, uint32_t inaddr, uint16_t *_port);

/*! Server function. Accept new connection on the initialized socket
* \param[in]		server				a pointer passed through c_socket_server_init()
* \param[out]		client				a location where to put the address of a socket initialized with c_socket_new and to be freed
* \return			::c_Socket_ErrorCode_Success (0) in case of success. Error codes specified in #c_Socket_ErrorCode_t.
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_socket_server_accept(const c_socket_t *server, c_socket_t **client);

/*! Client function. Connect to host, port specified by [addr]
* \param[in,out]	client				a location where to put the address of a socket initialized with c_socket_new and to be freed
* \param[in]		addr_server			the remote address
* \param[in]		addrlen				the size, in bytes, of the address structure pointed to by addr
* \return			::c_Socket_ErrorCode_Success (0) in case of success. Error codes specified in #c_Socket_ErrorCode_t.
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_socket_client_sockaddr_connect(c_socket_t **client, const struct sockaddr *addr_server, c_socklen_t addrlen);

/*! Client function. Connect to [hostname]:[port]
* \param[in,out]	client				a location where to put the address of a socket initialized with c_socket_new and to be freed
* \param[in]		hostname			the remote address. Either a hostname or an IPv4 address in standard dot notation.
* \param[in]		port				the remote port
* \return			::c_Socket_ErrorCode_Success (0) in case of success. Error codes specified in #c_Socket_ErrorCode_t.
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_socket_client_connect(c_socket_t **client, const char *hostname, uint16_t port);
/*! \} */

#ifdef __cplusplus
}
#endif

#endif // CAEN_INCLUDE_CAENSOCKET_H_
