/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENThread.h
*	\brief		Functions to handle threads, inspired to C11 threads.h.
*	\author
*
******************************************************************************/

#ifndef CAEN_INCLUDE_CAENTHREAD_H_
#define CAEN_INCLUDE_CAENTHREAD_H_

#include <stdbool.h>
#include <time.h>

#include <types/CAENThreadTypes.h>
#include <CAENUtility.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \defgroup ConditionFunctions Condition functions.
* \brief TODO
* \{ */
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_condition_init(c_condition_t *cond);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_condition_destroy(c_condition_t *cond);

c_acquires_lock_on_success(cond->_critical_section)
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_condition_lock(c_condition_t *cond);

c_releases_lock_on_success(cond->_critical_section)
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_condition_unlock(c_condition_t *cond);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_condition_wait(c_condition_t *cond, int32_t ms);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_condition_signal_one(c_condition_t *cond);

CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_condition_broadcast(c_condition_t *cond);

/*! \} */

/*! \defgroup SemaphoresFunctions Semaphores related functions
* \brief Platform independent functions to manage semaphores
* \{ */
/*!
* Initialize semaphore
* \param[out] s		a pointer to uninitialized #c_semaphore_t
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_semaphore_init(c_semaphore_t *s);
/*!
* Initialize multi semaphore
* \param[out] s		a pointer to uninitialized #c_semaphore_t
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_semaphore_multi_init(c_semaphore_t *s);
/*!
* Destroy semaphore
* \param[in] s		a pointer to initialized #c_semaphore_t
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_semaphore_destroy(c_semaphore_t *s);
/*!
* Wait semaphore
* \param[in] s		a pointer to initialized #c_semaphore_t
* \param[in] ms		timeout in milliseconds. Set to -1 to disable timeout.
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_semaphore_wait(c_semaphore_t *s, int32_t ms);
/*!
* Destroy semaphore
* \param[in] s		a pointer to initialized #c_semaphore_t
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_semaphore_post(c_semaphore_t *s);
/*! \} */

/*! \defgroup MutexFunctions Mutex related functions
* \brief Platform independent functions to manage mutexes
* \{ */
/*!
* Initialize mutex
* \param[out] m		a pointer to uninitialized #c_mutex_t
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_mutex_init(c_mutex_t *m);
/*!
* Lock mutex
* \param[in] m		a pointer to initialized #c_mutex_t
*/
c_acquires_lock_on_success(*m)
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_mutex_lock(c_mutex_t *m);
/*!
* Try to lock mutex
* \param[in] m		a pointer to initialized #c_mutex_t
*/
c_acquires_lock_on_success(*m) c_nodiscard
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_mutex_trylock(c_mutex_t *m);
/*!
* Unlock mutex
* \param[in] m		a pointer to initialized #c_mutex_t
*/
c_releases_lock_on_success(*m)
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_mutex_unlock(c_mutex_t *m);
/*!
* Destroy mutex
* \param[in] m		a pointer to initialized #c_mutex_t
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_mutex_destroy(c_mutex_t *m);
/*! \} */

/*! \defgroup ThreadFunctions Thread related functions
* \brief Platform independent functions to manage threads
* \{ */
/*!
* Initializer for invalid c_thread_t
*/
CAEN_UTILITY_DLLAPI c_thread_t CAEN_UTILITY_API c_thread_invalid();

/*! \defgroup ThreadFunctions Thread related functions
* \brief Platform independent functions to manage threads
* \{ */
/*!
* Create thread
* \param[out] thr			a pointer to uninitialized #c_thread_t
* \param[in] func			function to execute
* \param[out] arg			argument to pass to the function
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_thread_create(c_thread_t *thr, c_tstart_t func, void *arg);
/*!
* Join thread
* \param[in] thr			a pointer to initialized #c_thread_t
* \param[out] res			location to put the result code to
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_thread_join(c_thread_t thr, int *res);
/*!
* Join thread with timeout
* \param[in] thr			a pointer to initialized #c_thread_t
* \param[out] res			location to put the result code to
* \param[in] timeout_ms		timeout in milliseconds
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_thread_timedjoin(c_thread_t thr, int *res, int32_t timeout_ms);
/*!
* Set affinity to a specific set of CPU core
* \pre Linux only
* \param[in] thr			a pointer to initialized #c_thread_t
* \param[in] mask			a mask with bit set to 1 at the position of the required cores
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_thread_setaffinity(c_thread_t thr, uint64_t mask);
/*!
* Get the identifier of the calling thread.
*/
CAEN_UTILITY_DLLAPI c_thread_t CAEN_UTILITY_API c_thread_current(void);
/*!
* Provides a hint to the implementation to reschedule the execution of threads, allowing other threads to run
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_thread_yield(void);
/*!
* Detach a thread
* \param[in] thr			a pointer to initialized #c_thread_t
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_thread_detach(c_thread_t thr);
/*!
* Terminate the execution of the calling thread and set its result code to res.
* \param[in] res			result code
*/
c_noreturn
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_thread_exit(int res);
/*!
* Checks whether lhs and rhs refer to the same thread.
* \param[in] lhs			thread to compare
* \param[in] rhs			thread to compare
* \return true if equal
*/
CAEN_UTILITY_DLLAPI bool CAEN_UTILITY_API c_thread_equal(c_thread_t lhs, c_thread_t rhs);
/*!
* Blocks the execution of the current thread for a period of time
* \param[in] duration		pointer to the duration to sleep for
* \param[out] remaining		pointer to the object to put the remaining time on interruption; may be NULL, in which case it is ignored
* \return 0 on success, -1 if a signal occurred, other negative value if an error occurred
* \warning This function does not return values from #CAENThread_RetCode_t, for compatibility with C11 thrd_sleep().
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_thread_sleep(const struct timespec* duration, struct timespec* remaining);

/*!
* Blocks the execution of the current thread for a period of time
* \param[in] ms				period in ms
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_sleep(uint32_t ms);
/*! \} */

/*! \defgroup TicketsFunctions Tickets related functions
* \brief Platform independent implementation of fair mutex
* \{ */
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_ticket_init(c_ticket_t *ticket);
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_ticket_delete(c_ticket_t *ticket);
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_ticket_lock(c_ticket_t *ticket);
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_ticket_unlock(c_ticket_t *ticket);
/*! \} */

#ifdef __cplusplus
}
#endif

#endif // CAEN_INCLUDE_CAENTHREAD_H_
