/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENUtility.h
*	\brief		Main header and generic tools.
*	\author
*
******************************************************************************/

/******************************************************************************
* This library includes code from other open source projects.
*
* - "URL Parse"
*		Source:		https://github.com/Xsoda/url
*		License:	BSD 2-Clause "Simplified" License
*		Year:		2014
*		Author:		Xsoda
*
* - "Map"
*		Source:		https://github.com/rxi/map
*		License:	MIT license
*		Year:		2014
*		Author:		rxi
*
* - "POSIX Directory Browsing API for Windows"
*		Source:		https://github.com/octopusprime314/ReBoot/
*		License:	Copyright Kevlin Henney, 1997, 2003. All rights reserved.
*					Permission to use, copy, modify, and distribute this software and its
*					documentation for any purpose is hereby granted without fee, provided
*					that this copyright and permissions notice appear in all copies and
*					derivatives.
*					This software is supplied "as is" without express or implied warranty.
*					But that said, if there are any problems please get in touch.
*		Year:		1997, 2003
*		Author:		Kevlin Henney
*
* - "XGetopt - A Unix-compatible getopt() for MFC and Win32"
*		Source:		https://www.codeproject.com/Articles/1940/XGetopt-A-Unix-compatible-getopt-for-MFC-and-Win
*		License:	Public domain
*		Year:		2003
*		Author:		Hans Dietrich
*
******************************************************************************/

#ifndef CAEN_INCLUDE_CAENUTILITY_H_
#define CAEN_INCLUDE_CAENUTILITY_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stddef.h>

// Define exports macros (only if the _CAEN_UTILITY_EXPORT macro
// is defined the functions are exported in the dll)
#ifdef _WIN32
/*
 * __stdcall would introduce name mangling if exporting functions
 * with __declspec(dllexport). This would make GetProcAddress harder
 * to use, so it is better to use __cdecl.
 * Curiously, if using __stdcall with .def files rather than
 * __declspec(dllexport), there is no name namgling (this is the
 * approach of CAENVMELib, for example).
 *
 * The x86_64 architecture has its own calling convention, that
 * never introduces name mangling, so it doesn't mattter.
 *
 * See https://stackoverflow.com/a/6635067/3287591.
 */
#define CAEN_UTILITY_API __cdecl
#ifdef _CAEN_UTILITY_EXPORT
#define CAEN_UTILITY_DLLAPI __declspec(dllexport)
#else
#define CAEN_UTILITY_DLLAPI __declspec(dllimport)
#endif
#else
#define CAEN_UTILITY_API
#define CAEN_UTILITY_DLLAPI __attribute__((visibility("default")))
#endif

/*!
* \defgroup StringMacros String utility macros
* \brief To convert macro to string.
* \{ */
#define _c_macro_str_helper(x)			#x
#define c_macro_str(x)					_c_macro_str_helper(x)
/*! \} */

//! Platform independent macro to remove unreferenced parameter warning
#define c_unused_parameter(P)		((void)(P))

/*!
* \defgroup CompilerVersionMacros Compiler version utility macros
* \brief To get information about the used compiler
* \{ */

// GCC
#if defined(__GNUC__) && !defined(__clang__)
#define c_gcc_version		(__GNUC__ * 10000 + __GNUC_MINOR__ * 100 + __GNUC_PATCHLEVEL__)
#define c_compiler_version	"GCC v" c_macro_str(__GNUC__) "." c_macro_str(__GNUC_MINOR__) "." c_macro_str(__GNUC_PATCHLEVEL__)
#else
#define c_gcc_version		(-1)
#endif

// Clang
#ifdef __clang__
#define c_clang_version		(__clang_major__ * 10000 + __clang_minor__ * 100 + __clang_patchlevel__)
#define c_compiler_version	"Clang v" __clang_version__
#else
#define c_clang_version		(-1)
#endif

// Microsoft Visual C++
#if defined(_MSC_VER) && !defined(__clang__)
#define c_msc_version		(_MSC_VER)
#define c_compiler_version	"Visual C++ v" c_macro_str(_MSC_FULL_VER)
#else
#define c_msc_version		(-1)
#endif

#ifndef c_compiler_version
#define c_compiler_version	"Unknown"
#endif
/*! \} */

//! Throw away GCC attributes from Visual Studio, and viceversa
#ifdef _WIN32
#define __attribute__(A)
#else
#define __declspec(A)
#endif

//! No-operation macro (different from nop assemply instruction: this macro does not generate code)
#ifdef _WIN32
#define c_nop				NOP_FUNCTION
#else
#define c_nop				((void)0)
#endif

/*! \defgroup AttributeMacros Macros to set compiler attributes
* \brief Macros to set function attributes compilers and improve optimization.
* \{ */

/*! The format attribute specifies that a function takes printf, scanf, strftime or strfmon style arguments that should be type-checked against a format string.
*/
#define c_attribute_format(_FMT_,_ARGS_)	__attribute__((__format__(__printf__, _FMT_, _ARGS_)))

/*! The alloc_size attribute is used to tell the compiler that the function return value points to memory, where the size is given by one or two of the functions parameters.
* \details GCC uses this information to improve the correctness of __builtin_object_size. Works differently on clang: clang will only trace const pointers.
* \pre GCC version >= 4.3 (https://gcc.gnu.org/gcc-4.3/changes.html)
* \pre clang version >= 4.0.0 (http://releases.llvm.org/4.0.0/tools/clang/docs/AttributeReference.html)
*/
#if c_gcc_version >= 40300 || c_clang_version >= 40000
#define c_attribute_alloc_size(...)			__attribute__((__alloc_size__(__VA_ARGS__)))
#else
#define c_attribute_alloc_size(...)			__attribute__(())
#endif

/*! The allocator __declspec is used to mark custom allocator functions, to work with the Visual Studio native memory profiler
* \details	The native memory profiler works by collecting allocation ETW event data emitted by during runtime.
* 			Any functions that return a pointer to newly allocated heap memory can be decorated with __declspec(allocator).
* \pre Visual Studio version >= 1900 (2015)
*/
#define c_declspec_allocator				__declspec(allocator)

/*! This tells the compiler that a function is malloc-like.
* \details	It tells that the pointer `P` returned by the function cannot alias any other pointer valid when the function returns,
* 			and moreover no pointers to valid objects occur in any storage addressed by `P`.
* 			Using this attribute can improve optimization. Should not be used with `realloc`.
* \pre GCC version >= 2.96 (sys/cdefs.h)
*/
#define c_attribute_malloc					__attribute__((__malloc__))

/*! Pure functions. Calls to such functions can be subject to common subexpression elimination and loop optimization just as an arithmetic operator would be.
* \details Many functions have no effects except the return value and their return value depends only on the parameters and/or global variables.
* \pre GCC version >= 2.96 (sys/cdefs.h)
*/
#define c_attribute_pure					__attribute__((__pure__))

/*! This tells the compiler that an attribute must be non-null.
* \details	The nonull function attribute allows to mark pointer parameters which must not be NULL. It causes the compiler to check that arguments are non-null.
* 			If the compiler determines that a null pointer is passed in an argument slot marked as non-null, and the -Wnonnull option is enabled, a warning is issued.
*			The compiler may also choose to make optimizations based on the knowledge that certain function arguments will not be null.
* \pre GCC version >= 3.3 (sys/cdefs.h)
*/
#define c_attribute_nonnull(...)			__attribute__((__nonnull__(__VA_ARGS__)))

/*! Explicit fallthrough in switch cases.
* \details To suppress warning given by -Wimplicit-fallthrough in GCC version 7 and above.
* \pre GCC version >= 7.0 (https://gcc.gnu.org/gcc-7/changes.html)
*/
#if c_gcc_version >= 70000
#define c_attribute_fallthrough				__attribute__((__fallthrough__))
#else
#define c_attribute_fallthrough				c_nop
#endif

/*! Platform independent macro to force inline function
*/
#if c_gcc_version >= 0 || c_clang_version >= 0
#define c_force_inline						__attribute__((__always_inline__)) inline
#elif c_msc_version >= 0
#define c_force_inline						__forceinline
#else
#define c_force_inline						inline
#endif

/*! The compiler will warn if the return value of a function is not used
* \details Equivalent of C++17 [[nodiscard]], but pn Windows the check is performed on static code analysis
* \pre GCC version >= 3.4 (sys/cdefs.h)
* \pre MSC version >= 1700
*/
#if c_gcc_version >= 30400 || c_clang_version >= 40000
#define c_nodiscard							__attribute__((__warn_unused_result__))
#elif c_msc_version >= 1700
#define c_nodiscard							_Check_return_
#else
#define c_nodiscard
#endif

/*! Equivalent to C11 _Noreturn
* \details Equivalent of C++17 [[nodiscard]], but pn Windows the check is performed on static code analysis
* \pre GCC version >= 3.4 (sys/cdefs.h)
* \pre MSC version >= 1700
*/
#if c_gcc_version >= 20500 || c_clang_version >= 0
#define c_noreturn							__attribute__((__noreturn__))
#elif c_msc_version >= 1700
#define c_noreturn							__declspec(noreturn)
#else
#define c_noreturn
#endif

/*! Portable version of _When_ to annotate locks.
* \details May be used on a function definition (also known as a function body) in place of the list of annotations in the header.
*/
#if c_msc_version >= 0
#define c_acquires_lock_on_success(m)		_When_(return == CAENThread_RetCode_Success, _Acquires_lock_(m))
#define c_releases_lock_on_success(m)		_When_(return == CAENThread_RetCode_Success, _Releases_lock_(m))
#else
#define c_acquires_lock_on_success(m)
#define c_releases_lock_on_success(m)
#endif

/*! Portable version of _Use_decl_annotations_ macro to remove Visual Studio warnings in function definitions that have annotations in their declaration.
* \details May be used on a function definition (also known as a function body) in place of the list of annotations in the header.
*/
#if c_msc_version >= 0
#define c_use_decl_annotations				_Use_decl_annotations_
#else
#define c_use_decl_annotations
#endif

/*! \} */

/*!
* \defgroup LikelyMacro Macro to set likelihood of if statements
* \brief These macros communicate to the compiler that the conditional expression is likely or unlikely to yield a positive result
* \pre Linux only
* \{ */
#if c_gcc_version >= 0 || c_clang_version >= 0
#define c_likely(x)							__builtin_expect((x),1)
#define c_unlikely(x)						__builtin_expect((x),0)
#else
#define c_likely(x)							(x)
#define c_unlikely(x)						(x)
#endif
/*! \} */

//! Type of data
typedef enum {
	TYPE_INT8		 = 0,
	TYPE_UINT8		= 1,
	TYPE_INT16		= 2,
	TYPE_UINT16		= 3,
	TYPE_INT32		= 4,
	TYPE_UINT32		= 5,
	TYPE_INT64		= 6,
	TYPE_UINT64		= 7,
	TYPE_STRING		= 8,
	TYPE_LONG		= 9,
	TYPE_DOUBLE		= 10,
	TYPE_CHAR		= 11,
	TYPE_RESCODE	= 12,
	TYPE_SHORT		= 13,
	TYPE_MEMORY		= 14,

	TYPE_NONE		= 255,
} c_type_t;

//! Library return codes
typedef enum {
	c_Utility_ErrorCode_Success = 0,
	c_Utility_ErrorCode_GenericError = -1,
	c_Utility_ErrorCode_Map = -2,
	c_Utility_ErrorCode_Memory = -3,
	c_Utility_ErrorCode_NULL= -4,
} c_Utility_ErrorCode_t;

//! Generic function pointer
typedef void(*c_funcptr_t)(void);

/*! \defgroup StringManipulation String manipulation
* \brief Function to manupulate strings
* \{ */
CAEN_UTILITY_DLLAPI size_t CAEN_UTILITY_API c_srtrim(char *str, const char *remove);
CAEN_UTILITY_DLLAPI size_t CAEN_UTILITY_API c_sltrim(char *str, const char *remove);
CAEN_UTILITY_DLLAPI size_t CAEN_UTILITY_API c_strim(char *str, const char *remove);
/*! \} */

#ifdef __cplusplus
}
#endif

#endif // CAEN_INCLUDE_CAENUTILITY_H_
