/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENXMLParser.h
*	\brief		Tools to parse an XML file using libxml.
*	\author
*
******************************************************************************/

#ifndef CAEN_INCLUDE_CAENXMLPARSER_H_
#define CAEN_INCLUDE_CAENXMLPARSER_H_

#ifndef _NOUSEXML

/*!
* \defgroup XMLFunctions XML parser
* \brief Wrapper functions to libxml
*/

#ifdef __cplusplus
extern "C" {
#endif

#include <types/CAENXMLParserTypes.h>
#include <types/CAENMultiplatformTypes.h>
#include <CAENUtility.h>

#include <libxml/xpath.h>

/*!
* Create a new c_xmlfile_t struct given its filename, and call xmlReadFile().
* \param[in]		filename	path to XML file or URL
* \return			An initialized pointer. NULL in case of error.
* \ingroup			XMLFunctions
*/
c_nodiscard c_attribute_malloc c_attribute_nonnull(1)
CAEN_UTILITY_DLLAPI c_xmlfile_t * CAEN_UTILITY_API c_xml_newfile(const char *filename);

/*!
* Create a new c_xmlfile_t struct given a buffer in memory, and call xmlReadMemory().
* \param[in]		buffer		a pointer to a char array
* \param[in]		size		the size of the array
* \return			An initialized pointer. NULL in case of error.
* \ingroup			XMLFunctions
*/
c_nodiscard c_attribute_malloc c_attribute_nonnull(1)
CAEN_UTILITY_DLLAPI c_xmlfile_t * CAEN_UTILITY_API c_xml_newfile_from_memory(const char *buffer, int size);

/*!
* Free memory allocated by c_xml_newfile() of c_xml_newfile_from_buffer() and close the document.
* \param[in]		file		a pointer returned by c_xml_newfile() or c_xml_newfile_from_memory(). If NULL, no operation is performed.
* \ingroup			XMLFunctions
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_xml_freefile(c_xmlfile_t *file);

/*!
* Wrapper to xmlDocGetRootElement(file->doc).
* \param[in]		file		the file
* \return			the xmlNode pointer for the root. NULL in case of error.
* \ingroup			XMLFunctions
*/
c_nodiscard
CAEN_UTILITY_DLLAPI xmlNode * CAEN_UTILITY_API c_xml_getrootnode(const c_xmlfile_t *file);

/*!
* Get the complete path of the document.
* \param[in]		file		the file
* \return			a pointer to the filename
* \ingroup			XMLFunctions
*/
c_nodiscard
CAEN_UTILITY_DLLAPI const char * CAEN_UTILITY_API c_xml_getpath(const c_xmlfile_t *file);

/*!
* Uses 'query' expression to get a nodeset from 'file'.
* \param[in]		file		the file
* \param[in]		query		the expression
* \return			a pointer to a xmlNodeSet, to be freed with c_xml_freenodeset(). NULL in case of error.
* \ingroup			XMLFunctions
*/
c_nodiscard c_attribute_malloc
CAEN_UTILITY_DLLAPI xmlNodeSet * CAEN_UTILITY_API c_xml_getnodeset(c_xmlfile_t *file, const xmlChar *query);

/*!
* Wrapper to xmlXPathFreeNodeSet(nodeset).
* \param[in]		nodeset		a pointer returned by c_xml_getnodeset(). If NULL, no operation is performed.
* \ingroup			XMLFunctions
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_xml_freenodeset(xmlNodeSet *nodeset);

/*!
* Parses an xmlNode to get the value of the one named 'attr'.
* \param[in]		node		a node
* \param[in]		attribute	the name of the attribute to get
* \param[in]		warnEnable	if true, print a message to the log in case of error
* \return			a pointer the value of the attribute. NULL in case of error.
* \ingroup			XMLFunctions
*/
c_nodiscard
CAEN_UTILITY_DLLAPI const char * CAEN_UTILITY_API c_xml_getattributevalue(const xmlNode *node, const char *attribute, bool warnEnable);

/*!
* Returns the first random-ordered node which matches the query. To be used only for queries which produce for sure a single result.
* \param[in]		file		the file
* \param[in]		query		the query
* \return			a pointer the node. NULL in case number of results is not 1 (0 or more than 1).
* \ingroup			XMLFunctions
*/
c_nodiscard
CAEN_UTILITY_DLLAPI xmlNode * CAEN_UTILITY_API c_xml_getfirstnodefromxpathquery(c_xmlfile_t *file, const xmlChar *query);

/*!
* Parses an xmlNode to get the first one called 'nodename'.
* \param[in]		node		a node
* \param[in]		nodename	the name
* \return			a pointer the node. If 'node' itself matches 'nodename', 'node' is returned. NULL in case of error.
* \ingroup			XMLFunctions
*/
c_nodiscard
CAEN_UTILITY_DLLAPI xmlNode * CAEN_UTILITY_API c_xml_getnextelementfromname(xmlNode *node, const char *nodename);

/*!
* Parses an xmlNode siblings to get the first one called 'nodename'.
* \param[in]		node		a node
* \param[in]		nodename	the name
* \return			a pointer the node. NULL in case of error.
* \ingroup			XMLFunctions
*/
c_nodiscard
CAEN_UTILITY_DLLAPI xmlNode * CAEN_UTILITY_API c_getfirstsiblingelementfromname(xmlNode *node, const char *nodename);

/*!
* Parses an xmlNode children to get the first one called 'nodename'.
* \param[in]		node		a node
* \param[in]		nodename	the name
* \return			a pointer the node. NULL in case of error.
* \ingroup			XMLFunctions
*/
c_nodiscard
CAEN_UTILITY_DLLAPI xmlNode * CAEN_UTILITY_API c_xml_getfirstchildrenelementfromname(xmlNode *node, const char *nodename);

/*!
* Parses an xmlNode descendants to get the first one whose hierachy is 'nodename'.
* \param[in]		node		a node
* \param[in]		nodename	the name
* \return			a pointer the node. NULL in case of error.
* \ingroup			XMLFunctions
*/
c_nodiscard
CAEN_UTILITY_DLLAPI xmlNode * CAEN_UTILITY_API c_xml_getfirstdescendantelementfromname(xmlNode *node, const char *nodename);

/*!
* Get value in 'node' first child, taking into account possible double or hexadecimal representation.
* \param[in]		node		a node
* \param[out]		code		the value
* \return			::c_xml_RetCode_Success (0) in case of success. Error codes specified in #c_xml_RetCode_t.
* \ingroup			XMLFunctions
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_xml_getcodefromnodechild(const xmlNode *node, double *code);

/*!
* Get value in 'node[/code]', taking into account possible double or hexadecimal representation.
* \param[in]		node		a node
* \param[out]		code		the returned value
* \return			::c_xml_RetCode_Success (0) in case of success. Error codes specified in #c_xml_RetCode_t.
* \ingroup			XMLFunctions
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_xml_getcodefromnode(const xmlNode *node, double *code);

/*!
* Looks for a node called '/Definitions/${category}/${codename}' to get its value converted to double.
* Both double and hexadecimal integer parsing are supported.
* \param[in]		file		the file
* \param[in]		category	the category name
* \param[in]		codename	the codename
* \param[out]		code		the returned value
* \return			::c_xml_RetCode_Success (0) in case of success. Error codes specified in #c_xml_RetCode_t.
* \ingroup			XMLFunctions
*/
CAEN_UTILITY_DLLAPI int32_t CAEN_UTILITY_API c_xml_getcodefromcodename(const c_xmlfile_t *file, const char *__restrict category, const char *__restrict codename, double *code);

/*!
* Looks for a node called '/Definitions/${category}/${codename}' to get the attribute named 'attrname'.
* \param[in]		file		the file
* \param[in]		category	the category name
* \param[in]		codename	the codename
* \param[in]		attrname	the atribute
* \return			a pointer the value of the attribute. NULL in case of error.
* \ingroup			XMLFunctions
*/
c_nodiscard
CAEN_UTILITY_DLLAPI const char * CAEN_UTILITY_API c_xml_getattributevaluefromcodename(const c_xmlfile_t *file, const char *__restrict category, const char *__restrict codename, const char *__restrict attrname);

/*!
* Looks for a node called '/Definitions/XMLFunctions/${codename}' to dynamically load a function
* with name 'codename' from library specified by node's 'libname' attribute.
* \param[in]		file		the file
* \param[in]		codename	the codename
* \return			a pointer to that function. NULL in case of error.
* \ingroup			XMLFunctions
*/
c_nodiscard
CAEN_UTILITY_DLLAPI c_fcnhandle_t CAEN_UTILITY_API c_xml_getfunctionpointerfromcodename(const c_xmlfile_t *file, const char *codename);

/*!
* Wrapper to xmlStrPrintf().
* \param[out]		buf			the result buffer
* \param[in]		len			the result buffer length
* \param[in]		msg			the message with printf formatting
* \param[in]		...			extra parameters for the message
* \return			the number of characters written to \a buf or -1 if an error occurs
* \ingroup			XMLFunctions
*/
c_attribute_format(3, 4)
CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_xml_snprintf(xmlChar *buf, int len, const char *msg, ...);

/*!
* Length of a xmlChar's string.
* Wrapper to xmlStrlen().
* \param[in]		str			the xmlChar array
* \return			the number of xmlChar contained in the the xmlChar array
* \ingroup			XMLFunctions
*/
CAEN_UTILITY_DLLAPI int CAEN_UTILITY_API c_xml_strlen(const xmlChar *str);

/*!
* Version of LibXML parser.
* Wrapper to xmlParserVersion.
* \return			the version.
* \ingroup			XMLFunctions
*/
CAEN_UTILITY_DLLAPI const char * CAEN_UTILITY_API c_xml_parserversion(void);

/*!
* Cleanup of allocated resources. To be used when the module will not be used anymore.
* Wrapper to xmlCleanupParser(). Also close the libray, if some calls to c_xml_getfunctionpointerfromcodename() occurred.
* \ingroup			XMLFunctions
*/
CAEN_UTILITY_DLLAPI void CAEN_UTILITY_API c_xml_parsercleanup(void);

#ifdef __cplusplus
}
#endif

#endif // !_NOUSEXML

#endif // CAEN_INCLUDE_CAENXMLPARSER_H_
