/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
******************************************************************************
*
*	\file		CAENAllocator.c
*	\brief		Utilities to handle memory allocation in C minimizing memory leaks
*	\author		Francesco Pepe
*
******************************************************************************/

#include <CAENAllocator.h>

#include <CAENMultiplatform.h>
#include <CAENLinkedList.h>

c_use_decl_annotations c_allocator_t c_allocator_create() {
	c_allocator_t res;
	res.error_count = 0;
	res.allocated_resources = c_linkedlist_create();
	return res;
}

typedef struct {
	void* resource;
	c_resource_destroy_function destroy;
} _c_resource_t;

c_use_decl_annotations void* c_allocator_resource_create(c_allocator_t* allocator, c_resource_destroy_function df, void* resource) {
	if (allocator == NULL)
		return NULL;

	// Check if resource is NULL (allocation/creation may have failed in caller)
	if (resource == NULL) {
		allocator->error_count++;
		return NULL;
	}

	_c_resource_t* res = c_calloc(1, sizeof(_c_resource_t));
	if (res == NULL) {
		if (resource != NULL)
			df(resource);
		allocator->error_count++;
		return NULL;
	}

	res->resource = resource;
	res->destroy = df;

	if (c_linkedlist_insert_first(&(allocator->allocated_resources), res) != c_Utility_ErrorCode_Success) {
		c_free(res);
		if (resource != NULL)
			df(resource);
		allocator->error_count++;
		return NULL;
	}

	return resource;
}

c_use_decl_annotations void* c_allocator_malloc(c_allocator_t* allocator, size_t size) {
	return c_allocator_resource_create(allocator, c_free, c_malloc(size));
}

c_use_decl_annotations void* c_allocator_calloc(c_allocator_t* allocator, size_t nmemb, size_t size) {
	return c_allocator_resource_create(allocator, c_free, c_calloc(nmemb, size));
}

c_use_decl_annotations int32_t c_allocator_error_count(const c_allocator_t* allocator) {
	return allocator->error_count;
}

static int _resource_comparator(const void* r, const void* d) {
	// as per 'c_linkedlist_delete_data_compare', i-th list element is passed as
	// first argument, while given 'data' pointer is passed as second.
	return ((const _c_resource_t*)r)->resource == d;
}

static void _allocator_release_or_destroy(c_allocator_t* allocator, void* ptr, bool destroy) {
	_c_resource_t* resource = c_linkedlist_delete_data_compare(&(allocator->allocated_resources), ptr, _resource_comparator);
	if (destroy)
		resource->destroy(resource->resource);
	if (resource != NULL)
		c_free(resource);
}

void c_allocator_release(c_allocator_t* allocator, void* ptr) {
	_allocator_release_or_destroy(allocator, ptr, false);
}

void c_allocator_clear(c_allocator_t* allocator) {
	allocator->error_count = 0;
	while (allocator->allocated_resources.head != NULL) {
		_c_resource_t* resource = c_linkedlist_delete_first(&(allocator->allocated_resources));
		c_free(resource);
	}
}

void c_allocator_free(c_allocator_t* allocator, void* ptr) {
	_allocator_release_or_destroy(allocator, ptr, true);
}

void c_allocator_freeall(c_allocator_t* allocator) {
	if (allocator == NULL)
		return;
	for (c_listnode_t* node = allocator->allocated_resources.head; node != NULL; node = node->next) {
		_c_resource_t* resource = node->data;
		resource->destroy(resource->resource);
		// resource pointer free done in 'c_allocator_clear'
	}
	c_allocator_clear(allocator);
}
