/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENExpressionEvaluator.c
*	\brief		Tools to compute the numeric result of a string.
*	\author
*
******************************************************************************/

#include <CAENExpressionEvaluator.h>

#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <inttypes.h>
#include <stdbool.h>

#include <CAENMultiplatform.h>

int32_t _paren_count;
c_ee_err_t _err;
const char* _err_pos;

int64_t ParseAtom(const char ** expr);
int64_t ParseFactors(const char ** expr);
int64_t ParseSummands(const char ** expr);
int64_t ParseShiftands(const char ** expr);
int64_t ParseOR_ands(const char ** expr);
int64_t ParseAND_ands(const char ** expr);

// Parse a number or an expression in parenthesis
int64_t ParseAtom(const char * * expr) {
    bool isNegative = FALSE;
    int64_t res;
    char* end_ptr;
	
	// Skip spaces
	while (*(*expr) == ' ')
		(*expr)++;

	// Handle the sign before parenthesis (or before number)
	if (*(*expr) == '-') {
		isNegative = TRUE;
		(*expr)++;
	}
	if (*(*expr) == '+') {
		(*expr)++;
	}

	// Check if there is parenthesis
	if (*(*expr) == '(') {
		(*expr)++;
		_paren_count++;
		res = ParseAND_ands(expr);
		if (*(*expr) != ')') {
			// Unmatched opening parenthesis
			_err = EEE_PARENTHESIS;
			_err_pos = *expr;
			return 0;
		}
		(*expr)++;
		_paren_count--;
		return isNegative ? -res : res;
	}

	// It should be a number; convert it to double
	res = strtol(*expr, &end_ptr, 0);
	if (end_ptr == *expr) {
		// Report error
		_err = EEE_WRONG_CHAR;
		_err_pos = *expr;
		return 0;
	}
	// Advance the pointer and return the result
	*expr = end_ptr;
	return isNegative ? -res : res;
}

// Parse multiplication and division
int64_t ParseFactors(const char ** expr) {
	int64_t num1 = ParseAtom(expr);
	for(;;) {
        int64_t num2;
        char op;
        const char* pos;

		// Skip spaces
		while (*(*expr) == ' ')
			(*expr)++;
		// Save the operation and position
		op = *(*expr);
		pos = (*expr);
		if (op != '/' && op != '*' && op != '%')
			return num1;
		(*expr)++;
		num2 = ParseAtom(expr);
		// Perform the saved operation
		if (op == '/') { // division
			// Handle division by zero
			if (num2 == 0) {
				_err = EEE_DIVIDE_BY_ZERO;
				_err_pos = pos;
				return 0;
			}
			num1 = (int64_t)((double)num1 / (double)num2);
		}
        else if (op == '*') { // multiplication
			num1 *= num2;
        }
        else { // module
            num1 = num1 % num2;
        }
	}
}

// Parse addition and subtraction
int64_t ParseSummands(const char ** expr) {
	int64_t num1 = ParseFactors(expr);
	for(;;) {
        int64_t num2;
        char op;

		// Skip spaces
		while (*(*expr) == ' ')
			(*expr)++;
		op = *(*expr);
		if (op != '-' && op != '+')
			return num1;
		(*expr)++;
		num2 = ParseFactors(expr);
		if (op == '-')
			num1 -= num2;
		else
			num1 += num2;
	}
}

// Parse bit-shifting
int64_t ParseShiftands(const char ** expr) {
	int64_t num1 = ParseSummands(expr);
	for(;;) {
        int64_t num2;
        char op;

		// Skip spaces
		while (*(*expr) == ' ')
			(*expr)++;
		op = *(*expr);
		if (op != '<' && op != '>')
			return num1;
		(*expr)++;
        if (*(*expr) != op) { // '<' or '>' instead of '<<' or '>>'
            // Report error
		    _err = EEE_WRONG_CHAR;
		    _err_pos = *expr;
		    return 0;
        }
        (*expr)++;
		num2 = ParseSummands(expr);
		if (op == '<')
			num1 <<= num2;
		else
			num1 >>= num2;
	}
}

// Parse bitwise OR
int64_t ParseOR_ands(const char ** expr) {
	int64_t num1 = ParseShiftands(expr);
	for(;;) {
        int64_t num2;
        char op;

		// Skip spaces
		while (*(*expr) == ' ')
			(*expr)++;
		op = *(*expr);
		if (op != '|')
			return num1;
		(*expr)++;
		num2 = ParseShiftands(expr);
		num1 |= num2;
	}
}

// Parse bitwise AND
int64_t ParseAND_ands(const char ** expr) {
	int64_t num1 = ParseOR_ands(expr);
	for(;;) {
        int64_t num2;
        char op;

		// Skip spaces
		while (*(*expr) == ' ')
			(*expr)++;
		op = *(*expr);
		if (op != '&')
			return num1;
		(*expr)++;
		num2 = ParseOR_ands(expr);
		num1 &= num2;
	}
}

int64_t c_ee_exprEval(const char * expr) {
    int64_t res;
	_paren_count = 0;
	_err = EEE_NO_ERROR;
	res = ParseAND_ands(&expr);
	// Now, expr should point to '\0', and _paren_count should be zero
	if (_paren_count != 0 || *expr == ')') {
		_err = EEE_PARENTHESIS;
		_err_pos = expr;
		return 0;
	}
	if (*expr != '\0') {
		_err = EEE_WRONG_CHAR;
		_err_pos = expr;
		return 0;
	}
	return res;
}

c_ee_err_t c_ee_getErr(void) {
	return _err;
}

c_use_decl_annotations const char* c_ee_getErrPos(void) {
	return _err_pos;
}

#if 0
#define TEST_EXPRESSION(EXPR) assert(c_ee_exprEval(#EXPR) == (EXPR) && c_ee_getErr() == EEE_NO_ERROR)
void c_ee_testExprEval(void) {
	// Some simple expressions
	assert(c_ee_exprEval("1234") == 1234 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("1+2*3") == 7 && c_ee_getErr() == EEE_NO_ERROR);

	// Parenthesis
	assert(c_ee_exprEval("5*(4+4+1)") == 45 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("5*(2*(1+3)+1)") == 45 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("5*((1+3)*2+1)") == 45 && c_ee_getErr() == EEE_NO_ERROR);

	// Spaces
	assert(c_ee_exprEval("5 * ((1 + 3) * 2 + 1)") == 45 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("5 - 2 * ( 3 )") == -1 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("5 - 2 * ( ( 4 )  - 1 )") == -1 && c_ee_getErr() == EEE_NO_ERROR);

	// Sign before parenthesis
	assert(c_ee_exprEval("-(2+1)*4") == -12 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("-4*(2+1)") == -12 && c_ee_getErr() == EEE_NO_ERROR);

    // Other
    assert(c_ee_exprEval("0x1080 + 0x1 * 256") == (0x1080 + (0x1 << 8)) && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("(0x1080 + 0x2) << 8") == ((0x1080 + 0x2) << 8) && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("0x1080 | (0x3 << 8)") == (0x1080 | (0x3 << 8)) && c_ee_getErr() == EEE_NO_ERROR);
    TEST_EXPRESSION(0x1080 | (0x3 << 8));
    TEST_EXPRESSION(0x8180 + (4 * 4));
    TEST_EXPRESSION(0xF * 3 % 2);
    TEST_EXPRESSION((0xF * 3) % 2);
    TEST_EXPRESSION(0xF * (3 % 2));
    TEST_EXPRESSION(0xF % 5 * 2);
    TEST_EXPRESSION((0xF % 5) * 2);
    TEST_EXPRESSION(0xF % (5 * 2));
	
	// Fractional numbers
	/*assert(c_ee_exprEval("1.5/5") == 0.3 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("1/5e10") == 2e-11 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("(4-3)/(4*4)") == 0.0625 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("1/2/2") == 0.25 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("0.25 * .5 * 0.5") == 0.0625 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval(".25 / 2 * .5") == 0.0625 && c_ee_getErr() == EEE_NO_ERROR);*/
	
	// Repeated operators
	assert(c_ee_exprEval("1+-2") == -1 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("--2") == 2 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("2---2") == 0 && c_ee_getErr() == EEE_NO_ERROR);
	assert(c_ee_exprEval("2-+-2") == 4 && c_ee_getErr() == EEE_NO_ERROR);

	// === Errors ===
	// Parenthesis error
	c_ee_exprEval("5*((1+3)*2+1");
	assert(c_ee_getErr() == EEE_PARENTHESIS && strcmp(c_ee_getErrPos(), "") == 0);
	c_ee_exprEval("5*((1+3)*2)+1)");
	assert(c_ee_getErr() == EEE_PARENTHESIS && strcmp(c_ee_getErrPos(), ")") == 0);
	
	// Repeated operators (wrong)
	c_ee_exprEval("5*/2");
	assert(c_ee_getErr() == EEE_WRONG_CHAR && strcmp(c_ee_getErrPos(), "/2") == 0);
	
	// Wrong position of an operator
	c_ee_exprEval("*2");
	assert(c_ee_getErr() == EEE_WRONG_CHAR && strcmp(c_ee_getErrPos(), "*2") == 0);
	c_ee_exprEval("2+");
	assert(c_ee_getErr() == EEE_WRONG_CHAR && strcmp(c_ee_getErrPos(), "") == 0);
	c_ee_exprEval("2*");
	assert(c_ee_getErr() == EEE_WRONG_CHAR && strcmp(c_ee_getErrPos(), "") == 0);
	
	// Division by zero
	c_ee_exprEval("2/0");
	assert(c_ee_getErr() == EEE_DIVIDE_BY_ZERO && strcmp(c_ee_getErrPos(), "/0") == 0);
	c_ee_exprEval("3+1/(5-5)+4");
	assert(c_ee_getErr() == EEE_DIVIDE_BY_ZERO && strcmp(c_ee_getErrPos(), "/(5-5)+4") == 0);
	c_ee_exprEval("2/"); // Erroneously detected as division by zero, but that's ok for us
	assert(c_ee_getErr() == EEE_DIVIDE_BY_ZERO && strcmp(c_ee_getErrPos(), "/") == 0);
	
	// Invalid characters
	c_ee_exprEval("~5");
	assert(c_ee_getErr() == EEE_WRONG_CHAR && strcmp(c_ee_getErrPos(), "~5") == 0);
	c_ee_exprEval("5x");
	assert(c_ee_getErr() == EEE_WRONG_CHAR && strcmp(c_ee_getErrPos(), "x") == 0);

	// Multiply errors
	c_ee_exprEval("3+1/0+4$"); // Only one error will be detected (in this case, the last one)
	assert(c_ee_getErr() == EEE_WRONG_CHAR && strcmp(c_ee_getErrPos(), "$") == 0);
	c_ee_exprEval("3+1/0+4");
	assert(c_ee_getErr() == EEE_DIVIDE_BY_ZERO && strcmp(c_ee_getErrPos(), "/0+4") == 0);
	c_ee_exprEval("q+1/0)"); // ...or the first one
	assert(c_ee_getErr() == EEE_WRONG_CHAR && strcmp(c_ee_getErrPos(), "q+1/0)") == 0);
	c_ee_exprEval("+1/0)");
	assert(c_ee_getErr() == EEE_PARENTHESIS && strcmp(c_ee_getErrPos(), ")") == 0);
	c_ee_exprEval("+1/0");
	assert(c_ee_getErr() == EEE_DIVIDE_BY_ZERO && strcmp(c_ee_getErrPos(), "/0") == 0);
	
	// An emtpy string
	c_ee_exprEval("");
	assert(c_ee_getErr() == EEE_WRONG_CHAR && strcmp(c_ee_getErrPos(), "") == 0);
}
#endif

void c_ee_substituteIdxExpression(const char *expression, int32_t value, char *dest, size_t maxsize) {
	if (expression == NULL || dest == NULL)
		return;

	char *expr_mod_tmp = c_malloc(maxsize);
	if (expr_mod_tmp == NULL)
		goto QuitFunction;

    char wildcard[] = "idx";
    size_t wcnchar = sizeof(wildcard) - 1;
    char *ptr;

	dest[0] = '\0';
    strncat(dest, expression, maxsize - 1);

    while((ptr = strstr(dest, wildcard)) != NULL) {
        //char end = *(ptr + wcnchar);
        *ptr = '\0'; // temporary terminate string here
        snprintf(expr_mod_tmp, maxsize, "%s%"PRIi32"%s", dest, value, (ptr + wcnchar));
		dest[0] = '\0';
        strncat(dest, expr_mod_tmp, maxsize - 1);
    }

QuitFunction:
	c_free(expr_mod_tmp);
}

int32_t c_ee_modifyAndEvaluateExpression(const char *expression, int32_t idx, int64_t *result) {
	size_t originalsize = strlen(expression);
	const size_t additionalsize = 20;
	size_t totalsize = originalsize + additionalsize;
	char *expr_mod = c_malloc(totalsize);
	if (expr_mod == NULL)
		return -1;

	// substitue all "idx" entries in 'expression' with the value of 'idx'.
	c_ee_substituteIdxExpression(expression, idx, expr_mod, totalsize);

	int64_t res = c_ee_exprEval(expr_mod);
	if ((res == 0) && (c_ee_getErr() != EEE_NO_ERROR)) {
		c_free(expr_mod);
		return -1;
	}
	*result = res;
	c_free(expr_mod);
	return 0;
}
