/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
******************************************************************************
*
*	\file		CAENLinkedList.c
*	\brief		Utilities to implement linked lists
*	\author		Francesco Pepe
*
******************************************************************************/

#include <CAENLinkedList.h>

#include <CAENMultiplatform.h>

c_use_decl_annotations c_linkedlist_t c_linkedlist_create() {
	c_linkedlist_t res = { NULL, NULL };
	return res;
}

c_use_decl_annotations c_listnode_t* c_linkedlist_insert_after(c_listnode_t* node, void* newElement) {
	c_listnode_t* res = (c_listnode_t*)c_malloc(sizeof(c_listnode_t));
	if (res == NULL)
		return NULL;
	res->data = newElement;
	res->prev = node;
	if (node != NULL) {
		res->next = node->next;
		node->next = res;
		if (res->next != NULL)
			res->next->prev = res;
	}
	else {
		res->next = NULL;
	}
	return res;
}

c_use_decl_annotations c_listnode_t* c_linkedlist_insert_before(c_listnode_t* node, void* newElement) {
	c_listnode_t* res = (c_listnode_t*)c_malloc(sizeof(c_listnode_t));
	if (res == NULL)
		return NULL;
	res->data = newElement;
	res->next = node;
	if (node != NULL) {
		res->prev = node->prev;
		node->prev = res;
		if (res->prev != NULL)
			res->prev->next = res;
	}
	else {
		res->prev = NULL;
	}
	return res;
}

c_use_decl_annotations c_Utility_ErrorCode_t c_linkedlist_insert_last(c_linkedlist_t* list, void* newElement) {
	c_listnode_t* res = NULL;

	if (list == NULL)
		return c_Utility_ErrorCode_NULL;
	if ((list->head == NULL || list->tail == NULL) && list->head != list->tail)
		return c_Utility_ErrorCode_GenericError;

	res = c_linkedlist_insert_after(list->tail, newElement);
	if (res == NULL)
		return c_Utility_ErrorCode_Memory;
	list->tail = res;
	if (list->head == NULL)
		list->head = list->tail;

	return c_Utility_ErrorCode_Success;
}

c_use_decl_annotations c_Utility_ErrorCode_t c_linkedlist_insert_first(c_linkedlist_t* list, void* newElement) {
	c_listnode_t* res = NULL;

	if (list == NULL)
		return c_Utility_ErrorCode_NULL;
	if ((list->head == NULL || list->tail == NULL) && list->head != list->tail)
		return c_Utility_ErrorCode_GenericError;

	res = c_linkedlist_insert_before(list->head, newElement);
	if (res == NULL)
		return c_Utility_ErrorCode_Memory;
	list->head = res;
	if (list->tail == NULL)
		list->tail = list->head;

	return c_Utility_ErrorCode_Success;
}

c_use_decl_annotations void* c_linkedlist_delete(c_linkedlist_t* list, c_listnode_t* node) {
	void* data;
	if (node == NULL)
		return NULL;
	if (node->prev != NULL)
		node->prev->next = node->next;
	if (node->next != NULL)
		node->next->prev = node->prev;
	// Update head and tail references if needed.
	if (list->head == node)
		list->head = node->next;
	if (list->tail == node)
		list->tail = node->prev;
	// not really useful
	node->next = NULL;
	node->prev = NULL;
	data = node->data;
	c_free(node);
	return data;
}

static int _simplecompare(const void* d1, const void* d2) {
	return d1 == d2;
}

c_use_decl_annotations void* c_linkedlist_delete_data_compare(c_linkedlist_t* list, void* data, c_listdata_comparator_t cmp) {
	c_listnode_t* node;
	if (list == NULL)
		return NULL;
	node = list->head;
	while (node != NULL && cmp(node->data, data) == 0)
		node = node->next;
	if (node == NULL)
		return NULL;
	return c_linkedlist_delete(list, node);
}

c_use_decl_annotations void* c_linkedlist_delete_data(c_linkedlist_t* list, void* data) {
	return c_linkedlist_delete_data_compare(list, data, _simplecompare);
}

c_use_decl_annotations void* c_linkedlist_delete_last(c_linkedlist_t* list) {
	c_listnode_t *tail = NULL;

	if (list == NULL)
		return NULL;
	if ((list->head == NULL || list->tail == NULL) && list->head != list->tail)
		return NULL;

	tail = list->tail;
	if (tail == NULL)
		return NULL;

	return c_linkedlist_delete(list, tail);
}

c_use_decl_annotations void* c_linkedlist_delete_first(c_linkedlist_t* list) {
	c_listnode_t* head = NULL;

	if (list == NULL)
		return NULL;
	if ((list->head == NULL || list->tail == NULL) && list->head != list->tail)
		return NULL;

	head = list->head;
	if (head == NULL)
		return NULL;

	return c_linkedlist_delete(list, head);
}
