/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENLogger.c
*	\brief		Logger implemented in C
*	\author
*
******************************************************************************/

#include <CAENLogger.h>

#include <stdlib.h>

#include <CAENMultiplatform.h>
#include <CAENThread.h>

#ifdef LOGROTATION
#include <fcntl.h>
// TODO change the default fifo name to something unambiguous between different processes using c_logger.
#define LOGGER_ROTATE_FIFO_NAME "/tmp/rotatefifo"
static int32_t rotatepipe = -1;
#endif

struct global_logger {
	FILE *file;
	FILE *file_default;
	char *filename;
	char *lastHeader;
	c_mutex_t *mutex;
	c_logger_OpenMode_t openMode;
	uint32_t configuredMask; // set to lConfiguredMask when configured
};

static const uint32_t lConfiguredMask = 0xcafecae0;

// default log level is all (for retrocomp)
static uint32_t lSeverityMask = c_logger_Severity_ERROR | c_logger_Severity_WARNING | c_logger_Severity_INFO | c_logger_Severity_DEBUG;
static uint64_t lStartTimeMs = 0;
static struct global_logger *lGlobal = NULL;
static const char lSeverityMap[][3] = {
	"UU", // UNKNOWN
	"EE", // ERROR
	"WW", // WARNING
	"UU", // UNKNOWN
	"II", // INFO
	"UU", // UNKNOWN
	"UU", // UNKNOWN
	"UU", // UNKNOWN
	"DD", // DEBUG
};

static int32_t _lopenfile(void) {
	if (lGlobal->file != NULL)
		return c_logger_Success; // assume already opened

	const char *openFlags;

	switch (lGlobal->openMode) {
	case c_logger_OpenMode_W:  openFlags = "w";  break;
	case c_logger_OpenMode_A:  openFlags = "a";  break;
	case c_logger_OpenMode_RW: openFlags = "r+"; break;
	case c_logger_OpenMode_RA: openFlags = "a+"; break;
	default: return c_logger_WrongOpenMode;
	}

	lGlobal->file = fopen(lGlobal->filename, openFlags);
	if (lGlobal->file == NULL)
		return c_logger_FileOpenError;
	return c_logger_Success;
}

#ifdef LOGROTATION
static void _lcheckrotation(void) {
	if (rotatepipe == -1)
		rotatepipe = open(LOGGER_ROTATE_FIFO_NAME, O_RDONLY | O_NDELAY);
	if (rotatepipe != -1) {
		// Read from rotatepipe and check if a rotation has been requested
		char rotbuff[11];
		ssize_t numb = read(rotatepipe, rotbuff, 10);
		rotbuff[numb] = '\0';
		if (strcmp(rotbuff, "ROTATE\n") == 0) {
			// logrotate changed the filename. Close and reopen it.
			if (lGlobal != NULL) {
				// IMPORTANT NOTE: no need to lock here since logger is already locked.
				if (lGlobal->file != NULL) {
					FILE *oldFileBkp = lGlobal->file;
					lGlobal->file = NULL;
					if (_lopenfile() != c_logger_Success) {
						// ERROR opening new file. keep old one..
						// Check if we can log an error even inside the logger itself!
						lGlobal->file = oldFileBkp;
					}
					else {
						fclose(oldFileBkp);
					}
				}
			}
		}
	}
}
#endif

static bool _lcheckmask(uint32_t mask) {
	return (mask == lConfiguredMask);
}

static bool _lisconfigured(void) {
	bool configured = FALSE;

	if (lGlobal != NULL)
		configured = _lcheckmask(lGlobal->configuredMask);

	return configured;
}

// secure version withmutex lock
static bool _lisconfigured_locked(void) {
	bool configured = FALSE;

	if (lGlobal != NULL) {
		// _lconfigure lock the mutex while configuring
		c_mutex_lock(lGlobal->mutex);
		configured = _lcheckmask(lGlobal->configuredMask);
		c_mutex_unlock(lGlobal->mutex);
	}

	return configured;
}

static void _lfreeglobal(void) {
#ifdef LOGROTATION
	if (rotatepipe != -1) {
		close(rotatepipe);
		rotatepipe = -1;
	}
#endif

	if (lGlobal == NULL)
		return;

	c_mutex_lock(lGlobal->mutex);

	// close the file if opened
	if (lGlobal->file != NULL)
		fclose(lGlobal->file);

	c_mutex_unlock(lGlobal->mutex);

	c_mutex_destroy(lGlobal->mutex);
	c_free(lGlobal->mutex);
	c_free(lGlobal->lastHeader);
	c_free(lGlobal->filename);
	c_free(lGlobal);

	lGlobal = NULL;
}

static int32_t c_attribute_format(4, 0) _lvprintf(const char *module_tree, c_logger_Severity s, int32_t line, const char *__restrict fmt, va_list args) {
#ifdef LOGROTATION
	_lcheckrotation();
#endif

	const double time = (double)(c_get_time() - lStartTimeMs) * 0.001;

	// [time in seconds, 3 decimals and 12 chars in total][severity][source_file]
	if (fprintf(lGlobal->file, "[%12.3f][%s][%s::%"PRIi32"]: ", time, lSeverityMap[s], module_tree, line) < 0)
		return c_logger_PrintError;

	if (vfprintf(lGlobal->file, fmt, args) < 0)
		return c_logger_PrintError;

	if (fputc('\n', lGlobal->file) == EOF)
		return c_logger_PrintError;

	if (fflush(lGlobal->file) == EOF)
		return c_logger_PrintError;

	return c_logger_Success;
}

// secure version with null check and mutex lock
static int32_t _lvprintf_s(const char *module_tree, c_logger_Severity s, int32_t line, const char *__restrict fmt, va_list args) {
	if (module_tree == NULL || fmt == NULL)
		return c_logger_InvalidArgument;

	c_mutex_lock(lGlobal->mutex);
	int32_t res = _lvprintf(module_tree, s, line, fmt, args);
	c_mutex_unlock(lGlobal->mutex);

	return res;
}

static int32_t _lconfigure(const char *filename, c_logger_OpenMode_t om) {
	int32_t res = c_logger_Success;
	bool opening = FALSE;
	bool mutex_locked = FALSE;
	char *expandedFileName = NULL;

	// use locked version, as another thread may be in this same function
	// and lGlobal could be != NULL while still not configured.
	if (_lisconfigured_locked()) {
		// logger already configured
		res = c_logger_LoggerAlreadyConfigured;
		goto QuitFunction;
	}

	lGlobal = c_malloc(sizeof(*lGlobal));
	if (lGlobal == NULL) {
		res = c_logger_OutOfMemory;
		goto QuitFunction;
	}

	opening = TRUE;

	// init the mutex
	lGlobal->mutex = c_malloc(sizeof(*lGlobal->mutex));
	if (lGlobal->mutex == NULL) {
		res = c_logger_OutOfMemory;
		goto QuitFunction;
	}

	c_mutex_init(lGlobal->mutex);
	c_mutex_lock(lGlobal->mutex);
	mutex_locked = TRUE;

	// Get optional verbosity level from from env. variable
	// Otherwise, keep default value
	char *verbosity_env = getenv(LOG_ENV_LEVEL);
	if (verbosity_env != NULL) {
		uint32_t level = (uint32_t)strtoul(verbosity_env, NULL, 0);
		lSeverityMask = (UINT32_C(1) << level) - 1;
	}

	// Get optional filename from env. variable
	char *filename_env = getenv(LOG_ENV_FILENAME);

	const char *real_filename = (filename_env != NULL) ? filename_env : filename;

	// expand env variables in the string filename
	// allocate expandedFileName inside!
	expandedFileName = c_getExpandedString(real_filename);
	if (expandedFileName == NULL) {
		res = c_logger_InvalidArgument;
		goto QuitFunction;
	}

	// set the filename
	lGlobal->filename = expandedFileName;

	// initialize the last header string
	lGlobal->lastHeader = c_calloc(2, sizeof(*lGlobal->lastHeader));
	if (lGlobal->lastHeader == NULL) {
		res = c_logger_OutOfMemory;
		goto QuitFunction;
	}

	// open the file
	lGlobal->openMode = om;
	lGlobal->file = NULL;
	res = _lopenfile();
	if (res != c_logger_Success)
		goto QuitFunction;

	// Store f pointer here, to be used by c_lresumefile() after a call to c_lchangefile()
	lGlobal->file_default = lGlobal->file;

	// set configured mask
	lGlobal->configuredMask = lConfiguredMask;

QuitFunction:
	if (mutex_locked)
		c_mutex_unlock(lGlobal->mutex);

	if (res != c_logger_Success && opening)
		_lfreeglobal();

	return res;
}

// PUBLIC

void c_lsetst(uint64_t time) {
	lStartTimeMs = time;
}

int32_t c_lchangefile(FILE *file) {
	int32_t res = c_logger_Success;

	// First of all check if the global internal logger is configured
	if (!_lisconfigured()) {
		res = c_logger_LoggerNotConfigured;
		goto QuitFunction;
	}

	// Set lGlobal->file to new file
	lGlobal->file = file;

QuitFunction:
	return res;
}

int32_t c_lresumefile(void) {
	int32_t res = c_logger_Success;

	// First of all check if the global internal logger is configured
	if (!_lisconfigured()) {
		res = c_logger_LoggerNotConfigured;
		goto QuitFunction;
	}

	// Set lGlobal->file to new file
	lGlobal->file = lGlobal->file_default;

QuitFunction:
	return res;
}

int32_t c_lsetsm(uint32_t sevMask) {
	lSeverityMask = sevMask;
	return c_logger_Success;
}

int32_t c_lgetsm(uint32_t *sevMask) {
	if (sevMask != NULL) {
		*sevMask = lSeverityMask;
		return c_logger_Success;
	}
	return c_logger_InvalidArgument;
}

int32_t c_lprintf(const c_locallogger_t *locallogger, c_logger_Severity s, int32_t line, const char *__restrict fmt, ...) {
	va_list args;
	va_start(args, fmt);
	int32_t ret = c_lvprintf(locallogger, s, line, fmt, args);
	va_end(args);
	return ret;
}

int32_t c_lvprintf(const c_locallogger_t *locallogger, c_logger_Severity s, int32_t line, const char *__restrict fmt, va_list args) {
	int32_t res = c_logger_Success;

	if (locallogger == NULL || fmt == NULL) {
		res = c_logger_InvalidArgument;
		goto QuitFunction;
	}

	if (!_lisconfigured()) {
		res = _lconfigure(locallogger->filename, c_logger_OpenMode_W);
		if (res != c_logger_Success)
			goto QuitFunction;
	}

	if (lSeverityMask & s) {
		res = _lvprintf_s(locallogger->moduletree, s, line, fmt, args);
		if (res != c_logger_Success)
			goto QuitFunction;
	}

QuitFunction:
	return res;
}

void c_ldeinit(void) {
	// deinit all allocated c_logger_t*
	if (!_lisconfigured_locked())
		return;

	// deinit lGlobal
	_lfreeglobal();
}
