/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2014 rxi
*
*	This file is part of the CAEN Utility.
*
*	This file is distributed under the MIT License.
*
*	SPDX-License-Identifier: MIT
*
***************************************************************************//*!
*
*	\file		CAENMap.c
*	\brief		Hash table. Implementation from rxi's "map" project.
*	\author		Giovanni Cerretani, rxi
*
******************************************************************************/

#include <CAENMap.h>

#include <stdlib.h>
#include <string.h>

#include <CAENMultiplatform.h>


struct c_map_node_t {
	char *key;
	void *value;
	uint32_t hash;
	c_map_node_t *next;
};


static uint32_t djb2a(const char *str) {
	uint32_t hash = UINT32_C(5381);
	char c;
	while ((c = *str++) != '\0')
		hash = (UINT32_C(33) * hash) ^ c;
	return hash;
}


static c_map_node_t *map_newnode(const char *key, const void *value, size_t vsize) {
	c_map_node_t *node = c_malloc(sizeof(*node));
	if (node == NULL)
		return NULL;

	node->key = c_strdup(key);
	if (node->key == NULL) {
		c_free(node);
		return NULL;
	}

	node->value = c_malloc(vsize);
	if (node->value == NULL) {
		c_free(node->key);
		c_free(node);
		return NULL;
	}

	c_memcpy(node->value, value, vsize);

	node->hash = djb2a(key);

	node->next = NULL;

	return node;
}


static void map_deletenode(c_map_node_t *node) {
	if (node == NULL)
		return;
	c_free(node->value);
	c_free(node->key);
	c_free(node);
}


static size_t map_bucketidx(const c_map_base_t *m, uint32_t hash) {
	/* If the implementation is changed to allow a non-power-of-2 bucket count,
	* the line below should be changed to use mod instead of AND */
	return (size_t)hash & (m->nbuckets - 1);
}


static void map_addnode(c_map_base_t *m, c_map_node_t *node) {
	size_t n = map_bucketidx(m, node->hash);
	node->next = m->buckets[n];
	m->buckets[n] = node;
}


static int32_t map_resize(c_map_base_t *m, size_t nbuckets) {
	/* Chain all nodes together */
	c_map_node_t *nodes = NULL;
	while (m->nbuckets--) {
		c_map_node_t *node = m->buckets[m->nbuckets];
		while (node != NULL) {
			c_map_node_t *next = node->next;
			node->next = nodes;
			nodes = node;
			node = next;
		}
	}
	/* Reset buckets */
	c_map_node_t **buckets = c_realloc(m->buckets, sizeof(*m->buckets) * nbuckets);
	if (buckets != NULL) {
		m->buckets = buckets;
		m->nbuckets = nbuckets;
	}
	if (m->buckets != NULL) {
		c_zeromem(m->buckets, sizeof(*m->buckets) * m->nbuckets);
		/* Re-add nodes to buckets */
		c_map_node_t *node = nodes;
		while (node != NULL) {
			c_map_node_t *next = node->next;
			map_addnode(m, node);
			node = next;
		}
	}
	/* Return error code if realloc() failed */
	return (buckets == NULL) ? c_Utility_ErrorCode_Map : c_Utility_ErrorCode_Success;
}


static c_map_node_t **map_getref(const c_map_base_t *m, const char *key) {
	if (m->nbuckets > 0) {
		const uint32_t hash = djb2a(key);
		c_map_node_t **next = &m->buckets[map_bucketidx(m, hash)];
		while (*next != NULL) {
			if ((*next)->hash == hash && !strcmp((*next)->key, key)) {
				return next;
			}
			next = &(*next)->next;
		}
	}
	return NULL;
}


int32_t _map_init(c_map_base_t *m, size_t nreserved) {
	// Clear the memory.
	m->buckets = NULL;
	m->nbuckets = 0;
	m->nnodes = 0;
	m->initialized = TRUE;

	// Pre-initialize buckets array only if reserved is a power of 2.
	// Anyway, it is reallocated automatically when needed.
	if ((nreserved > 0) && !(nreserved & (nreserved - 1)))
		return map_resize(m, nreserved);
	else
		return c_Utility_ErrorCode_Success;
}


void _map_deinit(c_map_base_t *m) {
	if (m == NULL)
		return;
	if (!m->initialized)
		return;

	while (m->nbuckets--) {
		c_map_node_t *node = m->buckets[m->nbuckets];
		while (node != NULL) {
			c_map_node_t *next = node->next;
			map_deletenode(node);
			node = next;
		}
	}
	c_free(m->buckets);
	m->initialized = FALSE;
}


c_use_decl_annotations void *_map_get(const c_map_base_t *m, const char *key) {
	if (m == NULL || key == NULL)
		return NULL;
	if (!m->initialized)
		return NULL;

	c_map_node_t **next = map_getref(m, key);
	return next ? (*next)->value : NULL;
}


int32_t _map_set(c_map_base_t *m, const char *key, const void *value, size_t vsize) {
	if (m == NULL || key == NULL || value == NULL)
		return c_Utility_ErrorCode_Map;
	if (!m->initialized)
		return c_Utility_ErrorCode_Map;

	/* Find & replace existing node */
	c_map_node_t **next = map_getref(m, key);
	if (next != NULL) {
		c_memcpy((*next)->value, value, vsize);
		return c_Utility_ErrorCode_Success;
	}
	/* Add new node */
	c_map_node_t *node = map_newnode(key, value, vsize);
	if (node == NULL)
		goto fail;
	if (m->nnodes >= m->nbuckets) {
		size_t n = (m->nbuckets > 0) ? (m->nbuckets << 1) : 1U;
		int32_t err = map_resize(m, n);
		if (err != c_Utility_ErrorCode_Success)
			goto fail;
	}
	map_addnode(m, node);
	m->nnodes++;
	return c_Utility_ErrorCode_Success;
fail:
	map_deletenode(node);
	return c_Utility_ErrorCode_Map;
}


void _map_remove(c_map_base_t *m, const char *key) {
	if (m == NULL || key == NULL)
		return;
	if (!m->initialized)
		return;

	c_map_node_t **next = map_getref(m, key);
	if (next != NULL) {
		c_map_node_t *node = *next;
		*next = (*next)->next;
		map_deletenode(node);
		m->nnodes--;
	}
}

c_use_decl_annotations size_t _map_size(const c_map_base_t *m) {
	if (m == NULL)
		return 0;
	return m->nnodes;
}

c_use_decl_annotations c_map_iter_t _map_iter(void) {
	c_map_iter_t iter;
	// we init bucketidx to max, so its increment will become from 0.
	// see function _map_next() for details.
	iter.bucketidx = SIZE_MAX;
	iter.node = NULL;
	return iter;
}

c_use_decl_annotations const char* _map_next(const c_map_base_t *m, c_map_iter_t *iter) {
	if (m == NULL || iter == NULL)
		return NULL;
	if (!m->initialized)
		return NULL;

	if (iter->node != NULL) {
		iter->node = iter->node->next;
		if (iter->node == NULL)
			goto nextBucket;
	} else {
	nextBucket:
		do {
			if (++iter->bucketidx >= m->nbuckets)
				return NULL;
			iter->node = m->buckets[iter->bucketidx];
		} while (iter->node == NULL);
	}
	return iter->node->key;
}
