/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENMultiplatform.c
*	\brief		Generic wrappers to platform-dependent functions
*	\author
*
******************************************************************************/

#ifdef _WIN32
#include <WinSock2.h>
#endif

#include <CAENMultiplatform.h>

#include <errno.h>
#include <inttypes.h>
#include <limits.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include <CAENLogger.h>
#include <CAENMap.h>

#ifndef _WIN32
#include <dlfcn.h> // dlopen, dlerror, dlsym, dlclose
#include <malloc.h> // malloc_usable_size
#include <signal.h>
#include <spawn.h>
#include <strings.h> // strcasecmp
#include <termios.h> // tcsetattr, termios, tcgetattr
#include <unistd.h> // for STDIN_FILENO, getcwd, access
#include <sys/stat.h>
#include <sys/time.h> // struct timeval, gettimeofday
#include <sys/types.h>
#include <sys/wait.h>
#else
#include <conio.h> // _getch
#include <direct.h> // _getcwd
#include <process.h> // _getpid
#include <tlhelp32.h> // CreateToolhelp32Snapshot, PROCESSENTRY32, ...
#include <sys/stat.h>
#include <sys/timeb.h> // _ftime_s
#include <sysinfoapi.h> // GetSystemInfo

// see c_popcount implementation
#if defined(_M_IX86) || (defined(_M_X64) && !defined(_M_ARM64EC))
#define _is_machine_x86_64
#include <intrin.h>
#ifndef __AVX__
#include <isa_availability.h>
#endif
#endif

#endif

#define KILL_BUFSIZE (128)

INIT_C_LOGGER("CAENMultiplatformLog.txt", "CAENMultiplatform.c");

// Decode Error
static void _decodeError(const char *functName, int32_t resCode) {
	const char *ErrMsg;

	switch (resCode) {
	case MP_code_Generic:
		ErrMsg = "Generic error.";
		break;
	case MP_code_LibLoad:
		ErrMsg = "Error loading dynamic library.";
		break;
	case MP_code_LibClose:
		ErrMsg = "Error closing dynamic library.";
		break;
	default:
		ErrMsg = "Unknown error.";
		break;
	}

	logMsg(c_logger_Severity_ERROR, "From %s: %s", functName, ErrMsg);
}

c_use_decl_annotations void* c_malloc(size_t size) {
	void *res;
#ifdef _WIN32
	res = HeapAlloc(GetProcessHeap(), 0, (SIZE_T)size);
#else
	res = malloc(size);
#endif
	return res;
}

c_use_decl_annotations void* c_realloc(void* ptr, size_t size) {
	void *res;
#ifdef _WIN32
	/*
	 * On realloc(), if ptr is NULL, the behavior is the same as calling malloc(size).
	 * This feature is not true for HeapReAlloc(), that seems to return NULL.
	 */
	if (c_likely(ptr != NULL))
		res = HeapReAlloc(GetProcessHeap(), 0, ptr, (SIZE_T)size);
	else
		res = c_malloc(size);
#else
	res = realloc(ptr, size);
#endif
	return res;
}

c_use_decl_annotations void* c_calloc(size_t nmemb, size_t size) {
	void *res;
#ifdef _WIN32
	res = HeapAlloc(GetProcessHeap(), HEAP_ZERO_MEMORY, (SIZE_T)(nmemb * size));
#else
	res = calloc(nmemb, size);
#endif
	return res;
}

void c_free(void *ptr) {
#ifdef _WIN32
	/*
	 * On 2021 the HeapFree documentation has been update: now the last parameter
	 * can be NULL. Previously it was explictly reported as undefined behavior.
	 * Probably it was just a documentation bug, because the last parameter is
	 * marked with the SAL macro _Frees_ptr_opt_ also on the old SDK version
	 * 10.0.16299.0, currently used to compile this library on Windows. It is
	 * reasonable to remove the check.
	 * See:
	 * - https://docs.microsoft.com/en-us/windows/win32/api/heapapi/nf-heapapi-heapfree
	 * - https://github.com/MicrosoftDocs/sdk-api/pull/610
	 *
	 * The return value is not used for consistency with standard free.
	 */
	const BOOL res = HeapFree(GetProcessHeap(), 0, ptr);
	c_unused_parameter(res);
#else
	free(ptr);
#endif
}

c_use_decl_annotations size_t c_malloc_size(void *ptr) {
	size_t res;
#ifdef _WIN32
	/*
	 * If the function fails, the return value is (SIZE_T)-1.
	 * The check for null is for consistency with GNU malloc_usable_size() behavior.
	 * See:
	 * - https://docs.microsoft.com/en-us/windows/win32/api/heapapi/nf-heapapi-heapsize
	 */
	if (c_unlikely(ptr == NULL))
		return 0;
	const SIZE_T s = HeapSize(GetProcessHeap(), 0, ptr);
	res = (s == (SIZE_T)-1) ? 0 : (size_t)s;
#else
	/*
	 * Surprisingly, malloc_usable_size() argument is not marked as const.
	 * If ptr is NULL, 0 is returned.
	 */
	res = malloc_usable_size(ptr);
#endif
	return res;
}

c_attribute_nonnull(1) static char* _strndup(const char *str, size_t size) {
	char *res = c_malloc(size + 1);
	if (c_likely(res != NULL)) {
		res[size] = '\0';
		res = memcpy(res, str, size); // c_memcpy is useless here, res and str cannot be null
	}
	return res;
}

// Version of strdup to be used with c_free()
c_use_decl_annotations char* c_strdup(const char *str) {
	if (c_unlikely(str == NULL))
		return NULL;
	const size_t len = strlen(str);
	return _strndup(str, len);
}

// Version of strndup to be used with c_free()
c_use_decl_annotations char* c_strndup(const char *str, size_t size) {
	if (c_unlikely(str == NULL))
		return NULL;
	const size_t len = strnlen(str, size);
	return _strndup(str, len);
}

// Get Unix time in milliseconds since 00:00:00 UTC, January 1, 1970.
c_use_decl_annotations uint64_t c_get_time(void) {
	uint64_t sec;
	uint64_t msec;
#ifdef _WIN32
	struct _timeb tv;
	errno_t ret = _ftime_s(&tv);
	if (ret != 0) {
		logMsg(c_logger_Severity_DEBUG, "%s(): _ftime_s() failed. Error %d", __func__, ret);
		return 0;
	}
	sec = (uint64_t)tv.time;
	msec = (uint64_t)tv.millitm;
#else
	struct timeval tv;
	int ret = gettimeofday(&tv, NULL);
	if (c_unlikely(ret != 0)) {
		logMsg(c_logger_Severity_DEBUG, "%s(): gettimeofday() failed. Error %d", __func__, errno);
		return 0;
	}
	sec = (uint64_t)tv.tv_sec;
	msec = (uint64_t)tv.tv_usec / UINT64_C(1000);
#endif
	return (sec * UINT64_C(1000)) + msec;
}

c_use_decl_annotations int c_getch(void) {
#ifdef _WIN32
	return _getch();
#else
	struct termios oldattr;
	if (tcgetattr(STDIN_FILENO, &oldattr) == -1) perror(NULL);
	struct termios newattr = oldattr;
	newattr.c_lflag &= ~(ICANON | ECHO);
	newattr.c_cc[VTIME] = 0;
	newattr.c_cc[VMIN] = 1;
	if (tcsetattr(STDIN_FILENO, TCSANOW, &newattr) == -1) perror(NULL);
	const int ch = getchar();
	if (tcsetattr(STDIN_FILENO, TCSANOW, &oldattr) == -1) perror(NULL);
	return ch;
#endif
}

c_use_decl_annotations int c_kbhit(void) {
#ifdef _WIN32
	return _kbhit();
#else
	struct termios oldattr;
	if (tcgetattr(STDIN_FILENO, &oldattr) == -1) perror(NULL);
	struct termios newattr = oldattr;
	newattr.c_lflag &= ~(ICANON | ECHO);
	newattr.c_cc[VTIME] = 0;
	newattr.c_cc[VMIN] = 1;
	if (tcsetattr(STDIN_FILENO, TCSANOW, &newattr) == -1) perror(NULL);
	/* check stdin (fd 0) for activity */
	fd_set read_handles;
	FD_ZERO(&read_handles);
	FD_SET(0, &read_handles);
	struct timeval timeout;
	timeout.tv_sec = timeout.tv_usec = 0;
	int status = select(0 + 1, &read_handles, NULL, NULL, &timeout);
	if (status < 0) perror(NULL);
	if (tcsetattr(STDIN_FILENO, TCSANOW, &oldattr) == -1) perror(NULL);
	return status;
#endif
}

c_use_decl_annotations int c_access(const char* pathname, int mode) {
#ifdef _WIN32
	// _access check for NULL
	return _access(pathname, mode);
#else
	if (pathname == NULL) {
		errno = EINVAL;
		return -1;
	}
	return access(pathname, mode);
#endif
}

char* c_getcwd(char* buf, size_t size) {
	// allocation of buf == NULL, supported on both Windows and Linux,
	// must be handled because allocation is done with malloc on both platforms,
	// making c_free unsuitable.
	if (buf == NULL) {
		char *local_buf;
		// Platform dependent behaviour here:
		// - Windows allocates AT LEAST size (works even if size required by _getcwd is > size)
		// - Linux allocates AT MOST size, or return NULL
		// For this reason, on Linux we use PATH_MAX as argument.
#ifdef _WIN32
		if (size > (size_t)INT_MAX)
			return NULL;
		local_buf = _getcwd(NULL, (int)size);
#else
		// we use the maximum size to be sure getcwd does not fail
		// final size is not this one, as we use strdup later
		local_buf = getcwd(NULL, PATH_MAX);
#endif
		if (local_buf == NULL) {
			errno = ENOMEM;
			return NULL;
		}
		buf = c_strdup(local_buf);
		if (buf == NULL) {
			free(local_buf); // free is required, not c_free
			errno = ENOMEM;
			return NULL;
		}
		// getcwd allocates at least `size` bytes
		if (c_malloc_size(buf) < size) {
			char *realloc_buf = c_realloc(buf, size);
			if (realloc_buf == NULL) {
				c_free(buf);
				free(local_buf); // free is required, not c_free
				errno = ENOMEM;
				return NULL;
			}
			buf = realloc_buf;
		}
		free(local_buf); // free is required, not c_free
		return buf;
	} else {
#ifdef _WIN32
		if (size > (size_t)INT_MAX)
			return NULL;
		return _getcwd(buf, (int)size);
#else
		return getcwd(buf, size);
#endif
	}
}

int c_chdir(const char* path) {
#ifdef _WIN32
	return _chdir(path);
#else
	return chdir(path);
#endif
}

int c_rmdir(const char* path) {
#ifdef _WIN32
	return _rmdir(path);
#else
	return rmdir(path);
#endif
}

c_use_decl_annotations int64_t c_getpid(void) {
	// pid_t on Linux is a signed interer type, so we use int64_t to be conservative
	// on windows _getpid return int
#ifdef _WIN32
	return (int64_t)_getpid();
#else
	return (int64_t)getpid();
#endif
}

c_use_decl_annotations int c_strcasecmp(const char* s1, const char* s2) {
#ifdef _WIN32
	return _stricmp(s1, s2);
#else
	return strcasecmp(s1, s2);
#endif
}

c_use_decl_annotations int c_strncasecmp(const char* s1, const char* s2, size_t n) {
#ifdef _WIN32
	return _strnicmp(s1, s2, n);
#else
	return strncasecmp(s1, s2, n);
#endif
}

// The variable must be in the form $(VARNAME)
c_use_decl_annotations char* c_getExpandedString(const char *filename) {
	if (filename == NULL)
		return NULL;
    char *result = NULL;
    char *varValue = NULL, *varName = NULL;
	char *p1 = strchr(filename, '$');
	char *p2 = strchr(filename, ')');
	char empty[] = "";

	size_t varNameLength, resultLength;

    if (p1 != NULL && p2 != NULL && p2 - p1 > 2) {
		varNameLength = p2 - p1 - 2; // -2 to remove "$("
        varName = c_malloc(varNameLength + 1);
		if (varName == NULL)
			goto QuitFunction;

        strncpy(varName, p1 + 2, varNameLength); // +2 to add "$("
        varName[varNameLength] = '\0';

        // Get the variable value
        varValue = getenv(varName);
        if(varValue == NULL) // use empty string if var not found
			varValue = empty;

        // Build the result
        resultLength = strlen(filename) - (strlen(varName) + 3) + strlen(varValue) + 1;
        result = c_malloc(resultLength);
		if (result == NULL)
			goto QuitFunction;

        strncpy(result, filename, p1 - filename);
		// Put NUL terminator that could be missing in case p1 - filename == 0.
		result[p1 - filename] = '\0';
        strcat(result, varValue);
        strcat(result, p2 + 1);
        if (strchr(result, '$')) {
            char* tmpResult = c_getExpandedString(result);
            c_free(result);
            result = tmpResult;
        }
    }
    else {
        result = c_strdup(filename);
		if (result == NULL)
			goto QuitFunction;
    }

QuitFunction:
    c_free(varName);
    return result;
}

int32_t c_getDirName(char *dest, const char *source) {
	const char dirseps[] = DIRSEPS_CHAR;

    logMsg(c_logger_Severity_DEBUG, "c_getDirName: source is: %s", source);
    if (dest != source)
        strcpy(dest, source);
    char *p1 = strrchr(dest, dirseps[0]);
#ifdef _WIN32
    // in Windows paths may also have '/'!
    char *p2 = strrchr(dest, dirseps[1]);
    char *last = max(p1, p2);
#else
    char *last = p1;
#endif
    // last may still be NULL (if source don't contain '\' or '/')
	if (last == NULL)
		dest[0] = '\0';
    else
        last[1] = '\0'; // setting last[1]=0 means truncate dest to the last '\' or '/'
    logMsg(c_logger_Severity_DEBUG, "c_getDirName: dest is: %s", dest);

    return MP_code_Success;
}

int32_t c_getBaseName(char *dest, const char *source) {
	if (dest == NULL || source == NULL)
		return MP_code_Generic;

    const char dirseps[] = DIRSEPS_CHAR;

    logMsg(c_logger_Severity_DEBUG, "c_getBaseName: source is: %s", source);
    char *p1 = strrchr(source, dirseps[0]);
#ifdef _WIN32
    // in Windows paths may also have '/'!
    char *p2 = strrchr(source, dirseps[1]);
    char *last = max(p1, p2);
#else
    char *last = p1;
#endif
    // last may still be NULL (if source don't contain '\' or '/')
    if (last == NULL)
        dest[0] = '\0';
    else
        strcpy(dest, last + 1);
    logMsg(c_logger_Severity_DEBUG, "c_getBaseName: dest is: %s", dest);

    return MP_code_Success;
}

static bool _isProcValid(c_Process_t *proc) {
	return (proc != NULL);
}

int32_t c_getCurrentEnvironment(c_environment_t *dest) {
	int32_t ret;
	if (dest == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): invalid null argument", __func__);
		return c_Utility_ErrorCode_GenericError;
	}
#ifdef _WIN32
	LPCH env = GetEnvironmentStringsA();
	size_t sz, totsz = 0;
	while ((sz = strlen(env + totsz)) != 0) {
		char *var = env + totsz;
		char *div = strchr(var, '=');
		char *key, *value;
		totsz += sz + 1;
		if (div == NULL) {
			logMsg(c_logger_Severity_ERROR, "%s(): Variable '%s' misses equal sign.", __func__, var);
			return c_Utility_ErrorCode_GenericError;
		}
		if (div == var)
			continue;  // for some unknown reason, windows may give lines starting with '='. Just skip them...
		*div = '\0';
		key = var;
		value = c_strdup(div + 1);
		if (value == NULL) {
			logMsg(c_logger_Severity_ERROR, "%s(): c_strdup failed", __func__);
			return c_Utility_ErrorCode_GenericError;
		}
		if ((ret = c_map_set(dest, key, value)) != c_Utility_ErrorCode_Success) {
			logMsg(c_logger_Severity_ERROR, "%s(): Error %" PRIi32 " setting keypair [%s] = %s", __func__, ret, key, value);
			*div = '='; // put back equal sign
			c_free(value);
			return ret;
		}
		*div = '='; // put back equal sign
	}
	FreeEnvironmentStringsA(env);
	ret = c_Utility_ErrorCode_Success;
#else
	if (environ != NULL) {
		int32_t i = 0;
		char *s;
		while ((s = environ[i++]) != NULL) {
			char *div = strchr(s, '=');
			char *key, *value;
			if (div == NULL) {
				logMsg(c_logger_Severity_ERROR, "%s(): Variable '%s' misses equal sign.", __func__, s);
				return c_Utility_ErrorCode_GenericError;
			}
			*div = '\0';
			key = s;
			value = c_strdup(div + 1);
			if (value == NULL) {
				logMsg(c_logger_Severity_ERROR, "%s(): c_strdup failed", __func__);
				return c_Utility_ErrorCode_GenericError;
			}
			if ((ret = c_map_set(dest, key, value)) != c_Utility_ErrorCode_Success) {
				logMsg(c_logger_Severity_ERROR, "%s(): Error %" PRIi32 " setting keypair [%s] = %s", __func__, ret, key, value);
				*div = '='; // put back equal sign
				c_free(value);
				return ret;
			}
			*div = '='; // put back equal sign
		}
	}
	ret = c_Utility_ErrorCode_Success;
#endif
	return ret;
}

c_use_decl_annotations c_Process_t* c_newProc(void) {
	const uint32_t initial_nbuckets = 128;
	int32_t ret = c_Utility_ErrorCode_Success;

	c_Process_t* proc = c_calloc(1, sizeof(*proc));

	if (proc == NULL) {
		ret = c_Utility_ErrorCode_GenericError;
		goto QuitFunction;
	}

	proc->execName = NULL;
	proc->argv[0] = NULL;
	proc->argc = 0;

	if ((ret = c_map_init(&proc->environment, initial_nbuckets)) != c_Utility_ErrorCode_Success) {
		logMsg(c_logger_Severity_ERROR, "Can't initialize environment map (err=%"PRIi32")", ret);
		goto QuitFunction;
	}

	if ((ret = c_getCurrentEnvironment(&proc->environment)) != c_Utility_ErrorCode_Success) {
		logMsg(c_logger_Severity_ERROR, "Can't get current environment (err=%"PRIi32")", ret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != c_Utility_ErrorCode_Success) {
		c_free(proc);
		proc = NULL;
	}
	return proc;
}

int32_t c_setProcExec(const char *exec, c_Process_t *proc) {
    if (!_isProcValid(proc))
        return -1;
    proc->execName = c_strdup(exec);
    if (proc->execName == NULL)
        return -2;

    if (proc->argv[0] == NULL)
        c_addProcArg(proc->execName, proc);
    else {
        c_free(proc->argv[0]);
		proc->argv[0] = c_strdup(proc->execName);
        if (proc->argv[0] == NULL)
            return -3;
    }
    return 0;
}

int32_t c_addProcArg(const char *arg, c_Process_t *proc) {
	if (!_isProcValid(proc))
		return -1;
	if (proc->argc >= MAX_PROC_NARGS)
		return -2;
	proc->argv[proc->argc] = c_strdup(arg);
	if (proc->argv[proc->argc] == NULL)
		return -3;

	proc->argc++;
	if (proc->argc < MAX_PROC_NARGS)
		proc->argv[proc->argc] = NULL;
	return 0;
}

int32_t c_addProcEnv(const char *varname, const char *varvalue, c_Process_t *proc) {
	char *val;
	int32_t ret;

	if (!_isProcValid(proc))
		return -1;

	// value must be copyed. key is strdup-ed inside set function
	val = c_strdup(varvalue);
	if ((ret = c_map_set(&proc->environment, varname, val)) != c_Utility_ErrorCode_Success) {
		logMsg(c_logger_Severity_ERROR, "Error %" PRIi32 " adding keyvalue pair [%s]:[%s] to environment map.", ret, varname, varvalue);
		return -2;
	}

	return 0;
}

static void _freePointersArray(char** arr, size_t size) {
	if (arr == NULL)
		return;
	for (size_t i = 0; i < size; i++)
		c_free(arr[i]);
	c_free(arr);
}

static int32_t _mapToStringArray(c_environment_t *map, char ***dest, size_t *count) {
	int32_t ret = 0;
	c_map_iter_t iter = c_map_iter();
	char **result = NULL;
	size_t totcnt = 0;
	size_t cnt = 0;

	*dest = NULL;
	*count = 0;

	if (map == NULL) {
		logMsg(c_logger_Severity_ERROR, "Error getting map size");
		ret = -1;
		goto QuitFunction;
	}

	totcnt = c_map_size(map);

	// allocate '*count' pointers to 'char'
	result = c_malloc(totcnt * sizeof(*result));
	if (result == NULL) {
		logMsg(c_logger_Severity_ERROR, "c_malloc failed");
		ret = -2;
		goto QuitFunction;
	}

	// initialize result array to avoid unlikely UB in the next loop
	for (size_t i = 0; i < totcnt; ++i)
		result[i] = NULL;

	for (const char* key = c_map_next(map, &iter); key != NULL; key = c_map_next(map, &iter)) {

		char **v = c_map_get(map, key);

		// kept for consistency: v cannot be NULL being key part of the iterator
		if (c_unlikely(v == NULL)) {
			logMsg(c_logger_Severity_ERROR, "Cannot get env. variable %s", key);
			ret = -3;
			goto QuitFunction;
		}

		char *value = *v;
		const size_t sz = strlen(key) + strlen(value) + 2;
		result[cnt] = c_malloc(sz * sizeof(*result[cnt]));
		if (result[cnt] == NULL) {
			logMsg(c_logger_Severity_ERROR, "Cannot allocate string of size %zu", sz);
			ret = -4;
			goto QuitFunction;
		}
		sprintf(result[cnt], "%s=%s", key, value);
		cnt++;
	}

	if (cnt != totcnt) {
		logMsg(c_logger_Severity_ERROR, "totcnt and cnt are not equal: %zu != %zu", totcnt, cnt);
		ret = -5;
	}

QuitFunction:
	if (ret != 0) {
		_freePointersArray(result, totcnt);
	}
	else {
		*dest = result;
		*count = totcnt;
	}
	return ret;
}

#ifdef _WIN32
static int _compareStringsInsensitive(const void* a, const void* b) {
	return c_strcasecmp(a, b);
}

static void _uninitHandlers(c_Process_t *proc) {
    proc->procInfo.dwProcessId = 0;
    proc->procInfo.dwThreadId = 0;
    proc->procInfo.hProcess = NULL;
    proc->procInfo.hThread = NULL;
    proc->jobObject = NULL;
}

static int32_t _startProcessWin(c_Process_t *proc) {
	int32_t res = CAENPROC_RetCode_Success;
	BOOL ok = TRUE;
    size_t stringlen=0;
    char *CommandString = NULL;
    JOBOBJECT_EXTENDED_LIMIT_INFORMATION jeli = { 0 };
    STARTUPINFOA                         si   = { 0 };
    const char *fName = NULL;
	char **envarray = NULL;
	char *env = NULL;
	size_t envc = 0;

    if(!_isProcValid(proc))
        return -1;
    if(proc->execName == NULL)
        return -2;
    if(strlen(proc->execName) == 0)
        return -2;

    _uninitHandlers(proc);

	stringlen = strlen(proc->execName) + 1;
	for(int32_t i = 1; i < proc->argc; i++) {
        if(proc->argv[i] != NULL)
            stringlen += strlen(proc->argv[i]) + 1;
        else
            break;
    }
	CommandString = c_malloc(stringlen + 5);
    if (CommandString == NULL)
        return -3;
    sprintf(CommandString, "\"%s\"", proc->execName);
    for(int32_t i = 1; i < proc->argc; i++) {
        if(proc->argv[i] != NULL) {
            strcat(CommandString, " ");
            strcat(CommandString, proc->argv[i]);
        }
        else
            break;
    }

    proc->jobObject = CreateJobObject(NULL, NULL);
    if(proc->jobObject == NULL) {
        proc->lastErr = GetLastError();
        fName = "CreateJobObject";
        res = CAENPROC_RetCode_ProcessFail;
        goto QuitFunction;
    }
    jeli.BasicLimitInformation.LimitFlags = JOB_OBJECT_LIMIT_KILL_ON_JOB_CLOSE | JOB_OBJECT_LIMIT_BREAKAWAY_OK;
    if(SetInformationJobObject(proc->jobObject, JobObjectExtendedLimitInformation, &jeli, sizeof(jeli)) == FALSE) {
        proc->lastErr = GetLastError();
        fName = "SetInformationJobObject";
        res = CAENPROC_RetCode_ProcessFail;
        goto QuitFunction;
    }
    //proc->startInfo.cb = sizeof(proc->startInfo);
    si.cb = sizeof(si);

	// create env:
	// Create an array of strings, sort it with qsort and put it
	// in the format required by windows.
	if (_mapToStringArray(&proc->environment, &envarray, &envc) != 0) {
		fName = "mapToStringArray";
		res = CAENPROC_RetCode_ProcessFail;
		goto QuitFunction;
	}

	if (envc > 0) {
		size_t envsz = 1; // NULL-TERMINATING byte
		qsort(envarray, envc, sizeof(*envarray), _compareStringsInsensitive);
		for (size_t e = 0; e < envc; e++) {
			const char *var = envarray[e];
			size_t addsz = strlen(var) + 1;
			env = c_realloc(env, envsz + addsz);
			if (env == NULL) {
				fName = "<AllocateEnv>";
				proc->lastErr = 1;
				res = CAENPROC_RetCode_Generic;
				goto QuitFunction;
			}
			strcpy(env + envsz - 1, var); // remove NULL-TERM byte from count
			envsz += addsz;
		}
		env[envsz - 1] = '\0';
	}

    ok = CreateProcessA(NULL,
                        (LPSTR)CommandString,
                        NULL,              // process security attributes
                        NULL,              // primary thread security attributes
                        TRUE,              // handles are inherited
                        CREATE_SUSPENDED | CREATE_BREAKAWAY_FROM_JOB | CREATE_NO_WINDOW,  // creation flags
                        env,               // use parent's environment
                        NULL,              // use parent's current directory
                        &si,               // STARTUPINFO pointer 
                        &proc->procInfo);  // receives PROCESS_INFORMATION
    if (!ok) {
        proc->lastErr = GetLastError();
        fName = "CreateProcess";
        res = CAENPROC_RetCode_ProcessFail;
        goto QuitFunction;
    }

    if(AssignProcessToJobObject(proc->jobObject, proc->procInfo.hProcess) == FALSE) {
        proc->lastErr = GetLastError();
        fName = "AssignProcessToJobObject";
        res = CAENPROC_RetCode_ProcessFail;
        goto QuitFunction;
    }

    ResumeThread(proc->procInfo.hThread);

QuitFunction:
    if(res != CAENPROC_RetCode_Success) {
        logMsg(c_logger_Severity_DEBUG, "%s(): Function %s exited with code %"PRIu32, __func__, fName, (uint32_t)proc->lastErr);
        c_endProcess(proc);
    }
	_freePointersArray(envarray, envc);
    return res;
}

static int32_t _endProcessWin(c_Process_t *proc) {
	UINT ret = 0;
    if (!_isProcValid(proc))
        return -1;

    if (proc->procInfo.hProcess != NULL) {
        WaitForSingleObject(proc->procInfo.hProcess, INFINITE);
        TerminateProcess(proc->procInfo.hProcess, ret);
        CloseHandle(proc->procInfo.hProcess);
    }
    if (proc->procInfo.hThread != NULL)
        CloseHandle(proc->procInfo.hThread);
    if (proc->jobObject != NULL) {
        TerminateJobObject(proc->jobObject, ret);
        CloseHandle(proc->jobObject);
    }

    _uninitHandlers(proc);

    return 0;
}

static int32_t _getNCPUWin(void) {
    SYSTEM_INFO info;
    GetSystemInfo(&info);
    return info.dwNumberOfProcessors;
}

#else

// WARNING, for some reason code definition of _GNU_SOURCE doesn't work, one have to give it to gcc in the Makefile
//#define _GNU_SOURCE 1
static int32_t _startProcessLin(c_Process_t *proc) {
    int32_t res;
	char **tempenv = NULL;
	char **envarray = NULL;
	size_t envc = 0;

	// create env:
	// Create an array of strings, sort it with qsort and put it
	// in the format required by windows.
	if ((res = _mapToStringArray(&proc->environment, &tempenv, &envc)) != 0) {
		logMsg(c_logger_Severity_ERROR, "_mapToStringArray() failed with error %" PRIi32, res);
		res = CAENPROC_RetCode_ProcessFail;
		goto QuitFunction;
	}

	// envarray must be NULL terminated.
	envarray = c_malloc((envc + 1) * sizeof(*envarray));
	if (envarray == NULL) {
		logMsg(c_logger_Severity_ERROR, "Can't alloc array for process environment");
		res = CAENPROC_RetCode_ProcessFail;
		goto QuitFunction;
	}
	c_memcpy(envarray, tempenv, envc * sizeof(*envarray));
	envarray[envc] = NULL;

	// NOTE: uncomment the following to redirect stdout and sterr of the process to
	// file "outlog". Also you need to pass '&child_fd_actions' as third arg. of
	// posix_spawn(...) to let it work.
//	char *outlog = "/tmp/foo-log";
//	posix_spawn_file_actions_t child_fd_actions;
//	if ((res = posix_spawn_file_actions_init (&child_fd_actions)) != 0) {
//		res = CAENPROC_RetCode_ProcessFail;
//		goto QuitFunction;
//	}
//	if ((res = posix_spawn_file_actions_addopen (&child_fd_actions, 1, outlog,
//			O_WRONLY | O_CREAT | O_TRUNC, 0644)) != 0) {
//		res = CAENPROC_RetCode_ProcessFail;
//		goto QuitFunction;
//	}
//	if ((res = posix_spawn_file_actions_adddup2 (&child_fd_actions, 1, 2)) != 0) {
//		res = CAENPROC_RetCode_ProcessFail;
//		goto QuitFunction;
//	}

	if ((res = posix_spawn(&proc->pid, proc->execName, NULL, NULL, proc->argv, envarray)) != 0) {
		logMsg(c_logger_Severity_ERROR, "posix_spawn() failed with error %" PRIi32, res);
		res = CAENPROC_RetCode_ProcessFail;
		goto QuitFunction;
	}

QuitFunction:
	c_free(envarray);
	_freePointersArray(tempenv, envc);
    return res;
}

static int32_t _endProcessLin(c_Process_t *proc) {
    int32_t status;

    if(!_isProcValid(proc))
        return -1;

    waitpid(proc->pid, &status, 0);
    return 0;
}

static int32_t _getNCPULin(void) {
    int32_t ret;
#ifdef _SC_NPROCESSORS_ONLN
    ret = sysconf(_SC_NPROCESSORS_ONLN);
#else
    ret = -1;
#endif
    return ret;
}

#endif

c_use_decl_annotations DIR* c_opendir(const char* name) {
#ifdef _WIN32
	DIR* dir = NULL;
	if (name && name[0]) {
		size_t base_length = strlen(name);
		// search pattern must end with suitable wildcard
		const char* all = strchr("/\\", name[base_length - 1]) ? "*" : "/*";
		if ((dir = c_malloc(sizeof(*dir))) != NULL && (dir->name = c_malloc(base_length + strlen(all) + 1)) != NULL) {
			strcat(strcpy(dir->name, name), all);
			dir->handle = (handle_type)_findfirst(dir->name, &dir->info);
			if (dir->handle != -1) {
				dir->result.d_name = 0;
			} else { // rollback
				c_free(dir->name);
				c_free(dir);
				dir = NULL;
			}
		} else { // rollback
			if (dir != NULL)
				c_free(dir->name);
			c_free(dir);
			dir = NULL;
			errno = ENOMEM;
		}
	} else {
		errno = EINVAL;
	}
	return dir;
#else
	return opendir(name);
#endif
}

int c_closedir(DIR* dir) {
#ifdef _WIN32
	int result = -1;
	if (dir) {
		if (dir->handle != -1) {
			result = _findclose(dir->handle);
		}
		c_free(dir->name);
		c_free(dir);
	}
	if (result == -1) { // map all errors to EBADF
		errno = EBADF;
	}
	return result;
#else
	return closedir(dir);
#endif
}

c_use_decl_annotations struct dirent* c_readdir(DIR* dir) {
#ifdef _WIN32
	struct dirent* result = NULL;
	if (dir && dir->handle != -1) {
		if (!dir->result.d_name || _findnext(dir->handle, &dir->info) != -1) {
			result = &dir->result;
			result->d_name = dir->info.name;
		}
	} else {
		errno = EBADF;
	}
	return result;
#else
	return readdir(dir);
#endif
}

void c_rewinddir(DIR* dir) {
#ifdef _WIN32
	if (dir && dir->handle != -1) {
		_findclose(dir->handle);
		dir->handle = (handle_type)_findfirst(dir->name, &dir->info);
		dir->result.d_name = 0;
	}
	else {
		errno = EBADF;
	}
#else
	rewinddir(dir);
#endif
}

int32_t c_startProcess(c_Process_t *proc) {
#ifdef _WIN32
    return _startProcessWin(proc);
#else
    return _startProcessLin(proc);
#endif
}

int32_t c_endProcess(c_Process_t *proc) {
#ifdef _WIN32
	return _endProcessWin(proc);
#else
	return _endProcessLin(proc);
#endif
}

void c_freeProc(c_Process_t *proc) {
	c_map_iter_t iter = c_map_iter();
    if(proc == NULL)
        return;

    for(int32_t i = 0; i < proc->argc; i++) {
        if(proc->argv[i] == NULL)
			break;
		c_free(proc->argv[i]);
    }
	// Free environment
	for (const char* key = c_map_next(&proc->environment, &iter); key != NULL; key = c_map_next(&proc->environment, &iter)) {
		char **v = c_map_get(&proc->environment, key);
		if (v != NULL)
			c_free(*v);
	}
	c_map_deinit(&proc->environment);

    c_free(proc->execName);
    proc->argc=0;

#ifdef _WIN32
    _uninitHandlers(proc);
#endif
}

c_use_decl_annotations int32_t c_getNCPU(void) {
#ifdef _WIN32
	return _getNCPUWin();
#else
	return _getNCPULin();
#endif
}

int32_t c_pause(void) {
    puts("Press any key to continue...");
    if (c_getch() == 0) {} // to avoid warn_unused_result
    return MP_code_Success;
}

int32_t c_mkdir(const char* path) {
#ifdef _WIN32
	const int r = _mkdir(path);
#else
	const int r = mkdir(path, ACCESSPERMS);
#endif
	return (r == 0 || errno == EEXIST) ? 0 : -1;
}

int32_t c_mkdirp(const char *dir) {
    char *tmp = c_strdup(dir);
    size_t len;
    char dirseps[] = DIRSEPS_STR;
    int32_t ret = 0;
    
    len = strlen(tmp);

    // remove trailing separators (if present)
    while (strchr(dirseps, tmp[len - 1]) != NULL) {
        tmp[len - 1] = '\0';
        // check if 'len' is already 0, if yes the path
        // is invalid.
        if (len == 0) {
            c_free(tmp);
            return -1;
        }
        len--;
    }

    // parse the string creating every directory we find
    for (char *p = tmp + 1; *p != '\0'; ++p) {
        if (strchr(dirseps, *p) != NULL) {
            char csave = *p;
            *p = '\0';
            c_mkdir(tmp);
            *p = csave;

            // go ahead until we exhaust consecutive dirseps
			for (; (p[1] != '\0') && strchr(dirseps, p[1]) != NULL; ++p);
        }
    }
    ret = c_mkdir(tmp);
    c_free(tmp);
    return ret;
}

int32_t c_dlload(const char* libName, c_libhandle_t *libHandle) {
    int32_t ret = MP_code_Success;
#ifdef _WIN32
	*libHandle = LoadLibraryA(libName);
#else
    size_t linuxLibNameSize = 3 + strlen(libName) + 3 + 1;
    char *linuxLibName = c_malloc(linuxLibNameSize);
    if (linuxLibName == NULL)
    	return MP_code_Generic;
    snprintf(linuxLibName, linuxLibNameSize, "lib%s.so", libName);
    *libHandle = dlopen(linuxLibName, RTLD_LAZY);
    c_free(linuxLibName);
    char *error = dlerror();
    if (error != NULL) {
        logMsg(c_logger_Severity_ERROR, "%s", error);
        *libHandle = NULL;
    }
#endif
    if (*libHandle == NULL) {
        ret = MP_code_LibLoad;
        _decodeError(__func__, ret);
    }
    return ret;
}

int32_t c_dfload(c_libhandle_t hLib, const char* fcnName, c_fcnhandle_t *funcPtr) {
    int32_t ret = MP_code_Success;
#ifdef _WIN32
    *funcPtr = GetProcAddress(hLib, fcnName);
	if (*funcPtr == NULL) {
		logMsg(c_logger_Severity_ERROR, "GetProcAddress failed. GetLastError = %"PRIu32".", (uint32_t)GetLastError());
	}
#else
    dlerror(); // clear any current error
    *funcPtr = dlsym(hLib, fcnName);
    char* error = dlerror();
    if (error != NULL) {
        logMsg(c_logger_Severity_ERROR, "%s", error);
        *funcPtr = NULL;
    }
#endif
    if (*funcPtr == NULL) {
        ret = MP_code_LibLoad;
        _decodeError(__func__, ret);
    }
    return ret;
}

int32_t c_dlclose(c_libhandle_t hLib) {
    int32_t ret = MP_code_Success;
#ifdef _WIN32
    if (!FreeLibrary(hLib))
        ret = MP_code_LibClose;
#else
    dlclose(hLib);
    char *error = dlerror();
    if (error != NULL) {
        logMsg(c_logger_Severity_ERROR, "%s", error);
        ret = MP_code_LibClose;
    }
#endif
    if (ret != MP_code_Success)
        _decodeError(__func__, ret);
    return ret;
}

static void _tmToSystemtime(const struct tm* src, c_systemtime_t *dest, uint16_t millis) {
	dest->wYear = (unsigned short)(src->tm_year) + 1900;
	dest->wMonth = (unsigned short)(src->tm_mon) + 1;
	dest->wDay = (unsigned short)(src->tm_mday);
	dest->wHour = (unsigned short)(src->tm_hour);
	dest->wMinute = (unsigned short)(src->tm_min);
	dest->wSecond = (unsigned short)(src->tm_sec);
	dest->wDayOfWeek = (unsigned short)(src->tm_wday);
	// assume milliseconds already range from 0:999
	dest->wMilliseconds = millis;

	// tm accepts seconds from 0->60, SYSTEMTIME from 0->59
	if (src->tm_sec == 60) {
		dest->wMinute++;
		dest->wSecond = 0;
	}
}

static void _systemtimeToTm(const c_systemtime_t* src, struct tm* dest) {
	dest->tm_year = src->wYear - 1900;
	dest->tm_mon = src->wMonth - 1;
	dest->tm_mday = src->wDay;
	dest->tm_hour = src->wHour;
	dest->tm_min = src->wMinute;
	dest->tm_sec = src->wSecond;
	dest->tm_wday = src->wDayOfWeek;
	dest->tm_isdst = 0; //this field must be initialized in order to avoid a random usage of DST as an offset of 3600 seconds
						//see https://en.cppreference.com/w/c/chrono/mktime
						//"A negative value of arg->tm_isdst causes mktime to attempt to determine if Daylight Saving Time was in effect in the specified time"
}

void c_GetLocalTime(c_systemtime_t* now) {
#ifdef _WIN32
	GetLocalTime(now);
#else
	struct tm *timeinfo;
	struct timespec spec;
	uint16_t millis = 0;
	clock_gettime(CLOCK_REALTIME, &spec);
	timeinfo = localtime(&spec.tv_sec);
	millis = (uint16_t)(spec.tv_nsec / 1.e6);
	_tmToSystemtime(timeinfo, now, millis);
#endif
}

void c_GetSystemTime(c_systemtime_t *t) {
#ifdef _WIN32
	GetSystemTime(t);
#else
	struct tm *timeinfo;
	struct timespec spec;
	uint16_t millis = 0;
	clock_gettime(CLOCK_REALTIME, &spec);
	timeinfo = gmtime(&spec.tv_sec);
	millis = (uint16_t)(spec.tv_nsec / 1.e6);
	_tmToSystemtime(timeinfo, t, millis);
#endif
}

c_use_decl_annotations double c_DiffSystemTime_sec(const c_systemtime_t *s1, const c_systemtime_t *s2) {
	double res;

#ifdef _WIN32
	/*
	 * According to SYSTEMTIME documentation, we have to:
	 * - Convert the SYSTEMTIME structure to a FILETIME structure.
	 * - Copy the resulting FILETIME structure to a ULARGE_INTEGER structure.
	 * - Use normal 64-bit arithmetic on the ULARGE_INTEGER value.
	 *
	 * According to FILETIME documentation, it contains a 64-bit value
	 * representing the number of 100-nanosecond intervals since
	 * January 1, 1601 (UTC). So, it must be divide by 1.e7 to obtain seconds.
	 *
	 * See:
	 * - https://docs.microsoft.com/it-it/windows/win32/api/minwinbase/ns-minwinbase-systemtime
	 * - https://docs.microsoft.com/en-us/windows/win32/api/minwinbase/ns-minwinbase-filetime
	 */

	FILETIME f1, f2;
	ULARGE_INTEGER u1, u2;

	if (!(SystemTimeToFileTime(s1, &f1) && SystemTimeToFileTime(s2, &f2))) {
		logMsg(c_logger_Severity_ERROR, "SystemTimeToFileTime failed. GetLastError = %"PRIu32".", (uint32_t)GetLastError());
		return nan("");
	}

	u1.HighPart = f1.dwHighDateTime;
	u1.LowPart = f1.dwLowDateTime;
	u2.HighPart = f2.dwHighDateTime;
	u2.LowPart = f2.dwLowDateTime;

	/*

	 */
	res = (u2.QuadPart - u1.QuadPart) / 1.e7;
#else
	struct tm tm1, tm2;
	time_t t1, t2;
	/*
	 * mdiff can also be negative, but this is ok... see comment
	 * below on _systemtimeToTm() function.
	 */
	int16_t mdiff = (int16_t)s2->wMilliseconds - (int16_t)s1->wMilliseconds;
	_systemtimeToTm(s1, &tm1);
	_systemtimeToTm(s2, &tm2);
	t1 = mktime(&tm1);
	t2 = mktime(&tm2);
	res = difftime(t2, t1);
	/*
	 * Also consider millis, considering that _systemtimeToTm()
	 * function truncates to seconds.
	 */
	res += (mdiff / 1000.0);
#endif

	return res;
}

#ifndef _WIN32
// To be replaced with C11 _Generic, when available.
// 32 bit
#if UINT32_MAX == UINT_MAX
#define _gcc_builtin(function, value)		__builtin_##function(value)
#elif UINT32_MAX == ULONG_MAX
#define _gcc_builtin(function, value)		__builtin_##function##l(value)
#elif UINT32_MAX == ULLONG_MAX
#define _gcc_builtin(function, value)		__builtin_##function##ll(value)
#else
#error cannot implement GCC 32-bit builtin functions
#endif
// 64 bit
#if UINT64_MAX == UINT_MAX
#define _gcc_builtin_64(function, value)	__builtin_##function(value)
#elif UINT64_MAX == ULONG_MAX
#define _gcc_builtin_64(function, value)	__builtin_##function##l(value)
#elif UINT64_MAX == ULLONG_MAX
#define _gcc_builtin_64(function, value)	__builtin_##function##ll(value)
#else
#error cannot implement GCC 64-bit builtin functions
#endif
#endif

#ifdef _WIN32
#ifdef _is_machine_x86_64
/*
 * The assembly instructin popcnt is available only since SSE4.
 *
 * On MSVC __popcnt is not like GCC __builtin_popcount: the former generates
 * the assembly instruction popcnt, while the latter is smarter, and generates
 * a code independent on the target machine: it generates popcnt only if SSE4
 * is available, and uses a specific fallback library function in the other cases.
 *
 * This implementation is copied from MSVC std::popcount, available on C++20.
 * __AVX__ defined implies SSE4 support.
 *
 * Fallback implementation copied from https://stackoverflow.com/a/109025/3287591
 * Interestingly, GCC 10 and Clang 10 recognize this pattern and emit popcnt
 * if available on the target architecture.
 */

static bool _definitely_have_popcnt() {
#ifndef __AVX__
	extern int __isa_available;
	return (__isa_available >= __ISA_AVAILABLE_SSE42);
#else
	return true;
#endif
}
#endif
static uint32_t _popcount_fallback(uint32_t value) {
	value -= (value >> 1) & UINT32_C(0x55555555);
	value = (value & UINT32_C(0x33333333)) + ((value >> 2) & UINT32_C(0x33333333));
	value = (value + (value >> 4)) & UINT32_C(0x0F0F0F0F);
	return (value * UINT32_C(0x01010101)) >> 24;
}

static uint32_t _popcount64_fallback(uint64_t value) {
	value -= (value >> 1) & UINT64_C(0x5555555555555555);
	value = (value & UINT64_C(0x3333333333333333)) + ((value >> 2) & UINT64_C(0x3333333333333333));
	value = (value + (value >> 4)) & UINT64_C(0x0F0F0F0F0F0F0F0F);
	return (value * UINT64_C(0x0101010101010101)) >> 56;
}
#endif

// See https://stackoverflow.com/questions/355967/how-to-use-msvc-intrinsics-to-get-the-equivalent-of-this-gcc-code
c_use_decl_annotations uint32_t c_ctz(uint32_t value) {
#ifdef _WIN32
	DWORD res;
	// value == 0 -> BitScanForward return false, res undefined
	return BitScanForward(&res, (DWORD)value) ? (uint32_t)(res) : UINT32_C(32);
#else
	// value == 0 -> __builtin_ctz undefined
	return (value != 0) ? (uint32_t)_gcc_builtin(ctz, value) : UINT32_C(32);
#endif
}

// See https://stackoverflow.com/questions/355967/how-to-use-msvc-intrinsics-to-get-the-equivalent-of-this-gcc-code
c_use_decl_annotations uint32_t c_clz(uint32_t value) {
#ifdef _WIN32
	DWORD res;
	// value == 0 -> BitScanReverse return false, res undefined
	return BitScanReverse(&res, (DWORD)value) ? (uint32_t)(UINT32_C(31) - res) : UINT32_C(32);
#else
	// value == 0 -> __builtin_clz undefined
	return (value != 0) ? (uint32_t)_gcc_builtin(clz, value) : UINT32_C(32);
#endif
}

c_use_decl_annotations uint32_t c_popcount(uint32_t value) {
#ifdef _WIN32
#ifdef _is_machine_x86_64
	if (!_definitely_have_popcnt())
		return _popcount_fallback(value);
	return __popcnt(value);
#else
	return _popcount_fallback(value);
#endif
#else
	return (uint32_t)_gcc_builtin(popcount, value);
#endif
}

c_use_decl_annotations uint32_t c_ctz64(uint64_t value) {
#ifdef _WIN32
	DWORD res;
	// value == 0 -> BitScanForward64 return false, res undefined
	return BitScanForward64(&res, (DWORD64)value) ? (uint32_t)(res) : UINT32_C(64);
#else
	// value == 0 -> __builtin_ctz undefined
	return (value != 0) ? (uint32_t)_gcc_builtin_64(ctz, value) : UINT32_C(64);
#endif
}

c_use_decl_annotations uint32_t c_clz64(uint64_t value) {
#ifdef _WIN32
	DWORD res;
	// value == 0 -> BitScanReverse64 return false, res undefined
	return BitScanReverse64(&res, (DWORD64)value) ? (uint32_t)(UINT32_C(63) - res) : UINT32_C(64);
#else
	// value == 0 -> __builtin_clz undefined
	return (value != 0) ? (uint32_t)_gcc_builtin_64(clz, value) : UINT32_C(64);
#endif
}

c_use_decl_annotations uint32_t c_popcount64(uint64_t value) {
#ifdef _WIN32
#ifdef _is_machine_x86_64
	if (!_definitely_have_popcnt())
		return _popcount64_fallback(value);
#ifdef _M_IX86
	// __popcnt64 available only on x64
	return __popcnt((uint32_t)(value >> 32)) + __popcnt((uint32_t)(value));

#else
	return (uint32_t)__popcnt64((DWORD64)value);
#endif
#else
	return _popcount64_fallback(value);
#endif
#else
	return (uint32_t)_gcc_builtin_64(popcount, value);
#endif
}

// Used to convert result of frexp to integer.
// In IEEE 754 implementation this can be equal to 2^DBL_MANT_DIG (2^53), to avoid precision loss.
// Anyway, we use a custom number to be sure the same conversion is used in all the architectures.
// The exponent can be any number between DBL_MANT_DIG and 63.
static const double twop60 = 1.152921504606846976e+18; // 2^(+60) "0x43b00000, 0x00000000"
static const double twom60 = 8.673617379884035472e-19; // 2^(-60) "0x3c300000, 0x00000000"

c_use_decl_annotations int64_t c_frexp(double value, int16_t *exponent, int8_t *classification) {
	if (c_unlikely(exponent == NULL || classification == NULL))
		return 0;

	// Get normalized fraction and exponent
	int exp;
	double norm_fraction;
	int64_t mantissa;

	// Get the classification and handle special cases
	switch (fpclassify(value)) {
	case FP_INFINITE:
		*classification = CAEN_FP_INFINITE;
		exp = 0;
		// Mantissa is used to carry the sign.
		mantissa = signbit(value) ? INT64_C(-1) : INT64_C(1);
		break;
	case FP_NAN:
		*classification = CAEN_FP_NAN;
		exp = 0;
		// Mantissa is used to carry the sign.
		// NaN payload is discarded.
		// signbit() is the only function that allow to get NaN sign
		mantissa = signbit(value) ? INT64_C(-1) : INT64_C(1);
		break;
	case FP_ZERO:
		*classification = CAEN_FP_ZERO;
		exp = 0;
		// Mantissa is used to carry the sign.
		mantissa = signbit(value) ? INT64_C(-1) : INT64_C(1);
		break;
	case FP_SUBNORMAL:
		*classification = CAEN_FP_SUBNORMAL;
		// Get integer mantissa
		norm_fraction = frexp(value, &exp);
		mantissa = (int64_t)(norm_fraction * twop60);
		break;
	case FP_NORMAL:
	default:
		*classification = CAEN_FP_NORMAL;
		// Get integer mantissa
		norm_fraction = frexp(value, &exp);
		mantissa = (int64_t)(norm_fraction * twop60);
	}

	*exponent = (int16_t)exp;

	return mantissa;
}

c_use_decl_annotations double c_ldexp(int64_t mantissa, int16_t exponent, int8_t classification) {

	// Get normalized fraction from integer mantissa
	const double norm_fraction = (double)mantissa * twom60;
	double value;

	// Get the classification and handle special cases
	switch (classification) {
	case CAEN_FP_INFINITE:
		// Mantissa is used to carry the sign.
		// NOTE: norm_fraction should have the same sign as mantissa, so we don't have to cast it
		value = copysign(HUGE_VAL, norm_fraction);
		break;
	case CAEN_FP_NAN:
		// Mantissa is used to carry the sign.
		// NaN payload is discarded.
		// copysign() is the only function that allow to set NaN sign
		// NOTE: norm_fraction should have the same sign as mantissa, so we don't have to cast it
		value = copysign(NAN, norm_fraction);
		break;
	case CAEN_FP_ZERO:
		// Mantissa is used to carry the sign.
		value = copysign(0., norm_fraction);
		break;
	case CAEN_FP_SUBNORMAL:
	case CAEN_FP_NORMAL:
	default:
		// Get the value from normalized fraction and exponent
		value = ldexp(norm_fraction, (int)exponent);
	}

	return value;
}


/*
 * Copied from "XGetopt - A Unix-compatible getopt() for MFC and Win32" v1.2, by Hans Dietrich,
 * released in the public domain.
 * https://www.codeproject.com/Articles/1940/XGetopt-A-Unix-compatible-getopt-for-MFC-and-Win
 */
#ifdef _WIN32 // On Linux, defined by unistd.h

int c_mp_opterr = 0; 	// global err. (unused)
int c_mp_optopt = 0; 	// global opt. err.
int c_mp_optind = 0; 	// global argv index
char *c_mp_optarg;		// global argument pointer

c_use_decl_annotations int c_mp_getopt(int argc, char *const *argv, const char *options) {
	static char *next = NULL;
	if (c_mp_optind == 0)
		next = NULL;

	c_mp_optarg = NULL;

	if (next == NULL || *next == '\0') {
		if (c_mp_optind == 0)
			c_mp_optind++;

		if (c_mp_optind >= argc || argv[c_mp_optind][0] != '-' || argv[c_mp_optind][1] == '\0') {
			c_mp_optarg = NULL;
			if (c_mp_optind < argc)
				c_mp_optarg = argv[c_mp_optind];
			return -1;
		}

		if (strcmp(argv[c_mp_optind], "--") == 0) {
			c_mp_optind++;
			c_mp_optarg = NULL;
			if (c_mp_optind < argc)
				c_mp_optarg = argv[c_mp_optind];
			return -1;
		}

		next = argv[c_mp_optind];
		next++;		// skip past -
		c_mp_optind++;
	}

	char c = *next++;
	char *cp = strchr(options, c);

	if (cp == NULL || c == ':') {
		c_mp_optopt = c;
		return '?';
	}

	cp++;
	if (*cp == ':') {
		if (*next != '\0') {
			c_mp_optarg = next;
			next = NULL;
		}
		else if (c_mp_optind < argc) {
			c_mp_optarg = argv[c_mp_optind];
			c_mp_optind++;
		}
		else {
			c_mp_optopt = c;
			return '?';
		}
	}

	return c;
}

#endif

c_use_decl_annotations bool c_checkFileExists(const char *fname) {
	if (c_unlikely(fname == NULL))
		return FALSE;
	FILE * const f = fopen(fname, "r");
	if (f == NULL)
		return FALSE;
	return (fclose(f) == 0);
}

c_use_decl_annotations uint64_t c_getCurrentTimeRepresentation(void) {
	time_t rawtime;
	struct tm * timeinfo;
	char datetimestr[sizeof("YYYYMMDDHHMMSS")];
	char p1[sizeof("YYYYMMDD")];
	char p2[sizeof("HHMMSS")];
	int tmp;
	uint64_t res;
	size_t nc;

	time(&rawtime);
	timeinfo = localtime(&rawtime);
	strftime(datetimestr, sizeof(datetimestr), "%Y%m%d%H%M%S", timeinfo);
	nc = sizeof(p1) - 1;
	strncpy(p1, datetimestr, nc);
	p1[nc] = '\0';
	nc = sizeof(p2) - 1;
	strncpy(p2, datetimestr + sizeof(p1) - 1, nc);
	p2[nc] = '\0';
	tmp = atoi(p1);
	res = (uint64_t)tmp * 1000000;
	tmp = atoi(p2);
	res += tmp;
	return res;
}

int32_t c_remAllFiles(const char *dir) {
	struct dirent *de;
	DIR *dr = c_opendir(dir);
	char *path = c_malloc(1024); // HACK no check on size
	struct stat st;
	int32_t ret = 0;

	if (dr == NULL || path == NULL) {
		logMsg(c_logger_Severity_WARNING, "Can't alloc filename or open directory %s. Files won't be deleted.", dir);
		ret = -1;
		goto QuitFunction;
	}

	while ((de = c_readdir(dr)) != NULL) {
		const char *fname = de->d_name;
		if (!strcmp(fname, ".") || !strcmp(fname, "..")) // skip self and prev. dir.
			continue;
		sprintf(path, "%s" DIRSEP0 "%s", dir, fname);
		if (stat(path, &st) != 0) {
			logMsg(c_logger_Severity_WARNING, "Error on stat(%s)", path);
			ret = -2;
			goto QuitFunction;
		}
		if (st.st_mode & S_IFDIR) {
			if (c_remAllFiles(path) != 0) {
				ret = -3;
				goto QuitFunction;
			}
			if (c_rmdir(path) != 0) {
				logMsg(c_logger_Severity_WARNING, "Error on rmdir(%s)", path);
				ret = -4;
				goto QuitFunction;
			}
		}
		else {
			if (remove(path) != 0) {
				logMsg(c_logger_Severity_WARNING, "Error on remove(%s)", path);
				ret = -5;
				goto QuitFunction;
			}
		}
	}

QuitFunction:
	if (dr != NULL)
		c_closedir(dr);
	c_free(path);
	return ret;
}


int32_t c_killProcess(const char *procName) {
	int32_t ret = 0;

	// len(basename) is <= len(procName)
	char *fname = c_malloc(strlen(procName) + 1);
	if (fname == NULL)
		return -1;

	c_getBaseName(fname, procName);

#ifdef _WIN32
	CONST HANDLE hSnapShot = CreateToolhelp32Snapshot(TH32CS_SNAPALL, 0);
	PROCESSENTRY32 pEntry;
	pEntry.dwSize = sizeof(pEntry);
	CONST DWORD currPID = GetCurrentProcessId();

	for (BOOL hRes = Process32First(hSnapShot, &pEntry); hRes; hRes = Process32Next(hSnapShot, &pEntry)) {
		// do not kill himself!
		if (strcmp(pEntry.szExeFile, fname) == 0 && pEntry.th32ProcessID != currPID) {
			HANDLE hProcess = OpenProcess(PROCESS_TERMINATE, 0, pEntry.th32ProcessID);
			if (hProcess != NULL) {
				TerminateProcess(hProcess, 9);
				CloseHandle(hProcess);
			}
		}
	}

	CloseHandle(hSnapShot);
	
#else
	char buf[KILL_BUFSIZE];
	FILE *fp;
	char *cmd = NULL;

	const char *format = "ps -A | grep -w '%s' | grep -v 'grep' | awk '{print $1}'";

	// Check size to allocate, and allocate
    int needed = snprintf(NULL, 0, format, fname);
	if (needed < 0) {
		ret = -1;
		goto QuitFunction;
	}

    logMsg(c_logger_Severity_DEBUG, "c_killProcess: Needed %d chars", needed);
	cmd = c_malloc(needed + 1);
	if (cmd == NULL) {
		ret = -1;
		goto QuitFunction;
	}

	sprintf(cmd, format, fname);
    logMsg(c_logger_Severity_DEBUG, "c_killProcess: cmd = '%s'", cmd);

	if ((fp = popen(cmd, "r")) == NULL) {
        logMsg(c_logger_Severity_WARNING, "c_killProcess: Error opening pipe!");
        ret = -1;
		goto QuitFunction;
	}

	while (fgets(buf, KILL_BUFSIZE, fp) != NULL) {
        int64_t pid;
        int64_t this_pid = c_getpid();
        if (sscanf(buf, "%" SCNd64, &pid) == 1) {
            if (pid != this_pid) {
                logMsg(c_logger_Severity_DEBUG, "c_killProcess: Killing PID %" PRIi64, pid);
                int32_t l_ret = kill((pid_t)pid, SIGKILL);
                if (ret == 0 && l_ret != 0)
                    ret = l_ret;
            }
            else {
                logMsg(c_logger_Severity_DEBUG, "c_killProcess: Will not kill PID %" PRIi64 " (this process).", pid);
            }
		}
		else {
            logMsg(c_logger_Severity_WARNING, "c_killProcess: Can't parse PID from string %s", buf);
		}
	}

	if (pclose(fp)) {
        logMsg(c_logger_Severity_WARNING, "c_killProcess: Can't close pipe.");
        ret = -1;
		goto QuitFunction;
	}

QuitFunction:
	c_free(cmd);
#endif
    c_free(fname);
	return ret;
}

#undef _is_machine_x86_64
