/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENRandom.c
*	\brief		Pseudo-random number generator implemented on MT19937.
*	\author		Giovanni Cerretani
*
******************************************************************************/

#include <CAENRandom.h>

#include <math.h>
#include <inttypes.h>

#include <CAENLogger.h>
#include <CAENMultiplatform.h>

INIT_C_LOGGER("CAENRandomLog.txt", "CAENRandom.c");

/*
 * The code is partially inspired from ROOT MathCore Library TRandom3,
 * developed by Peter Malzacher and released under GNU LGPL 2 (or any later).
 */

/*
* Types used by this generator
*/
struct _initializedstuff {
	bool initialized;
	bool seeded;
};

struct _librarystate {
	struct _initializedstuff v32;
	struct _initializedstuff v64;
};

struct _statevector {
	uint32_t *v32;
	uint64_t *v64;
};

struct _integer {
	uint32_t v32;
	uint64_t v64;
};

struct _internalstate {
	struct _statevector vector;
	struct _integer index;
};

/*
* Internal state
*/
static struct _librarystate lState;							//!< Library state.
static struct _internalstate iState;						//!< Internal state.

/*
* Generator constants
*/
//! The number of bits of each word in the state sequence.
static const struct _integer kW = {
	UINT32_C(32),
	UINT64_C(64)
};

//! The shift size used on twists to transform the values.
static const struct _integer kM = {
	UINT32_C(397),
	UINT64_C(156)
};

//! The number of elements in the state sequence (degree of recurrence).
static const struct _integer kN = {
	UINT32_C(624),
	UINT64_C(312)
};

//! The XOR mask applied as the linear function on each twist.
static const struct _integer kA = {
	UINT32_C(0x9908b0df),
	UINT64_C(0xb5026f5aa96619e9)
};

//! The XOR mask used as parameter `b` in the tempering process of the generation algorithm.
static const struct _integer kB = {
	UINT32_C(0x9d2c5680),
	UINT64_C(0x71d67fffeda60000)
};

//! The XOR mask used as parameter `c` in the tempering process of the generation algorithm.
static const struct _integer kC = {
	UINT32_C(0xefc60000),
	UINT64_C(0xfff7eee000000000)
};

//! The XOR mask used as parameter `d` in the tempering process of the generation algorithm.
static const struct _integer kD = {
	UINT32_C(0xffffffff),
	UINT64_C(0x5555555555555555)
};

//! The shift size of parameter `s` used in the tempering process of the generation algorithm.
static const struct _integer kS = {
	UINT32_C(7),
	UINT64_C(17)
};

//! The shift size of parameter `t` used in the tempering process of the generation algorithm.
static const struct _integer kT = {
	UINT32_C(15),
	UINT64_C(37)
};

//! The shift size of parameter `u` used in the tempering process of the generation algorithm.
static const struct _integer kU = {
	UINT32_C(11),
	UINT64_C(29)
};

//! The shift size of parameter `l` used in the tempering process of the generation algorithm.
static const struct _integer kL = {
	UINT32_C(18),
	UINT64_C(43)
};

//! The initialization multiplier used to seed the state sequence when a single value is used as seed.
static const struct _integer kF = {
	UINT32_C(1812433253),
	UINT64_C(6364136223846793005)
};

//! The default seed used on construction or seeding.
static const struct _integer kDefaultSeed = {
	UINT32_C(5489),
	UINT64_C(5489)
};

//! The mask with 31 1's, where 31 is the number of bits that mark the separation point of words on each twist.
static const struct _integer kLowerMask = {
	UINT32_C(0x7fffffff),
	UINT64_C(0x7fffffff)
};

int32_t c_rand32_init(void) {
	if (lState.v32.initialized)
		return c_Utility_ErrorCode_Success;
	
	// Initialize memory for the state vector
	iState.vector.v32 = c_malloc(sizeof(*iState.vector.v32) * (size_t)kN.v32);
	if (iState.vector.v32 == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): state vector allocation failed.", __func__);
		return c_Utility_ErrorCode_GenericError;
	}

	lState.v32.initialized = TRUE;

	return c_Utility_ErrorCode_Success;
}

void c_rand32_deinit(void) {
	if (!lState.v32.initialized)
		return;
	lState.v32.initialized = FALSE;
	lState.v32.seeded = FALSE;
	c_free(iState.vector.v32);
	iState.vector.v32 = NULL;
	iState.index.v32 = 0;
}

int32_t c_rand64_init(void) {
	if (lState.v64.initialized)
		return c_Utility_ErrorCode_Success;

	// Initialize memory for the state vector
	iState.vector.v64 = c_malloc(sizeof(*iState.vector.v64) * (size_t)kN.v64);
	if (iState.vector.v64 == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): state vector allocation failed.", __func__);
		return c_Utility_ErrorCode_GenericError;
	}

	lState.v64.initialized = TRUE;

	return c_Utility_ErrorCode_Success;
}

void c_rand64_deinit(void) {
	if (!lState.v64.initialized)
		return;
	lState.v64.initialized = FALSE;
	lState.v64.seeded = FALSE;
	c_free(iState.vector.v64);
	iState.vector.v64 = NULL;
	iState.index.v64 = 0;
}

void c_rand32_seed(uint32_t seed) {
	// Check if initialized.
	if (!lState.v32.initialized) {
		logMsg(c_logger_Severity_WARNING, "%s(): generator not initialized. Calling c_rand32_init() and continuing...", __func__);
		c_rand32_init();
	}

	uint32_t prev = iState.vector.v32[0] = seed;
	for (uint32_t i = 1; i < kN.v32; ++i)
		prev = iState.vector.v32[i] = (i + kF.v32 * (prev ^ (prev >> (kW.v32 - 2))));

	iState.index.v32 = kN.v32;
	lState.v32.seeded = TRUE;
}

void c_rand64_seed(uint64_t seed) {
	// Check if initialized.
	if (!lState.v64.initialized) {
		logMsg(c_logger_Severity_WARNING, "%s(): generator not initialized. Calling c_rand64_init() and continuing...", __func__);
		c_rand64_init();
	}

	uint64_t prev = iState.vector.v64[0] = seed;
	for (uint64_t i = 1; i < kN.v64; ++i)
		prev = iState.vector.v64[i] = (i + kF.v64 * (prev ^ (prev >> (kW.v64 - 2))));

	iState.index.v64 = kN.v64;
	lState.v64.seeded = TRUE;
}

static void _twist32(void) {
	uint32_t i = 0;

	for (; i < kN.v32 - kM.v32; ++i) {
		uint32_t tmp = (iState.vector.v32[i] & ~kLowerMask.v32) | (iState.vector.v32[i + 1] & kLowerMask.v32);
		iState.vector.v32[i] = iState.vector.v32[i + kM.v32] ^ (tmp >> 1) ^ ((tmp & 1) ? kA.v32 : 0);
	}

	for (; i < kN.v32 - 1; ++i) {
		uint32_t tmp = (iState.vector.v32[i] & ~kLowerMask.v32) | (iState.vector.v32[i + 1] & kLowerMask.v32);
		iState.vector.v32[i] = iState.vector.v32[i + kM.v32 - kN.v32] ^ (tmp >> 1) ^ ((tmp & 1) ? kA.v32 : 0);
	}

	uint32_t tmp = (iState.vector.v32[kN.v32 - 1] & ~kLowerMask.v32) | (iState.vector.v32[0] & kLowerMask.v32);
	iState.vector.v32[kN.v32 - 1] = iState.vector.v32[kM.v32 - 1] ^ (tmp >> 1) ^ ((tmp & 1) ? kA.v32 : 0);

	iState.index.v32 = 0;
}

static void _twist64(void) {
	uint64_t i = 0;

	for (; i < kN.v64 - kM.v64; ++i) {
		uint64_t tmp = (iState.vector.v64[i] & ~kLowerMask.v64) | (iState.vector.v64[i + 1] & kLowerMask.v64);
		iState.vector.v64[i] = iState.vector.v64[i + kM.v64] ^ (tmp >> 1) ^ ((tmp & 1) ? kA.v64 : 0);
	}

	for (; i < kN.v64 - 1; ++i) {
		uint64_t tmp = (iState.vector.v64[i] & ~kLowerMask.v64) | (iState.vector.v64[i + 1] & kLowerMask.v64);
		iState.vector.v64[i] = iState.vector.v64[i + kM.v64 - kN.v64] ^ (tmp >> 1) ^ ((tmp & 1) ? kA.v64 : 0);
	}

	uint64_t tmp = (iState.vector.v64[kN.v64 - 1] & ~kLowerMask.v64) | (iState.vector.v64[0] & kLowerMask.v64);
	iState.vector.v64[kN.v64 - 1] = iState.vector.v64[kM.v64 - 1] ^ (tmp >> 1) ^ ((tmp & 1) ? kA.v64 : 0);

	iState.index.v64 = 0;
}

c_use_decl_annotations uint32_t c_rand32_int(void) {
	if (c_unlikely(!lState.v32.seeded)) {
		logMsg(c_logger_Severity_WARNING, "%s(): generator not seeded. Calling c_rand32_seed() with default seed %"PRIu32" and continuing...", __func__, kDefaultSeed.v32);
		c_rand32_seed(kDefaultSeed.v32);
	}

	// Twist
	if (c_unlikely(iState.index.v32 >= kN.v32))
		_twist32();

	uint32_t y = iState.vector.v32[iState.index.v32++];
	y ^= (y >> kU.v32) & kD.v32;
	y ^= (y << kS.v32) & kB.v32;
	y ^= (y << kT.v32) & kC.v32;
	y ^= (y >> kL.v32);

	return y;
}

c_use_decl_annotations uint64_t c_rand64_int(void) {
	if (c_unlikely(!lState.v64.seeded)) {
		logMsg(c_logger_Severity_WARNING, "%s(): generator not seeded. Calling c_rand64_seed() with default seed %"PRIu64" and continuing...", __func__, kDefaultSeed.v64);
		c_rand64_seed(kDefaultSeed.v64);
	}

	// Twist
	if (c_unlikely(iState.index.v64 >= kN.v64))
		_twist64();

	uint64_t y = iState.vector.v64[iState.index.v64++];
	y ^= (y >> kU.v64) & kD.v64;
	y ^= (y << kS.v64) & kB.v64;
	y ^= (y << kT.v64) & kC.v64;
	y ^= (y >> kL.v64);

	return y;
}

c_use_decl_annotations double c_rand(void) {
	const double twom32 = 0x1p-32;
	return c_rand32_int() * twom32;
}

c_use_decl_annotations double c_rand64(void) {
	const double twom64 = 0x1p-64;
	return c_rand64_int() * twom64;
}

c_use_decl_annotations double c_rand64_normal(void) {
	static bool valid = FALSE;
	static double res_next = 0.;
	double res_this;

	if (!valid) {
		double u, v, s;

		do {
			// get a point in the unit circle
			u = fma(2., c_rand64(), -1.);
			v = fma(2., c_rand64(), -1.);
			s = u * u + v * v;
		} while (s > 1.);

		const double f = sqrt(-2. * log(s) / s);

		// save second value for next call
		res_this = f * u;
		res_next = f * v;
		valid = TRUE;
	}
	else {
		res_this = res_next;
		valid = FALSE;
	}

	return res_this;
}
