/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENSerDes.c
*	\brief		TCP/IP CAEN protocol.
*	\author		Giovanni Cerretani
*
******************************************************************************/

#ifdef _WIN32
#include <WS2tcpip.h>
#endif

#include <CAENSerDes.h>

#include <CAENLogger.h>
#include <CAENMultiplatform.h>
#include <CAENThread.h>
#include <CAENSocket.h>

INIT_C_LOGGER("CAENSerDesLog.log", "CAENSerDes.c");

#define PKTHEAD_PROTOVERS_SIZE  (sizeof(PKTHEAD_PROTOVERS_TYPE))
#define PKTHEAD_TOTLEN_SIZE     (sizeof(PKTHEAD_TOTLEN_TYPE))
#define PKTHEAD_CMD_SIZE        (sizeof(PKTHEAD_CMD_TYPE))
#define PKTHEAD_NPARAMS_SIZE    (sizeof(PKTHEAD_NPARAMS_TYPE))
#define PKTHEAD_TOTALSIZE       (PKTHEAD_PROTOVERS_SIZE + PKTHEAD_TOTLEN_SIZE + PKTHEAD_CMD_SIZE + PKTHEAD_NPARAMS_SIZE)

static PKTHEAD_PROTOVERS_TYPE * getProtoVersPtr(uint8_t *header) {
	uint8_t *datapointer = header;
	return ((PKTHEAD_PROTOVERS_TYPE*)datapointer);
}

static const PKTHEAD_PROTOVERS_TYPE* getProtoVersConstPtr(const uint8_t* header) {
	const uint8_t* datapointer = header;
	return ((const PKTHEAD_PROTOVERS_TYPE*)datapointer);
}

static PKTHEAD_TOTLEN_TYPE * getSizePtr(uint8_t *header) {
	uint8_t *datapointer = header + PKTHEAD_PROTOVERS_SIZE;
	return ((PKTHEAD_TOTLEN_TYPE*)datapointer);
}

static PKTHEAD_CMD_TYPE * getCmdPtr(uint8_t *header) {
	uint8_t *datapointer = header + PKTHEAD_PROTOVERS_SIZE + PKTHEAD_TOTLEN_SIZE;
	return ((PKTHEAD_CMD_TYPE*)datapointer);
}

static const PKTHEAD_CMD_TYPE* getCmdConstPtr(const uint8_t* header) {
	const uint8_t* datapointer = header + PKTHEAD_PROTOVERS_SIZE + PKTHEAD_TOTLEN_SIZE;
	return ((const PKTHEAD_CMD_TYPE*)datapointer);
}

static PKTHEAD_NPARAMS_TYPE * getParamsNumPtr(uint8_t *header) {
	uint8_t *datapointer = header + PKTHEAD_PROTOVERS_SIZE + PKTHEAD_TOTLEN_SIZE + PKTHEAD_CMD_SIZE;
	return ((PKTHEAD_NPARAMS_TYPE*)datapointer);
}

static bool checkProtoVersPtr(const uint8_t *header, uint8_t version) {
	const PKTHEAD_PROTOVERS_TYPE *datapointer = getProtoVersConstPtr(header);
	PKTHEAD_PROTOVERS_TYPE packet_version = c_ntoh16(*datapointer);
	return (packet_version == CAEN_PKT_VERSION(version));
}

static PKTHEAD_TOTLEN_TYPE * setSizeToHeader(uint8_t *header, PKTHEAD_TOTLEN_TYPE size) {
	PKTHEAD_TOTLEN_TYPE *datapointer = getSizePtr(header);
	PKTHEAD_TOTLEN_TYPE s = c_hton32(size);
	return c_memcpy(datapointer, &s, PKTHEAD_TOTLEN_SIZE);
}

static PKTHEAD_NPARAMS_TYPE * incrHeaderParamsNum(uint8_t *header) {
	PKTHEAD_NPARAMS_TYPE *datapointer = getParamsNumPtr(header);
	PKTHEAD_NPARAMS_TYPE num = *datapointer;
	num = c_ntoh16(num);
	num++;
	num = c_hton16(num);
	return c_memcpy(datapointer, &num, PKTHEAD_NPARAMS_SIZE);
}

static size_t getSizeOfType(c_type_t t) {
	switch (t) {
	case TYPE_INT8:		return sizeof(int8_t);
	case TYPE_UINT8:	return sizeof(uint8_t);
	case TYPE_INT16:	return sizeof(int16_t);
	case TYPE_UINT16:	return sizeof(uint16_t);
	case TYPE_INT32:	return sizeof(int32_t);
	case TYPE_UINT32:	return sizeof(uint32_t);
	case TYPE_INT64:	return sizeof(int64_t);
	case TYPE_UINT64:	return sizeof(uint64_t);
	case TYPE_LONG:		return sizeof(int32_t);
	case TYPE_DOUBLE:	return sizeof(double);
	case TYPE_CHAR:		return sizeof(char);
	case TYPE_SHORT:	return sizeof(int16_t);
	case TYPE_RESCODE:
	case TYPE_STRING:
	case TYPE_MEMORY:
	case TYPE_NONE:
	default:			return 0;
	}
}

c_use_decl_annotations uint8_t* c_createheader(uint16_t version, PKTHEAD_CMD_TYPE cmd, size_t *allocatedSize) {
	if (allocatedSize == NULL)
		return NULL;
	PKTHEAD_PROTOVERS_TYPE protovers = CAEN_PKT_VERSION(version);
	PKTHEAD_NPARAMS_TYPE nparams = 0;
	PKTHEAD_TOTLEN_TYPE totlen = PKTHEAD_TOTALSIZE;
	uint8_t *header = c_malloc(totlen);
	if (header == NULL)
		*allocatedSize = 0;
	else {
		protovers = c_hton16(protovers);
		c_memcpy(getProtoVersPtr(header), &protovers, PKTHEAD_PROTOVERS_SIZE);

		*allocatedSize = (size_t)totlen;
		setSizeToHeader(header, totlen);

		cmd = c_hton16(cmd);
		c_memcpy(getCmdPtr(header), &cmd, PKTHEAD_CMD_SIZE);

		nparams = c_hton16(nparams);
		c_memcpy(getParamsNumPtr(header), &nparams, PKTHEAD_NPARAMS_SIZE);
	}
	return header;
}

c_use_decl_annotations PKTHEAD_CMD_TYPE c_getcmd(const uint8_t *buffer) {
	PKTHEAD_CMD_TYPE cmd = CMD_INVALID_LOCAL;
	if (buffer != NULL && checkProtoVersPtr(buffer, 0)) { // Check buffer starting with header
		const PKTHEAD_CMD_TYPE *cmdAddr = getCmdConstPtr(buffer);
		cmd = c_ntoh16(*cmdAddr);
	}
	return cmd;
}

c_use_decl_annotations uint8_t* c_recv_packet(const c_socket_t *sckt, PKTHEAD_CMD_TYPE *cmd, PKTHEAD_TOTLEN_TYPE *totSize, PKTHEAD_NPARAMS_TYPE *totParams) {
	uint8_t header[PKTHEAD_TOTALSIZE];
	uint8_t *buffer = NULL;
	c_ssize_t recvsize;
	PKTHEAD_TOTLEN_TYPE buffersize;
	PKTHEAD_CMD_TYPE l_cmd;
	PKTHEAD_TOTLEN_TYPE l_totSize;
	PKTHEAD_NPARAMS_TYPE l_totParams;

	if (sckt == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): Invalid c_socket_t.", __func__);
		return NULL;
	}

	// Receive the header containing the size of the packet
	recvsize = c_recv(sckt, header, PKTHEAD_TOTALSIZE);
	if (recvsize != PKTHEAD_TOTALSIZE) {
		logMsg(c_logger_Severity_ERROR, "%s(): recv() returned wrong header size.", __func__);
		goto ErrFunc;
	}

	l_cmd = *(getCmdPtr(header));
	*cmd = c_ntoh16(l_cmd);
	l_totSize = *(getSizePtr(header));
	*totSize = c_ntoh32(l_totSize);
	l_totParams = *(getParamsNumPtr(header));
	*totParams = c_ntoh16(l_totParams);

	// Receive the rest of the packet, now that we know its size.
	buffersize = *totSize - PKTHEAD_TOTALSIZE; // We already got the header

	if (buffersize > 0) {
		buffer = c_malloc(buffersize);
		if (buffer == NULL) {
			goto ErrFunc;
		}

		recvsize = c_recv(sckt, buffer, buffersize);
		if (recvsize != (c_ssize_t)buffersize) {
			logMsg(c_logger_Severity_ERROR, "%s(): recv() returned wrong packet size.", __func__);
			goto ErrFunc;
		}
	}
	
	return buffer;

ErrFunc:
	*cmd = CMD_INVALID_LOCAL;
	*totSize = 0;
	*totParams = 0;
	c_free(buffer);
	return NULL;
}

c_use_decl_annotations uint8_t* c_recv_packet_unlock(c_socket_t *sckt, PKTHEAD_CMD_TYPE *cmd, PKTHEAD_TOTLEN_TYPE *totSize, PKTHEAD_NPARAMS_TYPE *totParams) {
	uint8_t *ret = c_recv_packet(sckt, cmd, totSize, totParams);
	if (c_mutex_unlock(&sckt->mutex) != CAENThread_RetCode_Success) {
		logMsg(c_logger_Severity_ERROR, "%s(): Error in c_mutex_unlock().", __func__);
		c_free(ret);
		return NULL;
	}

	return ret;
}

typedef uint16_t		PARAMTYPE_TYPE;
typedef uint16_t		PARAMNUM_TYPE;

static PARAMTYPE_TYPE ptypeton(PARAMTYPE_TYPE type) {
	return c_hton16(type);
}

static PARAMTYPE_TYPE ntoptype(PARAMTYPE_TYPE type) {
	return c_ntoh16(type);
}

static PARAMNUM_TYPE pnumton(PARAMNUM_TYPE type) {
	return c_hton16(type);
}

static PARAMNUM_TYPE ntopnum(PARAMNUM_TYPE type) {
	return c_ntoh16(type);
}

static c_type_t toggleSigned(c_type_t type) {
	switch (type) {
	case TYPE_INT8: return TYPE_UINT8;
	case TYPE_UINT8: return TYPE_INT8;
	case TYPE_INT16: return TYPE_UINT16;
	case TYPE_UINT16: return TYPE_INT16;
	case TYPE_INT32: return TYPE_UINT32;
	case TYPE_UINT32: return TYPE_INT32;
	case TYPE_INT64: return TYPE_UINT64;
	case TYPE_UINT64: return TYPE_INT64;
	default: return TYPE_NONE;
	}
}

// Serialize of base types
c_use_decl_annotations uint8_t* serialize_int8_t(const int8_t *src, uint32_t num, uint8_t *buffer, size_t *allocSize) {
	const c_type_t type = TYPE_INT8;
	size_t paramSize = getSizeOfType(type);
	size_t sizeToAdd = sizeof(PARAMTYPE_TYPE) + sizeof(PARAMNUM_TYPE) + (num * paramSize);
	if (buffer == NULL || *allocSize == 0) {
		buffer = NULL; // TODO possible memleak
		*allocSize = 0;
	}
	else {
		if (!checkProtoVersPtr(buffer, 0)) // Check buffer starting with header
			return NULL;
		// realloc buffer to host new parameter
		buffer = c_realloc(buffer, *allocSize + sizeToAdd);
		if (buffer == NULL) // Check buffer allocation
			return NULL;
		else {
			uint8_t *ptr = buffer + *allocSize; // Point to the new buffer's free space
			PARAMTYPE_TYPE pType = ptypeton(type);
			PARAMNUM_TYPE pNum = pnumton((PARAMNUM_TYPE)num);
			// Copy ParamType to buffer
			c_memcpy(ptr, &pType, sizeof(PARAMTYPE_TYPE));
			ptr += sizeof(PARAMTYPE_TYPE);
			// Copy ParamNum to buffer
			c_memcpy(ptr, &pNum, sizeof(PARAMNUM_TYPE));
			ptr += sizeof(PARAMNUM_TYPE);
			// Copy parameters to buffer
			// TODO we can also do a straight c_memcpy for 1byte sized types
			for (uint32_t i = 0; i < num; i++) {
				int8_t value = src[i];
				c_memcpy(ptr, &value, paramSize);
				ptr += paramSize;
			}
			*allocSize += sizeToAdd;
			setSizeToHeader(buffer, (PKTHEAD_TOTLEN_TYPE)*allocSize);
			incrHeaderParamsNum(buffer);
		}
	}
	return buffer;
}
c_use_decl_annotations uint8_t* serialize_char(const char *src, uint32_t num, uint8_t *buffer, size_t *allocSize) {
	uint32_t len = (uint32_t)strnlen(src, (size_t)num);
	return serialize_int8_t((const int8_t*)src, len, buffer, allocSize);
}
c_use_decl_annotations uint8_t* serialize_uint8_t(const uint8_t *src, uint32_t num, uint8_t *buffer, size_t *allocSize) {
	const c_type_t type = TYPE_UINT8;
	size_t paramSize = getSizeOfType(type);
	size_t sizeToAdd = sizeof(PARAMTYPE_TYPE) + sizeof(PARAMNUM_TYPE) + (num * paramSize);
	if (buffer == NULL || *allocSize == 0) {
		buffer = NULL; // TODO possible memleak
		*allocSize = 0;
	}
	else {
		if (!checkProtoVersPtr(buffer, 0)) // Check buffer starting with header
			return NULL;
		// realloc buffer to host new parameter
		buffer = c_realloc(buffer, *allocSize + sizeToAdd);
		if (buffer == NULL) // Check buffer allocation
			return NULL;
		else {
			uint8_t *ptr = buffer + *allocSize; // Point to the new buffer's free space
			PARAMTYPE_TYPE pType = ptypeton(type);
			PARAMNUM_TYPE pNum = pnumton((PARAMNUM_TYPE)num);
			// Copy ParamType to buffer
			c_memcpy(ptr, &pType, sizeof(PARAMTYPE_TYPE));
			ptr += sizeof(PARAMTYPE_TYPE);
			// Copy ParamNum to buffer
			c_memcpy(ptr, &pNum, sizeof(PARAMNUM_TYPE));
			ptr += sizeof(PARAMNUM_TYPE);
			// Copy parameters to buffer
			for (uint32_t i = 0; i < num; i++) {
				uint8_t value = src[i];
				c_memcpy(ptr, &value, paramSize);
				ptr += paramSize;
			}
			*allocSize += sizeToAdd;
			setSizeToHeader(buffer, (PKTHEAD_TOTLEN_TYPE)*allocSize);
			incrHeaderParamsNum(buffer);
		}
	}
	return buffer;
}
c_use_decl_annotations uint8_t* serialize_int16_t(const int16_t *src, uint32_t num, uint8_t *buffer, size_t *allocSize) {
	const c_type_t type = TYPE_INT16;
	size_t paramSize = getSizeOfType(type);
	size_t sizeToAdd = sizeof(PARAMTYPE_TYPE) + sizeof(PARAMNUM_TYPE) + (num * paramSize);
	if (buffer == NULL || *allocSize == 0) {
		buffer = NULL; // TODO possible memleak
		*allocSize = 0;
	}
	else {
		if (!checkProtoVersPtr(buffer, 0)) // Check buffer starting with header
			return NULL;
		// realloc buffer to host new parameter
		buffer = c_realloc(buffer, *allocSize + sizeToAdd);
		if (buffer == NULL) // Check buffer allocation
			return NULL;
		else {
			uint8_t *ptr = buffer + *allocSize; // Point to the new buffer's free space
			PARAMTYPE_TYPE pType = ptypeton(type);
			PARAMNUM_TYPE pNum = pnumton((PARAMNUM_TYPE)num);
			// Copy ParamType to buffer
			c_memcpy(ptr, &pType, sizeof(PARAMTYPE_TYPE));
			ptr += sizeof(PARAMTYPE_TYPE);
			// Copy ParamNum to buffer
			c_memcpy(ptr, &pNum, sizeof(PARAMNUM_TYPE));
			ptr += sizeof(PARAMNUM_TYPE);
			// Copy parameters to buffer
			for (uint32_t i = 0; i < num; i++) {
				int16_t value = c_hton16(src[i]);
				c_memcpy(ptr, &value, paramSize);
				ptr += paramSize;
			}
			*allocSize += sizeToAdd;
			setSizeToHeader(buffer, (PKTHEAD_TOTLEN_TYPE)*allocSize);
			incrHeaderParamsNum(buffer);
		}
	}
	return buffer;
}
c_use_decl_annotations uint8_t* serialize_uint16_t(const uint16_t *src, uint32_t num, uint8_t *buffer, size_t *allocSize) {
	const c_type_t type = TYPE_UINT16;
	size_t paramSize = getSizeOfType(type);
	size_t sizeToAdd = sizeof(PARAMTYPE_TYPE) + sizeof(PARAMNUM_TYPE) + (num * paramSize);
	if (buffer == NULL || *allocSize == 0) {
		buffer = NULL; // TODO possible memleak
		*allocSize = 0;
	}
	else {
		if (!checkProtoVersPtr(buffer, 0)) // Check buffer starting with header
			return NULL;
		// realloc buffer to host new parameter
		buffer = c_realloc(buffer, *allocSize + sizeToAdd);
		if (buffer == NULL) // Check buffer allocation
			return NULL;
		else {
			uint8_t *ptr = buffer + *allocSize; // Point to the new buffer's free space
			PARAMTYPE_TYPE pType = ptypeton(type);
			PARAMNUM_TYPE pNum = pnumton((PARAMNUM_TYPE)num);
			// Copy ParamType to buffer
			c_memcpy(ptr, &pType, sizeof(PARAMTYPE_TYPE));
			ptr += sizeof(PARAMTYPE_TYPE);
			// Copy ParamNum to buffer
			c_memcpy(ptr, &pNum, sizeof(PARAMNUM_TYPE));
			ptr += sizeof(PARAMNUM_TYPE);
			// Copy parameters to buffer
			for (uint32_t i = 0; i < num; i++) {
				uint16_t value = c_hton16(src[i]);
				c_memcpy(ptr, &value, paramSize);
				ptr += paramSize;
			}
			*allocSize += sizeToAdd;
			setSizeToHeader(buffer, (PKTHEAD_TOTLEN_TYPE)*allocSize);
			incrHeaderParamsNum(buffer);
		}
	}
	return buffer;
}
c_use_decl_annotations uint8_t* serialize_int32_t(const int32_t *src, uint32_t num, uint8_t *buffer, size_t *allocSize) {
	const c_type_t type = TYPE_INT32;
	size_t paramSize = getSizeOfType(type);
	size_t sizeToAdd = sizeof(PARAMTYPE_TYPE) + sizeof(PARAMNUM_TYPE) + (num * paramSize);
	if (buffer == NULL || *allocSize == 0) {
		buffer = NULL; // TODO possible memleak
		*allocSize = 0;
	}
	else {
		if (!checkProtoVersPtr(buffer, 0)) // Check buffer starting with header
			return NULL;
		// realloc buffer to host new parameter
		buffer = c_realloc(buffer, *allocSize + sizeToAdd);
		if (buffer == NULL) // Check buffer allocation
			return NULL;
		else {
			uint8_t *ptr = buffer + *allocSize; // Point to the new buffer's free space
			PARAMTYPE_TYPE pType = ptypeton(type);
			PARAMNUM_TYPE pNum = pnumton((PARAMNUM_TYPE)num);
			// Copy ParamType to buffer
			c_memcpy(ptr, &pType, sizeof(PARAMTYPE_TYPE));
			ptr += sizeof(PARAMTYPE_TYPE);
			// Copy ParamNum to buffer
			c_memcpy(ptr, &pNum, sizeof(PARAMNUM_TYPE));
			ptr += sizeof(PARAMNUM_TYPE);
			// Copy parameters to buffer
			for (uint32_t i = 0; i < num; i++) {
				int32_t value = c_hton32(src[i]);
				c_memcpy(ptr, &value, paramSize);
				ptr += paramSize;
			}
			*allocSize += sizeToAdd;
			setSizeToHeader(buffer, (PKTHEAD_TOTLEN_TYPE)*allocSize);
			incrHeaderParamsNum(buffer);
		}
	}
	return buffer;
}
c_use_decl_annotations uint8_t* serialize_uint32_t(const uint32_t *src, uint32_t num, uint8_t *buffer, size_t *allocSize) {
	const c_type_t type = TYPE_UINT32;
	size_t paramSize = getSizeOfType(type);
	size_t sizeToAdd = sizeof(PARAMTYPE_TYPE) + sizeof(PARAMNUM_TYPE) + (num * paramSize);
	if (buffer == NULL || *allocSize == 0) {
		buffer = NULL; // TODO possible memleak
		*allocSize = 0;
	}
	else {
		if (!checkProtoVersPtr(buffer, 0)) // Check buffer starting with header
			return NULL;
		// realloc buffer to host new parameter
		buffer = c_realloc(buffer, *allocSize + sizeToAdd);
		if (buffer == NULL) // Check buffer allocation
			return NULL;
		else {
			uint8_t *ptr = buffer + *allocSize; // Point to the new buffer's free space
			PARAMTYPE_TYPE pType = ptypeton(type);
			PARAMNUM_TYPE pNum = pnumton((PARAMNUM_TYPE)num);
			// Copy ParamType to buffer
			c_memcpy(ptr, &pType, sizeof(PARAMTYPE_TYPE));
			ptr += sizeof(PARAMTYPE_TYPE);
			// Copy ParamNum to buffer
			c_memcpy(ptr, &pNum, sizeof(PARAMNUM_TYPE));
			ptr += sizeof(PARAMNUM_TYPE);
			// Copy parameters to buffer
			for (uint32_t i = 0; i < num; i++) {
				uint32_t value = c_hton32(src[i]);
				c_memcpy(ptr, &value, paramSize);
				ptr += paramSize;
			}
			*allocSize += sizeToAdd;
			setSizeToHeader(buffer, (PKTHEAD_TOTLEN_TYPE)*allocSize);
			incrHeaderParamsNum(buffer);
		}
	}
	return buffer;
}
c_use_decl_annotations uint8_t* serialize_int64_t(const int64_t *src, uint32_t num, uint8_t *buffer, size_t *allocSize) {
	const c_type_t type = TYPE_INT64;
	size_t paramSize = getSizeOfType(type);
	size_t sizeToAdd = sizeof(PARAMTYPE_TYPE) + sizeof(PARAMNUM_TYPE) + (num * paramSize);
	if (buffer == NULL || *allocSize == 0) {
		buffer = NULL; // TODO possible memleak
		*allocSize = 0;
	}
	else {
		if (!checkProtoVersPtr(buffer, 0)) // Check buffer starting with header
			return NULL;
		// realloc buffer to host new parameter
		buffer = c_realloc(buffer, *allocSize + sizeToAdd);
		if (buffer == NULL) // Check buffer allocation
			return NULL;
		else {
			uint8_t *ptr = buffer + *allocSize; // Point to the new buffer's free space
			PARAMTYPE_TYPE pType = ptypeton(type);
			PARAMNUM_TYPE pNum = pnumton((PARAMNUM_TYPE)num);
			// Copy ParamType to buffer
			c_memcpy(ptr, &pType, sizeof(PARAMTYPE_TYPE));
			ptr += sizeof(PARAMTYPE_TYPE);
			// Copy ParamNum to buffer
			c_memcpy(ptr, &pNum, sizeof(PARAMNUM_TYPE));
			ptr += sizeof(PARAMNUM_TYPE);
			// Copy parameters to buffer
			for (uint32_t i = 0; i < num; i++) {
				int64_t value = c_hton64(src[i]);
				c_memcpy(ptr, &value, paramSize);
				ptr += paramSize;
			}
			*allocSize += sizeToAdd;
			setSizeToHeader(buffer, (PKTHEAD_TOTLEN_TYPE)*allocSize);
			incrHeaderParamsNum(buffer);
		}
	}
	return buffer;
}
c_use_decl_annotations uint8_t* serialize_uint64_t(const uint64_t *src, uint32_t num, uint8_t *buffer, size_t *allocSize) {
	const c_type_t type = TYPE_UINT64;
	size_t paramSize = getSizeOfType(type);
	size_t sizeToAdd = sizeof(PARAMTYPE_TYPE) + sizeof(PARAMNUM_TYPE) + (num * paramSize);
	if (buffer == NULL || *allocSize == 0) {
		buffer = NULL; // TODO possible memleak
		*allocSize = 0;
	}
	else {
		if (!checkProtoVersPtr(buffer, 0)) // Check buffer starting with header
			return NULL;
		// realloc buffer to host new parameter
		buffer = c_realloc(buffer, *allocSize + sizeToAdd);
		if (buffer == NULL) // Check buffer allocation
			return NULL;
		else {
			uint8_t *ptr = buffer + *allocSize; // Point to the new buffer's free space
			PARAMTYPE_TYPE pType = ptypeton(type);
			PARAMNUM_TYPE pNum = pnumton((PARAMNUM_TYPE)num);
			// Copy ParamType to buffer
			c_memcpy(ptr, &pType, sizeof(PARAMTYPE_TYPE));
			ptr += sizeof(PARAMTYPE_TYPE);
			// Copy ParamNum to buffer
			c_memcpy(ptr, &pNum, sizeof(PARAMNUM_TYPE));
			ptr += sizeof(PARAMNUM_TYPE);
			// Copy parameters to buffer
			for (uint32_t i = 0; i < num; i++) {
				uint64_t value = c_hton64(src[i]);
				c_memcpy(ptr, &value, paramSize);
				ptr += paramSize;
			}
			*allocSize += sizeToAdd;
			setSizeToHeader(buffer, (PKTHEAD_TOTLEN_TYPE)*allocSize);
			incrHeaderParamsNum(buffer);
		}
	}
	return buffer;
}
c_use_decl_annotations uint8_t* serialize_float(const float *src, uint32_t num, uint8_t *buffer, size_t *allocSize) {
	// cppcheck-suppress invalidPointerCast
	return serialize_int32_t((const int32_t*)src, num, buffer, allocSize);
}
c_use_decl_annotations uint8_t* serialize_double(const double *src, uint32_t num, uint8_t *buffer, size_t *allocSize) {
	int64_t *mantissa = c_malloc(num * sizeof(*mantissa));
	int16_t *exponent = c_malloc(num * sizeof(*exponent));
	int8_t *classification = c_malloc(num * sizeof(*classification));

	if (mantissa == NULL || exponent == NULL || classification == NULL) {
		buffer = NULL;
		goto exit;
	}

	// Expand double values in mantissa, exponent and classification
	for (uint32_t i = 0; i < num; i++)
		mantissa[i] = c_frexp(src[i], &exponent[i], &classification[i]);

	// Serialize them separately
	buffer = serialize_int64_t(mantissa, num, buffer, allocSize);
	buffer = serialize_int16_t(exponent, num, buffer, allocSize);
	buffer = serialize_int8_t(classification, num, buffer, allocSize);

exit:

	c_free(mantissa);
	c_free(exponent);
	c_free(classification);

	return buffer;
}

// Deserialize of base types
c_use_decl_annotations uint8_t* deserialize_int8_t(int8_t *dest, uint32_t num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_INT8;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum != num) // Wrong parameters number.
		return NULL;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	// NOTE: for byte-sized types we can simply memcopy
	// since we don't have endianness problems.
	c_memcpy(dest, buffer, num * pSize);
	buffer += num * pSize;

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_int8_t_array(int8_t *dest, uint32_t maxnum, uint32_t *num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_INT8;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);

	if (gNum > maxnum) // Error
		return NULL;

	*num = gNum;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	// NOTE: for byte-sized types we can simply memcopy
	// since we don't have endianness problems.
	c_memcpy(dest, buffer, gNum * pSize);
	buffer += gNum * pSize;

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_char(char *dest, uint32_t maxnum, uint8_t *buffer) {
	//return deserialize_int8_t((int8_t*)dest, num, buffer);
	uint32_t num;
	buffer = deserialize_int8_t_array((int8_t*)dest, maxnum, &num, buffer);
	// if the function succeded, memset the remaining string to 0
	if (buffer != NULL)
		c_zeromem(&dest[num], maxnum - num);
	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_uint8_t(uint8_t *dest, uint32_t num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_UINT8;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum != num) // Wrong parameters number.
		return NULL;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	// NOTE: for byte-sized types we can simply memcopy
	// since we don't have endianness problems.
	c_memcpy(dest, buffer, num * pSize);
	buffer += num * pSize;

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_uint8_t_array(uint8_t *dest, uint32_t maxnum, uint32_t *num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_UINT8;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);

	if (gNum > maxnum) // Error
		return NULL;

	*num = gNum;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	// NOTE: for byte-sized types we can simply memcopy
	// since we don't have endianness problems.
	c_memcpy(dest, buffer, gNum * pSize);
	buffer += gNum * pSize;

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_int16_t(int16_t *dest, uint32_t num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_INT16;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum != num) // Wrong parameters number.
		return NULL;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	for (uint32_t i = 0; i < num; i++) {
		c_memcpy(&dest[i], buffer, pSize);
		dest[i] = c_ntoh16(dest[i]);
		buffer += pSize;
	}

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_int16_t_array(int16_t *dest, uint32_t maxnum, uint32_t *num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_INT16;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum > maxnum) // Too much parameters
		return NULL;
	*num = gNum;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	for (uint32_t i = 0; i < gNum; i++) {
		c_memcpy(&dest[i], buffer, pSize);
		dest[i] = c_ntoh16(dest[i]);
		buffer += pSize;
	}

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_uint16_t(uint16_t *dest, uint32_t num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_UINT16;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum != num) // Wrong parameters number.
		return NULL;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	for (uint32_t i = 0; i < num; i++) {
		c_memcpy(&dest[i], buffer, pSize);
		dest[i] = c_ntoh16(dest[i]);
		buffer += pSize;
	}

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_uint16_t_array(uint16_t *dest, uint32_t maxnum, uint32_t *num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_UINT16;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum > maxnum) // Too much parameters
		return NULL;
	*num = gNum;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	for (uint32_t i = 0; i < gNum; i++) {
		c_memcpy(&dest[i], buffer, pSize);
		dest[i] = c_ntoh16(dest[i]);
		buffer += pSize;
	}

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_int32_t(int32_t *dest, uint32_t num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_INT32;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum != num) // Wrong parameters number.
		return NULL;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	for (uint32_t i = 0; i < num; i++) {
		c_memcpy(&dest[i], buffer, pSize);
		dest[i] = c_ntoh32(dest[i]);
		buffer += pSize;
	}

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_int32_t_array(int32_t *dest, uint32_t maxnum, uint32_t *num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_INT32;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum > maxnum) // Too much parameters
		return NULL;
	*num = gNum;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	for (uint32_t i = 0; i < gNum; i++) {
		c_memcpy(&dest[i], buffer, pSize);
		dest[i] = c_ntoh32(dest[i]);
		buffer += pSize;
	}

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_uint32_t(uint32_t *dest, uint32_t num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_UINT32;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum != num) // Wrong parameters number.
		return NULL;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	for (uint32_t i = 0; i < num; i++) {
		c_memcpy(&dest[i], buffer, pSize);
		dest[i] = c_ntoh32(dest[i]);
		buffer += pSize;
	}

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_uint32_t_array(uint32_t *dest, uint32_t maxnum, uint32_t *num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_UINT32;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum > maxnum) // Too much parameters
		return NULL;
	*num = gNum;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	for (uint32_t i = 0; i < gNum; i++) {
		c_memcpy(&dest[i], buffer, pSize);
		dest[i] = c_ntoh32(dest[i]);
		buffer += pSize;
	}

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_int64_t(int64_t *dest, uint32_t num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_INT64;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum != num) // Wrong parameters number.
		return NULL;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	for (uint32_t i = 0; i < num; i++) {
		c_memcpy(&dest[i], buffer, pSize);
		dest[i] = c_ntoh64(dest[i]);
		buffer += pSize;
	}

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_int64_t_array(int64_t *dest, uint32_t maxnum, uint32_t *num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_INT64;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum > maxnum) // Too much parameters
		return NULL;
	*num = gNum;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	for (uint32_t i = 0; i < gNum; i++) {
		c_memcpy(&dest[i], buffer, pSize);
		dest[i] = c_ntoh64(dest[i]);
		buffer += pSize;
	}

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_uint64_t(uint64_t *dest, uint32_t num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_UINT64;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum != num) // Wrong parameters number.
		return NULL;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	for (uint32_t i = 0; i < num; i++) {
		c_memcpy(&dest[i], buffer, pSize);
		dest[i] = c_ntoh64(dest[i]);
		buffer += pSize;
	}

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_uint64_t_array(uint64_t *dest, uint32_t maxnum, uint32_t *num, uint8_t *buffer) {
	PARAMTYPE_TYPE type;
	PARAMNUM_TYPE gNum = 0;
	const c_type_t expType = TYPE_UINT64;
	size_t pSize = getSizeOfType(expType);
	if (buffer == NULL)
		return NULL;

	// Get param type (accept identically signed and unsigned)
	c_memcpy(&type, buffer, sizeof(PARAMTYPE_TYPE));
	type = ntoptype(type);
	if (type != expType && type != toggleSigned(expType)) // Invalid type.
		return NULL;
	buffer += sizeof(PARAMTYPE_TYPE);

	// Get param num
	c_memcpy(&gNum, buffer, sizeof(PARAMNUM_TYPE));
	gNum = ntopnum(gNum);
	if (gNum > maxnum) // Too much parameters
		return NULL;
	*num = gNum;
	buffer += sizeof(PARAMNUM_TYPE);

	// Get params
	for (uint32_t i = 0; i < gNum; i++) {
		c_memcpy(&dest[i], buffer, pSize);
		dest[i] = c_ntoh64(dest[i]);
		buffer += pSize;
	}

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_float(float *dest, uint32_t num, uint8_t *buffer) {
	// cppcheck-suppress invalidPointerCast
	return deserialize_int32_t((int32_t*)dest, num, buffer);
}
c_use_decl_annotations uint8_t* deserialize_float_array(float *dest, uint32_t maxnum, uint32_t *num, uint8_t *buffer) {
	// cppcheck-suppress invalidPointerCast
	return deserialize_int32_t_array((int32_t*)dest, maxnum, num, buffer);
}
c_use_decl_annotations uint8_t* deserialize_double(double *dest, uint32_t num, uint8_t *buffer) {
	int64_t *mantissa = c_malloc(num * sizeof(*mantissa));
	int16_t *exponent = c_malloc(num * sizeof(*exponent));
	int8_t *classification = c_malloc(num * sizeof(*classification));

	if (mantissa == NULL || exponent == NULL || classification == NULL) {
		buffer = NULL;
		goto exit;
	}

	// Serialize them separately
	buffer = deserialize_int64_t(mantissa, num, buffer);
	buffer = deserialize_int16_t(exponent, num, buffer);
	buffer = deserialize_int8_t(classification, num, buffer);

	// Expand double values in mantissa, exponent and classification
	for (uint32_t i = 0; i < num; i++)
		dest[i] = c_ldexp(mantissa[i], exponent[i], classification[i]);

exit:

	c_free(mantissa);
	c_free(exponent);
	c_free(classification);

	return buffer;
}
c_use_decl_annotations uint8_t* deserialize_double_array(double *dest, uint32_t maxnum, uint32_t *num, uint8_t *buffer) {
	int64_t *mantissa = c_malloc(maxnum * sizeof(*mantissa));
	int16_t *exponent = c_malloc(maxnum * sizeof(*exponent));
	int8_t *classification = c_malloc(maxnum * sizeof(*classification));

	if (mantissa == NULL || exponent == NULL || classification == NULL) {
		buffer = NULL;
		goto exit;
	}

	// Serialize them separately
	buffer = deserialize_int64_t_array(mantissa, maxnum, num, buffer);
	buffer = deserialize_int16_t_array(exponent, maxnum, num, buffer);
	buffer = deserialize_int8_t_array(classification, maxnum, num, buffer);

	// Expand double values in mantissa, exponent and classification
	for (uint32_t i = 0; i < *num; i++)
		dest[i] = c_ldexp(mantissa[i], exponent[i], classification[i]);

exit:

	c_free(mantissa);
	c_free(exponent);
	c_free(classification);

	return buffer;
}
