/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENThread.c
*	\brief		Functions to handle threads, inspired to C11 threads.h.
*	\author
*
******************************************************************************/

#include <CAENThread.h>

#ifdef _WIN32
#include <process.h>
#else
#include <unistd.h>
#include <pthread.h>
#include <sched.h>
#include <errno.h>
#include <time.h> // for CLOCK_REALTIME
#endif

#include <CAENUtility.h>
#include <CAENMultiplatform.h>
#include <CAENLogger.h>

INIT_C_LOGGER("CAENThreadLog.txt", "CAENThread.c");

/*
 * Windows implementation is mostly copied from https://github.com/microsoft/STL/blob/main/stl/src/cthread.cpp,
 * that is used as backend for std::thread in the Windows C++ standard library. Even if undocumented, it has
 * the same API of C11 threads.
 *
 * Linux implementation is inspired from the glibc, based on pthread. C11 threads are almost identical to
 * pthread, with minor differences like the signature of the function pointer passed to thrd_create, returning
 * int instead of void*.
 *
 * As extension, we add boolean field to c_thread_t to mark the validity of the object, be used for internal
 * checks.
 */

int32_t c_thread_setaffinity(c_thread_t thr, uint64_t mask) {
	int32_t ret = CAENThread_RetCode_Success;
#ifdef _WIN32
#if 0 // HACK removed because it gave problems in Windows 10...
	const DWORD_PTR r = SetThreadAffinityMask(thr.handle, (DWORD_PTR)mask);
	if (r == 0)
		ret = CAENThread_RetCode_Error;
#else
	c_unused_parameter(thr);
	c_unused_parameter(mask);
#endif
#else
	cpu_set_t cpuset;
	CPU_ZERO(&cpuset);
	for (size_t i = 0; mask != 0; ++i, mask >>= 1)
		if (mask & 0x1)
			CPU_SET(i, &cpuset);
	int r = pthread_setaffinity_np(thr.handle, sizeof(cpu_set_t), &cpuset);
	switch (r) {
	case 0:
		break;
	case EFAULT:
	case EINVAL:
	case ESRCH:
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#endif
	return ret;
}

c_thread_t c_thread_current(void) {
	c_thread_t result;
	result.valid = TRUE;
#ifdef _WIN32
	result.handle = NULL;
	result.id = GetCurrentThreadId();
#else
	result.handle = pthread_self();
#endif
	return result;
}

void c_thread_yield(void) {
#ifdef _WIN32
	/*
	 * Can be implemented with:
	 * - Sleep(0)
	 * - SwitchToThread()
	 * We use the second solution, as it is used by Microsoft on _Thrd_yield() on xthreads.h.
	 * See https://stackoverflow.com/a/1383966/3287591 for detils.
	 */
	SwitchToThread();
#else
	sched_yield();
#endif
}

int32_t c_thread_detach(c_thread_t thr) {
	int32_t ret = CAENThread_RetCode_Success;
#ifdef _WIN32
	if (!CloseHandle(thr.handle))
		ret = CAENThread_RetCode_Error;
#else
	int r = pthread_detach(thr.handle);
	switch (r) {
	case 0:
		break;
	case EINVAL:
	case ESRCH:
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#endif
	return ret;
}

void c_thread_exit(int res) {
#ifdef _WIN32
	_endthreadex((unsigned int)res);
#else
	pthread_exit((void*)(intptr_t)res);
#endif
}

bool c_thread_equal(c_thread_t lhs, c_thread_t rhs) {
	if (!lhs.valid && !rhs.valid)
		return true; // FALSE validity is enough to declare invalid c_thread_t equality
#ifdef _WIN32
	return lhs.id == rhs.id;
#else
	return pthread_equal(lhs.handle, rhs.handle);
#endif
}

#ifndef _WIN32
// creates a timespec representing time NOW+ms from epoch
static struct timespec _getTimeSpecFromNow(uint32_t ms) {
	struct timespec ts;
	uint32_t sec = (ms / 1000);
	uint32_t nsec = (ms - sec * 1000) * 1000000;
	clock_gettime(CLOCK_REALTIME, &ts);
	ts.tv_sec += sec;
	ts.tv_nsec += nsec;
	//tv_nsec can be now higher than 1e9
	ts.tv_sec += ts.tv_nsec / 1000000000;
	ts.tv_nsec = ts.tv_nsec % 1000000000;
	return ts;
}
#endif

int32_t c_semaphore_init(c_semaphore_t *s) {
	if (c_unlikely(s == NULL))
		return CAENThread_RetCode_Error;
	int32_t ret = CAENThread_RetCode_Success;
#ifndef _WIN32
	if (sem_init(s, 0, 0) != 0)
		ret = CAENThread_RetCode_Error;
#else
	const HANDLE sem = CreateSemaphoreA(NULL, 0, 1, NULL);
	if (sem == NULL)
		ret = CAENThread_RetCode_Error;
	else
		*s = sem;
#endif
	return ret;
}

int32_t c_semaphore_multi_init(c_semaphore_t *s) {
	if (c_unlikely(s == NULL))
		return CAENThread_RetCode_Error;
	int32_t ret = CAENThread_RetCode_Success;
#ifndef _WIN32
	if (sem_init(s, 0, 0) != 0)
		ret = CAENThread_RetCode_Error;
#else
	const HANDLE semaphore = CreateSemaphoreA(NULL, 0, MAXLONG, NULL);
	if (semaphore == NULL)
		ret = CAENThread_RetCode_Error;
	else
		*s = semaphore;
#endif
	return ret;
}

int32_t c_semaphore_destroy(c_semaphore_t *s) {
	if (c_unlikely(s == NULL))
		return CAENThread_RetCode_Error;
	int32_t ret = CAENThread_RetCode_Success;
#ifndef _WIN32
	if (sem_destroy(s) != 0)
		ret = CAENThread_RetCode_Error;
#else
	if (!CloseHandle(*s))
		ret = CAENThread_RetCode_Error;
#endif
	return ret;
}

int32_t c_semaphore_wait(c_semaphore_t *s, int32_t ms) {
	if (c_unlikely(s == NULL))
		return CAENThread_RetCode_Error;
	int32_t ret = CAENThread_RetCode_Success;
#ifndef _WIN32
	int r;
	switch (ms) {
	case 0:
		r = (sem_timedwait(s, &(struct timespec){ 0 }) == 0) ? 0 : errno;
		break;
	case INFINITE:
		r = (sem_wait(s) == 0) ? 0 : errno;
		break;
	default: {
		const struct timespec ts = _getTimeSpecFromNow(ms);
		r = (sem_timedwait(s, &ts) == 0) ? 0 : errno;
		break;
	}
	}
	switch (r) {
	case 0:
		break;
	case ETIMEDOUT:
		ret = CAENThread_RetCode_Timedout;
		break;
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#else
	const DWORD r = WaitForSingleObjectEx(*s, (DWORD)ms, FALSE);
	switch (r) {
	case WAIT_OBJECT_0:
		break;
	case WAIT_TIMEOUT:
		ret = CAENThread_RetCode_Timedout;
		break;
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#endif
	return ret;
}

int32_t c_semaphore_post(c_semaphore_t *s) {
	if (c_unlikely(s == NULL))
		return CAENThread_RetCode_Error;
	int32_t ret = CAENThread_RetCode_Success;
#ifndef _WIN32
	if (sem_post(s) != 0)
		ret = CAENThread_RetCode_Error;
#else
	if (!ReleaseSemaphore(*s, 1, NULL))
		ret = CAENThread_RetCode_Error;
#endif
	return ret;
}

int32_t c_mutex_init(c_mutex_t *m) {
	if (c_unlikely(m == NULL))
		return CAENThread_RetCode_Error;
	int32_t ret = CAENThread_RetCode_Success;
#ifndef _WIN32
	const int r = pthread_mutex_init(m, NULL);
	switch (r) {
	case 0:
		break;
	case ENOMEM:
		ret = CAENThread_RetCode_Nomem;
		break;
	case EAGAIN:
	case EPERM:
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#else
	const HANDLE mutex = CreateMutexA(NULL, FALSE, NULL);
	if (mutex == NULL)
		ret = CAENThread_RetCode_Error;
	else
		*m = mutex;
#endif
	return ret;
}

c_use_decl_annotations int32_t c_mutex_lock(c_mutex_t *m) {
	if (c_unlikely(m == NULL))
		return CAENThread_RetCode_Error;
	int32_t ret = CAENThread_RetCode_Success;
#ifndef _WIN32
	const int r = pthread_mutex_lock(m);
	switch (r) {
	case 0:
		break;
	case EAGAIN:
	case EINVAL:
	case EDEADLK:
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#else
	const DWORD r = WaitForSingleObjectEx(*m, INFINITE, FALSE);
	if (r != WAIT_OBJECT_0)
		ret = CAENThread_RetCode_Error;
#endif
	return ret;
}

c_use_decl_annotations int32_t c_mutex_trylock(c_mutex_t *m) {
	if (c_unlikely(m == NULL))
		return CAENThread_RetCode_Error;
	int32_t ret = CAENThread_RetCode_Success;
#ifndef _WIN32
	const int r = pthread_mutex_trylock(m);
	switch (r) {
	case 0:
		break;
	case EBUSY:
		ret = CAENThread_RetCode_Busy;
		break;
	case EAGAIN:
	case EINVAL:
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#else
	const DWORD ms_wait = 0;
	const DWORD r = WaitForSingleObjectEx(*m, ms_wait, FALSE);
	if (r != WAIT_OBJECT_0)
		ret = CAENThread_RetCode_Error;
#endif
	return ret;
}

c_use_decl_annotations int32_t c_mutex_unlock(c_mutex_t *m) {
	if (c_unlikely(m == NULL))
		return CAENThread_RetCode_Error;
	int32_t ret = CAENThread_RetCode_Success;
#ifndef _WIN32
	const int r = pthread_mutex_unlock(m);
	switch (r) {
	case 0:
		break;
	case EPERM:
	case EAGAIN:
	case EINVAL:
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#else
	if (!ReleaseMutex(*m))
		ret = CAENThread_RetCode_Error;
#endif
	return ret;
}

int32_t c_mutex_destroy(c_mutex_t *m) {
	if (c_unlikely(m == NULL))
		return CAENThread_RetCode_Error;
	int32_t ret = CAENThread_RetCode_Success;
#ifndef _WIN32
	const int r = pthread_mutex_destroy(m);
	switch (r) {
	case 0:
		break;
	case EBUSY:
		ret = CAENThread_RetCode_Busy;
		break;
	case EINVAL:
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#else
	if (!CloseHandle(*m))
		ret = CAENThread_RetCode_Error;
#endif
	return ret;
}

int32_t c_condition_init(c_condition_t *cond) {
#ifdef _WIN32
	InitializeConditionVariable(&(cond->_cond));
	InitializeCriticalSection(&(cond->_critical_section));
#else
	if (pthread_cond_init(&(cond->_cond), NULL) != 0)
		return CAENThread_RetCode_Error;
	if (c_mutex_init(&(cond->_mutex)) != CAENThread_RetCode_Success)
		return CAENThread_RetCode_Error;
#endif
	return CAENThread_RetCode_Success;
}

int32_t c_condition_destroy(c_condition_t *cond) {
#ifdef _WIN32
	DeleteCriticalSection(&(cond->_critical_section));
#else
	if (pthread_cond_destroy(&(cond->_cond)) != 0)
		return CAENThread_RetCode_Error;
	if (c_mutex_destroy(&(cond->_mutex)) != CAENThread_RetCode_Success)
		return CAENThread_RetCode_Error;
#endif
	return CAENThread_RetCode_Success;
}

c_use_decl_annotations int32_t c_condition_lock(c_condition_t *cond) {
#ifdef _WIN32
	EnterCriticalSection(&(cond->_critical_section));
	return CAENThread_RetCode_Success;
#else
	return c_mutex_lock(&(cond->_mutex));
#endif
}

c_use_decl_annotations int32_t c_condition_unlock(c_condition_t *cond) {
#ifdef _WIN32
	LeaveCriticalSection(&(cond->_critical_section));
	return CAENThread_RetCode_Success;
#else
	return c_mutex_unlock(&(cond->_mutex));
#endif
}

int32_t c_condition_wait(c_condition_t *cond, int32_t ms) {
	int32_t ret = CAENThread_RetCode_Success;
#ifdef _WIN32
	if (!SleepConditionVariableCS(&(cond->_cond), &(cond->_critical_section), (DWORD)ms)) {
		switch (GetLastError()) {
		case ERROR_TIMEOUT:
			ret = CAENThread_RetCode_Timedout;
			break;
		default:
			ret = CAENThread_RetCode_Error;
			break;
		}
	}
#else
	int r;
	switch (ms) {
	case 0:
		r = (pthread_cond_timedwait(&(cond->_cond), &(cond->_mutex), &(struct timespec){ 0 }) == 0) ? 0 : errno;
		break;
	case INFINITE:
		r = (pthread_cond_wait(&(cond->_cond), &(cond->_mutex)) == 0) ? 0 : errno;
		break;
	default: {
		const struct timespec ts = _getTimeSpecFromNow(ms);
		r = (pthread_cond_timedwait(&(cond->_cond), &(cond->_mutex), &ts) == 0) ? 0 : errno;
		break;
	}
	}
	switch (r) {
	case 0:
		break;
	case ETIMEDOUT:
		ret = CAENThread_RetCode_Timedout;
		break;
	case EINVAL:
	case EPERM:
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#endif
	return ret;
}

int32_t c_condition_signal_one(c_condition_t *cond) {
	int32_t ret = CAENThread_RetCode_Success;
#ifdef _WIN32
	WakeConditionVariable(&(cond->_cond));
#else
	const int r = pthread_cond_signal(&(cond->_cond));
	switch (r) {
	case 0:
		break;
	case EINVAL:
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#endif
	return ret;
}

int32_t c_condition_broadcast(c_condition_t *cond) {
	int32_t ret = CAENThread_RetCode_Success;
#ifdef _WIN32
	WakeAllConditionVariable(&(cond->_cond));
#else
	const int r = pthread_cond_broadcast(&(cond->_cond));
	switch (r) {
	case 0:
		break;
	case EINVAL:
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#endif
	return ret;
}

int32_t c_thread_sleep(const struct timespec* duration, struct timespec* remaining) {
	int32_t ret = CAENThread_RetCode_Success;
#ifdef _WIN32
	struct timespec start;
	if (timespec_get(&start, TIME_UTC) == 0)
		return CAENThread_RetCode_Error;
	const DWORD ms = (DWORD)(duration->tv_sec * 1000 + duration->tv_nsec / 1000000 + (((duration->tv_nsec % 1000000) == 0) ? 0 : 1));
	const DWORD r = SleepEx(ms, TRUE);
	if (r != 0) {
		if (remaining != NULL) {
			if (timespec_get(remaining, TIME_UTC) == 0)
				return CAENThread_RetCode_Error;
			remaining->tv_sec -= start.tv_sec;
			remaining->tv_nsec -= start.tv_nsec;
			if (remaining->tv_nsec < 0) {
				remaining->tv_nsec += 1000000000;
				remaining->tv_sec -= 1;
			}
		}
		ret = (r == WAIT_IO_COMPLETION) ? -1 : -2;
	}
#else
	const int r = nanosleep(duration, remaining);
	if (r != 0)
		ret = (errno == EINTR) ? -1 : -2;
#endif
	return ret;
}

int32_t c_sleep(uint32_t ms) {
	int32_t ret = CAENThread_RetCode_Success;
#ifdef _WIN32
	// SleepEx() never fails if bAlertable is FALSE
	SleepEx((DWORD)ms, FALSE);
#else // Linux
	// usleep() return 0 in case of success. errno could be read for error details.
	const int r = usleep(ms * 1000);
	if (r != 0) {
		switch (errno) {
		case EINTR:
		case EINVAL:
		default:
			ret = CAENThread_RetCode_Error;
		}
	}
#endif
	return ret;
}

struct tstart_wrapper_t {
	c_tstart_t f;
	void *arg;
};

// wrappers to make c_tstart_t the same on Windows and Linux
#ifdef _WIN32
static unsigned __stdcall _tstart_wrapper_function(void* _arg) {
#else
static void* _tstart_wrapper_function(void* _arg) {
#endif
	struct tstart_wrapper_t* wrapper = _arg;
	c_tstart_t f = wrapper->f;
	void* arg = wrapper->arg;
	c_free(wrapper);
	int r = f(arg);
#ifdef _WIN32
	return (unsigned)r;
#else
	return (void*)(intptr_t)r;
#endif
}

c_thread_t c_thread_invalid() {
	c_thread_t res;
#ifdef _WIN32 // Windows
	res.handle = NULL;
	res.id = 0;
#else
	res.handle = 0;
	res.valid = FALSE;
#endif
	return res;
}

int32_t c_thread_create(c_thread_t *thr, c_tstart_t t, void *arg) {
	int32_t ret = CAENThread_RetCode_Success;
	struct tstart_wrapper_t *wrapper = c_malloc(sizeof(*wrapper)); // freed by the thread
	if (wrapper == NULL) {
		return CAENThread_RetCode_Nomem;
	}
	wrapper->f = t;
	wrapper->arg = arg;
#ifdef _WIN32 // Windows
	thr->handle = (HANDLE)_beginthreadex(NULL, 0, _tstart_wrapper_function, wrapper, 0, &thr->id);
	if (thr->handle == NULL) {
		ret = CAENThread_RetCode_Error;
	}
#else // Linux
	const int r = pthread_create(&(thr->handle), NULL, _tstart_wrapper_function, (void*)wrapper);
	switch (r) {
	case 0:
		break;
	case EAGAIN:
	case EINVAL:
	case EPERM:
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
#endif
	if (ret == CAENThread_RetCode_Success)
		thr->valid = TRUE;
	else
		*thr = c_thread_invalid();
	return ret;
}

int32_t c_thread_join(c_thread_t thr, int *res) {
	return c_thread_timedjoin(thr, res, INFINITE);
}

int32_t c_thread_timedjoin(c_thread_t thr, int *res, int32_t ms) {
	int32_t ret = CAENThread_RetCode_Success;
#ifndef _WIN32
	int r;
	void *lres;
	switch (ms) {
	case INFINITE:
		r = pthread_join(thr.handle, &lres);
		break;
	default: {
		struct timespec ts = _getTimeSpecFromNow(ms);
		r = pthread_timedjoin_np(thr.handle, &lres, &ts);
		break;
	}
	}
	switch (r) {
	case 0:
		break;
	case ETIMEDOUT: // pthread_timedjoin_np only
		ret = CAENThread_RetCode_Timedout;
		break;
	case EBUSY: // pthread_timedjoin_np only
		ret = CAENThread_RetCode_Busy;
		break;
	case EINVAL:
	case ESRCH:
	case EDEADLK:
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
	if (res != NULL) {
		*res = (int)(intptr_t)lres;
	}
#else
	const DWORD r = WaitForSingleObjectEx(thr.handle, (DWORD)ms, FALSE);
	switch (r) {
	case WAIT_OBJECT_0:
		ret = CAENThread_RetCode_Success;
		break;
	case WAIT_TIMEOUT:
		ret = CAENThread_RetCode_Timedout;
		break;
	default:
		ret = CAENThread_RetCode_Error;
		break;
	}
	if (res != NULL) {
		DWORD lres;
		if (!GetExitCodeThread(thr.handle, &lres)) {
			ret = CAENThread_RetCode_Error;
		} else {
			*res = (int)lres;
		}
	}
	if (!CloseHandle(thr.handle))
		ret = CAENThread_RetCode_Error;
#endif
	return ret;
}

int32_t c_ticket_init(c_ticket_t *ticket) {
	ticket->_queue_tail = 0;
	ticket->_queue_head = 0;
	return c_condition_init(&ticket->_condition);
}

int32_t c_ticket_delete(c_ticket_t *ticket) {
	return c_condition_destroy(&ticket->_condition);
}

static uint64_t _getnewticketnumber(c_ticket_t* ticket) {
	uint64_t res = ticket->_queue_tail++;
	// check if 'queue_tail' overflowed. In case, reset everything keeping tail/head difference
	if (c_unlikely(ticket->_queue_tail < res)) {
		uint64_t diff = res - ticket->_queue_head; // NOTE: queue_me is always greater than head
		res = diff;
		ticket->_queue_tail = res + 1;
		ticket->_queue_head = 0;
	}
	return res;
}

int32_t c_ticket_lock(c_ticket_t *ticket) {

	int32_t ret = c_condition_lock(&ticket->_condition);
	if (ret != CAENThread_RetCode_Success)
		return ret;

	// lock only if this thread is not already holding the lock
	if (!c_thread_equal(ticket->_holding_thread_id, c_thread_current())) {

		uint64_t queue_me = _getnewticketnumber(ticket);

		while (ret == CAENThread_RetCode_Success && queue_me != ticket->_queue_head) {
			ret = c_condition_wait(&ticket->_condition, 1000);
			if (ret == CAENThread_RetCode_Timedout) {
				logMsg(c_logger_Severity_WARNING, "Ticket number %" PRIu64 " has timed out. Going on anyway.", queue_me);
				ret = CAENThread_RetCode_Success;
			}
		}
		// now the mutex is locked. Get the thread ID holding it.
		ticket->_holding_thread_id = c_thread_current();
	}
	else {
		// ERROR? DEBUG ASSERTION? IGNORE?
		logMsg(c_logger_Severity_WARNING, "A thread is trying to lock a ticket it already holds. Ignoring lock.");
	}

	// while() needed to remove C26165 on MSVC:
	// - on Windows c_condition_unlock cannot fail
	// - on Linux c_condition_unlock cannot fail if lock succeeded just before
	while (c_condition_unlock(&ticket->_condition) != CAENThread_RetCode_Success);

	return ret;
}

int32_t c_ticket_unlock(c_ticket_t *ticket) {

	int32_t ret = c_condition_lock(&ticket->_condition);
	if (ret != CAENThread_RetCode_Success)
		return ret;

	// unlock only if this thread is holding the lock
	if (c_thread_equal(ticket->_holding_thread_id, c_thread_current())) {
		ticket->_queue_head++;
		ticket->_holding_thread_id = c_thread_invalid();
		ret = c_condition_broadcast(&ticket->_condition);
	}
	else {
		// ERROR? DEBUG ASSERTION? IGNORE?
		logMsg(c_logger_Severity_WARNING, "A thread is trying to unlock a ticket it doesn't hold. Ignoring unlock.");
	}

	// while() needed to remove C26165 on MSVC:
	// - on Windows c_condition_unlock cannot fail
	// - on Linux c_condition_unlock cannot fail if lock succeeded just before
	while (c_condition_unlock(&ticket->_condition) != CAENThread_RetCode_Success);

	return ret;
}
