/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2014, Xsoda
*
*	This file is part of the CAEN Utility.
*
*	This file is distributed under the BSD 2-Clause "Simplified" License.
*
*	SPDX-License-Identifier: BSD-2-Clause
*
***************************************************************************//*!
*
*	\file		CAENUrl.c
*	\brief		Utilities to parse a standard URL.
*	\author		Giovanni Cerretani, Xsoda
*
******************************************************************************/

#include <CAENUrl.h>

#include <CAENLogger.h>
#include <CAENMultiplatform.h>

INIT_C_LOGGER("CAENUrlLog.txt", "CAENUrl.c");

static bool host_is_ipv4(const char *str) {
	if (str == NULL)
		return FALSE;
	while (*str != '\0') {
		if ((*str >= '0' && *str <= '9') || *str == '.')
			str++;
		else
			return FALSE;
	}
	return TRUE;
}

static void parse_query(c_url_field_t *url, const char *query) {
	const char *chr = strchr(query, '=');
	while (chr != NULL) {

		void *ptr = c_realloc(url->query, (url->query_num + 1) * sizeof(*url->query));
		if (ptr == NULL)
			return;
		url->query = ptr;
		url->query[url->query_num].name = c_strndup(query, chr - query);
		query = chr + 1;
		chr = strchr(query, '&');
		if (chr != NULL) {
			url->query[url->query_num].value = c_strndup(query, chr - query);
			url->query_num++;
			query = chr + 1;
			chr = strchr(query, '=');
		}
		else {
			url->query[url->query_num].value = c_strdup(query);
			url->query_num++;
			break;
		}
	}
}

c_use_decl_annotations c_url_field_t * c_url_parse(const char *str) {
	const char *pch;
	char *query = NULL;
	c_url_field_t *url = c_calloc(1, sizeof(*url));
	if (url == NULL) {
		return NULL;
	}

	if (str != NULL && str[0] != '\0') {
		url->href = c_strdup(str);
		pch = strchr(str, ':');   /* parse schema */
		if (pch != NULL && pch[1] == '/' && pch[2] == '/') {
			url->schema = c_strndup(str, pch - str);
			str = pch + 3;
		}
		else
			goto Fail;
		pch = strchr(str, '@');   /* parse user info */
		if (pch != NULL) {
			pch = strchr(str, ':');
			if (pch != NULL) {
				url->username = c_strndup(str, pch - str);
				str = pch + 1;
				pch = strchr(str, '@');
				if (pch != NULL) {
					url->password = c_strndup(str, pch - str);
					str = pch + 1;
				}
				else
					goto Fail;
			}
			else
				goto Fail;
		}
		if (str[0] == '[') {       /* parse host info */
			str++;
			pch = strchr(str, ']');
			if (pch != NULL) {
				url->host = c_strndup(str, pch - str);
				str = pch + 1;
				if (str[0] == ':') {
					str++;
					pch = strchr(str, '/');
					if (pch) {
						url->port = c_strndup(str, pch - str);
						str = pch + 1;
					}
					else {
						url->port = c_strdup(str);
						str += strlen(str);
					}
				}
				url->host_type = HOST_IPV6;
			}
			else
				goto Fail;
		}
		else {
			const char *pch_slash;
			pch = strchr(str, ':');
			pch_slash = strchr(str, '/');
			if (pch != NULL && ((pch_slash == NULL) || (pch < pch_slash))) {
				url->host = c_strndup(str, pch - str);
				str = pch + 1;
				pch = strchr(str, '/');
				if (pch != NULL) {
					url->port = c_strndup(str, pch - str);
					str = pch + 1;
				}
				else {
					url->port = c_strdup(str);
					str += strlen(str);
				}
			}
			else {
				pch = strchr(str, '/');
				if (pch != NULL) {
					url->host = c_strndup(str, pch - str);
					str = pch + 1;
				}
				else {
					url->host = c_strdup(str);
					str += strlen(str);
				}
			}
			url->host_type = host_is_ipv4(url->host) ? HOST_IPV4 : HOST_DOMAIN;
		}
		if (str[0] != '\0') {               /* parse path, query and fragment */
			pch = strchr(str, '?');
			if (pch != NULL) {
				url->path = c_strndup(str, pch - str);
				str = pch + 1;
				pch = strchr(str, '#');
				if (pch != NULL) {
					query = c_strndup(str, pch - str);
					str = pch + 1;
					url->fragment = c_strdup(str);
				}
				else {
					query = c_strdup(str);
					str += strlen(str);
				}
				parse_query(url, query);
				c_free(query);
			}
			else {
				pch = strchr(str, '#');
				if (pch != NULL) {
					url->path = c_strndup(str, pch - str);
					str = pch + 1;
					url->fragment = c_strdup(str);
					str += strlen(str);
				}
				else {
					url->path = c_strdup(str);
					str += strlen(str);
				}
			}
		}
	}
	else {
Fail:
		logMsg(c_logger_Severity_ERROR, "%s(): failed.", __func__);
		c_url_free(url);
		url = NULL;
	}
	return url;
}

void c_url_free(c_url_field_t *url) {
	if (url == NULL)
		return;

	c_free(url->href);
	c_free(url->schema);
	c_free(url->username);
	c_free(url->password);
	c_free(url->host);
	c_free(url->port);
	c_free(url->path);
	if (url->query != NULL) {
		for (uint32_t i = 0; i < url->query_num; ++i) {
			c_free(url->query[i].name);
			c_free(url->query[i].value);
		}
		c_free(url->query);
	}
	c_free(url->fragment);
	c_free(url);
}
