/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENUtility.c
*	\brief		Main header and generic tools.
*	\author
*
******************************************************************************/

#ifndef VISUAL_LEAK_DETECTOR_ENABLED
#define VISUAL_LEAK_DETECTOR_ENABLED (0)
#endif
#define IS_VLD_ENABLED (_DEBUG && _WIN32 && _CAEN_UTILITY_EXPORT && VISUAL_LEAK_DETECTOR_ENABLED)
#if IS_VLD_ENABLED
#include <vld.h>
#endif

#include <CAENUtility.h>

#include <string.h>

#include <CAENLogger.h>
#include <CAENRandom.h>
#include <CAENXMLParser.h>
#include <CAENMultiplatform.h>

// string right trim
size_t c_srtrim(char *str, const char *remove) {
    size_t l = strlen(str);
    while (strchr(remove, str[l - 1]) != NULL && l > 0) {
        str[l - 1] = '\0';
        l = strlen(str);
    }
    return strlen(str);
}

// string left trim
size_t c_sltrim(char *str, const char *remove) {
    char *ptr = str;
    while (strchr(remove, ptr[0]) != NULL && strlen(ptr) > 0)
        ptr++;
    for (size_t i = 0; i <= strlen(ptr); i++)
        str[i] = ptr[i];
    return strlen(str);
}

// string trim
size_t c_strim(char *str, const char *remove) {
    if (c_srtrim(str, remove) == 0)
        return 0;
    if (c_sltrim(str, remove) == 0)
        return 0;
    return strlen(str);
}

static int32_t _initLibrary(void) {
	// Get time (in ms) to initialize stuff
	const uint64_t time = c_get_time();

	// Set logger start time
	c_lsetst(time);

#ifndef _NOUSEXML
	// Test if the loaded XML version is compatible with the linked one.
	LIBXML_TEST_VERSION;

	// Overload the libxml memory functions with the library ones.
	// A function name, in this context, is converted to the address of the function.
	xmlMemSetup(c_free, c_malloc, c_realloc, c_strdup);
#endif //_NOUSEXML

	// Initialize random generators
	c_rand32_init();
	c_rand64_init();

	// Seed random generators with time
	c_rand32_seed((uint32_t)time);
	c_rand64_seed(time);

	return c_Utility_ErrorCode_Success;
}

static int32_t _closeLibrary(void) {

#ifndef _NOUSEXML
	// Clean XML stuff
	c_xml_parsercleanup();
#endif //_NOUSEXML

	// Deinit random generators
	c_rand32_deinit();
	c_rand64_deinit();

	// Deinit logger
	c_ldeinit();

	return c_Utility_ErrorCode_Success;
}

#ifdef _CAEN_UTILITY_EXPORT
#ifdef _WIN32
/**********************************************************************\
                                DllMain
\**********************************************************************/
BOOL WINAPI DllMain(HINSTANCE hModule, DWORD ul_reason_for_call, LPVOID lpReserved) {
	UNREFERENCED_PARAMETER(hModule);
	UNREFERENCED_PARAMETER(lpReserved);
	switch (ul_reason_for_call) {
	case DLL_PROCESS_ATTACH:
		_initLibrary();
		break;
	case DLL_PROCESS_DETACH:
		_closeLibrary();
		break;
	case DLL_THREAD_ATTACH:
		break;
	case DLL_THREAD_DETACH:
		break;
	}
	return TRUE;
}
#else
/**********************************************************************\
                        CAENUtility_init
\**********************************************************************/
void __attribute__ ((constructor)) CAENUtility_init(void) {
	_initLibrary();
}
/**********************************************************************\
                        CAENUtility_close
\**********************************************************************/
void __attribute__ ((destructor)) CAENUtility_close(void) {
	_closeLibrary();
}
#endif
#endif
