/******************************************************************************
*
*	CAEN SpA - Software Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*******************************************************************************
*
*	Copyright (C) 2019-2022 CAEN SpA
*
*	This file is part of the CAEN Utility.
*
*	The CAEN Utility is free software; you can redistribute it and/or
*	modify it under the terms of the GNU Lesser General Public
*	License as published by the Free Software Foundation; either
*	version 3 of the License, or (at your option) any later version.
*
*	The CAEN Utility is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
*	Lesser General Public License for more details.
*
*	You should have received a copy of the GNU Lesser General Public
*	License along with the CAEN Utility; if not, see
*	https://www.gnu.org/licenses/.
*
*	SPDX-License-Identifier: LGPL-3.0-or-later
*
***************************************************************************//*!
*
*	\file		CAENXMLParser.c
*	\brief		Tools to parse an XML file using libxml.
*	\author
*
******************************************************************************/

#ifndef _NOUSEXML

#include <CAENXMLParser.h>

#include <libxml/xpathInternals.h>
#include <libxml/globals.h>
#include <libxml/parser.h>
#include <libxml/xmlversion.h>
#include <libxml/xpath.h>

#include <CAENMultiplatform.h>
#include <CAENLogger.h>

static c_libhandle_t libHandle = NULL;

INIT_C_LOGGER("CAENXMLParserLog.txt", "CAENXMLParser.c");

c_use_decl_annotations c_xmlfile_t* c_xml_newfile(const char *filename) {
	c_xmlfile_t *file = c_malloc(sizeof(*file));
	if (file == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): Can't allocate space for c_xmlfile_t.", __func__);
		return file;
	}

	file->doc = xmlReadFile(filename, NULL, XML_PARSE_NOBLANKS | XML_PARSE_PEDANTIC);
	if (file->doc == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): Can't open XML file '%s'.", __func__, filename);
		c_free(file);
		return NULL;
	}

	file->name = c_strdup(filename);
	if (file->name == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): Can't alloc space for filename '%s'.", __func__, filename);
		xmlFreeDoc(file->doc);
		c_free(file);
		return NULL;
	}

	return file;
}

c_use_decl_annotations c_xmlfile_t* c_xml_newfile_from_memory(const char *buffer, int size) {
	c_xmlfile_t *file = c_malloc(sizeof(*file));
	if (file == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): Can't allocate space for c_xmlfile_t.", __func__);
		return file;
	}

	char filename[] = "memory";

	file->doc = xmlReadMemory(buffer, size, filename, NULL, XML_PARSE_NOBLANKS | XML_PARSE_PEDANTIC);
	if (file->doc == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): Can't open XML buffer.", __func__);
		c_free(file);
		return NULL;
	}

	file->name = c_strdup(filename);
	if (file->name == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): Can't alloc space for filename '%s'.", __func__, filename);
		xmlFreeDoc(file->doc);
		c_free(file);
		return NULL;
	}

	return file;
}

void c_xml_freefile(c_xmlfile_t *file) {
	if (file == NULL)
		return;

	if (file->doc != NULL)
		xmlFreeDoc(file->doc);

	c_free(file->name);
	c_free(file);
}

c_use_decl_annotations xmlNode* c_xml_getrootnode(const c_xmlfile_t *file) {
	return xmlDocGetRootElement(file->doc);
}

c_use_decl_annotations xmlNodeSet* c_xml_getnodeset(c_xmlfile_t *file, const xmlChar *query) {
	xmlXPathContext *context = xmlXPathNewContext(file->doc);
	if (context == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): Error in xmlXPathNewContext.", __func__);
		return NULL;
	}
	xmlXPathObject *result = xmlXPathEval(query, context);
	xmlXPathFreeContext(context);
	if (result == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): Error in xmlXPathEval.", __func__);
		return NULL;
	}
	if (xmlXPathNodeSetIsEmpty(result->nodesetval)) {
		xmlXPathFreeObject(result);
		return NULL;
	}

	// Copy the nodeset (copy done if the first argument of xmlXPathNodeSetMerge() is NULL)
	xmlNodeSet *nodeset = xmlXPathNodeSetMerge(NULL, result->nodesetval);
	
	xmlXPathFreeObject(result);

	return nodeset;
}

void c_xml_freenodeset(xmlNodeSet *nodeset) {
	if (nodeset != NULL)
		xmlXPathFreeNodeSet(nodeset);
}

c_use_decl_annotations const char* c_xml_getattributevalue(const xmlNode *node, const char *attribute, bool warnEnable) {
	xmlAttr *prop = xmlHasProp(node, (const xmlChar*)attribute);
	if (prop != NULL)
		return (const char *)XML_GET_CONTENT(prop->children);

    if (warnEnable)
        logMsg(c_logger_Severity_WARNING, "%s(): Node '%s' doesn't have attribute '%s'", __func__, node->name, attribute);

    return NULL;
}

c_use_decl_annotations xmlNode* c_xml_getfirstnodefromxpathquery(c_xmlfile_t *file, const xmlChar *query) {
    xmlNodeSet *nodeset = c_xml_getnodeset(file, query);
    xmlNode *result;
    if (xmlXPathNodeSetGetLength(nodeset) != 1) {
        logMsg(c_logger_Severity_ERROR, "%s(): More than one element match query '%s'", __func__, query);
        result = NULL;
    }
    else {
        result = xmlXPathNodeSetItem(nodeset, 0);
    }
    c_xml_freenodeset(nodeset);
    return result;
}

c_use_decl_annotations xmlNode* c_xml_getnextelementfromname(xmlNode *node, const char *nodename) {
	while (node != NULL) {
		if (node->type == XML_ELEMENT_NODE && xmlStrEqual(node->name, (const xmlChar*)nodename)) {
			return node;
		}
		else {
			node = xmlNextElementSibling(node);
		}
	}
    return node;
}

c_use_decl_annotations xmlNode* c_getfirstsiblingelementfromname(xmlNode *node, const char *nodename) {
    if (node == NULL) {
        logMsg(c_logger_Severity_ERROR, "%s(): Invalid argument. 'node' cannot be NULL to get first sibling from name.", __func__);
        return NULL;
    }

    return c_xml_getnextelementfromname(xmlNextElementSibling(node), nodename);
}

c_use_decl_annotations xmlNode* c_xml_getfirstchildrenelementfromname(xmlNode *node, const char *nodename) {
    if (node == NULL) {
        logMsg(c_logger_Severity_ERROR, "%s(): Invalid argument. 'node' cannot be NULL to get first child from name.", __func__);
        return NULL;
    }

    return c_xml_getnextelementfromname(xmlFirstElementChild(node), nodename);
}

// same as 'c_xml_getfirstchildrenelementfromname' but also navigates sublevels
c_use_decl_annotations xmlNode* c_xml_getfirstdescendantelementfromname(xmlNode *node, const char *nodename) {
    char *str;
    char *level;
    const char delim[] = "/";

    if (node == NULL) {
        logMsg(c_logger_Severity_ERROR, "%s(): Invalid argument. 'node' cannot be NULL to get first descendant from name.", __func__);
        return NULL;
    }
    if (nodename == NULL) {
        logMsg(c_logger_Severity_ERROR, "%s(): Invalid argument. 'nodename' cannot be NULL to get first descendant from name.", __func__);
        return NULL;
    }

    str = c_strdup(nodename);
    if (str == NULL) {
        logMsg(c_logger_Severity_ERROR, "%s(): Can't allocate a string.", __func__);
        return NULL;
    }

    level = strtok(str, delim);
    while (level != NULL) {
        node = c_xml_getfirstchildrenelementfromname(node, level);
        if (node == NULL)
            break;
        level = strtok(NULL, delim);
    }
    c_free(str);
    return node;
}

c_use_decl_annotations const char * c_xml_getpath(const c_xmlfile_t *file) {
	if (file == NULL)
		return NULL;
	return file->name;
}

int32_t c_xml_getcodefromnodechild(const xmlNode *node, double *code) {
	if (node == NULL || code == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): Invalid argument.", __func__);
		return c_xml_RetCode_NULL;
	}

	xmlChar *valuestr = xmlNodeGetContent(node);
	if (valuestr == NULL)
		return c_xml_RetCode_NotFound;

	char *end_ptr;
	double res = strtod((char*)valuestr, &end_ptr);

	// If no conversion can be performed, strtod returns 0 and end_ptr is set to valuestr.
	if (end_ptr == (char*)valuestr) {
		logMsg(c_logger_Severity_ERROR, "%s(): Can't parse a value from string '%s'.", __func__, valuestr);
		xmlFree(valuestr);
		return c_xml_RetCode_ParseError;
	}

	*code = res;

	xmlFree(valuestr);
	return c_xml_RetCode_Success;
}

int32_t c_xml_getcodefromnode(const xmlNode *node, double *code) {
	if (node == NULL || code == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): Invalid argument.", __func__);
		return c_xml_RetCode_NULL;
	}

	const char *valuestr = c_xml_getattributevalue(node, "code", TRUE);
	if (valuestr == NULL)
		return c_xml_RetCode_NotFound;

	char *end_ptr;
	double res = strtod(valuestr, &end_ptr);

	// If no conversion can be performed, strtod returns 0 and end_ptr is set to valuestr.
	if (end_ptr == valuestr) {
		logMsg(c_logger_Severity_ERROR, "%s(): Can't parse a value from string '%s'.", __func__, valuestr);
		return c_xml_RetCode_ParseError;
	}

	*code = res;
	return c_xml_RetCode_Success;
}

int32_t c_xml_getcodefromcodename(const c_xmlfile_t *file, const char *__restrict category, const char *__restrict codename, double *code) {
    if (codename == NULL) {
        logMsg(c_logger_Severity_ERROR, "%s(): Invalid argument. 'codename' cannot be NULL.", __func__);
        return c_xml_RetCode_NULL;
    }

	xmlNode *node = c_xml_getrootnode(file);
    // don't control here if ROOT is 'Definitions', we assume it.
    node = c_xml_getfirstdescendantelementfromname(node, category);
    node = c_xml_getfirstchildrenelementfromname(node, codename);

    if (node == NULL) {
        logMsg(c_logger_Severity_ERROR, "%s(): Cannot find a node named '%s/%s' in definition's XML.", __func__, category, codename);
        return c_xml_RetCode_NotFound;
    }

	int32_t ret = c_xml_getcodefromnodechild(node, code);
    if (ret != c_xml_RetCode_Success) {
        logMsg(c_logger_Severity_ERROR, "%s(): Can't get code of parameter '%s'", __func__, codename);
        return ret;
    }

    return c_xml_RetCode_Success;
}

// NOTE: we can return the pointer here since it is a member of structure 'xmlNode', which lives inside
// definitions document. So, since definitions documents lives, the pointer to the attribute name lives too.
c_use_decl_annotations const char * c_xml_getattributevaluefromcodename(const c_xmlfile_t *file, const char *__restrict category, const char *__restrict codename, const char *__restrict attrname) {
    const char *valuestr;
    xmlNode *node = NULL;

    if (codename == NULL) {
        logMsg(c_logger_Severity_ERROR, "%s(): Invalid argument. 'codename' cannot be NULL.", __func__);
        return NULL;
    }

    node = xmlDocGetRootElement(file->doc);
    // don't control here if ROOT is 'Definitions', we assume it.
    node = c_xml_getfirstdescendantelementfromname(node, category);
    node = c_xml_getfirstchildrenelementfromname(node, codename);

    if (node == NULL) {
        logMsg(c_logger_Severity_ERROR, "%s(): Cannot find a node named '%s/%s' in definition's XML.", __func__, category, codename);
        return NULL;
    }

    valuestr = c_xml_getattributevalue(node, attrname, TRUE);
    if (valuestr == NULL) {
        logMsg(c_logger_Severity_ERROR, "%s(): Node named '%s/%s' doesn't have attribute '%s' in definition's XML.", __func__, category, codename, attrname);
        return NULL;
    }

    return valuestr;
}

//void * c_xml_getfunctionpointerfromcodename(const char *codename) {
c_use_decl_annotations c_fcnhandle_t c_xml_getfunctionpointerfromcodename(const c_xmlfile_t *file, const char *codename) {
    xmlNode *node;
    const char *libname;
    int32_t ret;
    c_fcnhandle_t fcnHandle;

    node = xmlDocGetRootElement(file->doc);
    // don't control here if ROOT is 'Definitions', we assume it.
    if (node != NULL)
        node = c_xml_getfirstchildrenelementfromname(node, "Functions");
    if (node != NULL)
        node = c_xml_getfirstchildrenelementfromname(node, codename);

    if (node == NULL) {
        logMsg(c_logger_Severity_ERROR, "%s(): Cannot find a node named '%s' in definition's XML.", __func__, codename);
        return NULL;
    }

    libname = c_xml_getattributevalue(node, "libname", TRUE);
    if (libname == NULL) {
        logMsg(c_logger_Severity_ERROR, "%s(): Missing mandatory attribute 'libname' for function '%s'", __func__, codename);
        return NULL;
    }

    if (libHandle == NULL) {
        if ((ret = c_dlload(libname, &libHandle)) != MP_code_Success) {
            logMsg(c_logger_Severity_ERROR, "%s(): Library '%s' load failed with DynUtils error %"PRIi32".", __func__, libname, ret);
            return NULL;
        }
    }
    if ((ret = c_dfload(libHandle, codename, &fcnHandle)) != MP_code_Success) {
        logMsg(c_logger_Severity_ERROR, "%s(): Function '%s' load from library '%s' failed with DynUtils error %"PRIi32".", __func__, codename, libname, ret);
        return NULL;
    }

    //return (void*)fcnHandle;
    return fcnHandle;
}

void c_xml_parsercleanup(void) {
	if (libHandle == NULL)
		return;

	int32_t ret = c_dlclose(libHandle);
	if (ret != MP_code_Success) {
		logMsg(c_logger_Severity_ERROR, "%s(): Can't close library", __func__);
		return;
	}

	libHandle = NULL;
}

int c_xml_snprintf(xmlChar *buf, int len, const char *msg, ...) {
	va_list args;
	va_start(args, msg);

#if LIBXML_VERSION < 20904
	// third argument is (const char *) only since version 2.9.4
	int ret = xmlStrVPrintf(buf, len, (const xmlChar *)msg, args);
#else
	int ret = xmlStrVPrintf(buf, len, msg, args);
#endif

	va_end(args);
	return ret;
}

int c_xml_strlen(const xmlChar *str) {
	return xmlStrlen(str);
}

const char * c_xml_parserversion(void) {
	return xmlParserVersion;
}

#endif // _NOUSEXML
