#include "examples.h"
#include <time.h>

CAEN_MCA_HANDLE CAEN_MCA_EXAMPLES_OpenDeviceLocalhost(void) {
	//! [OpenDevice]
	int32_t ret;
	CAEN_MCA_HANDLE device = CAEN_MCA_OpenDevice("eth://localhost:56342", &ret, NULL);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}
	//! [OpenDevice]

	return device;
}

CAEN_MCA_HANDLE CAEN_MCA_EXAMPLES_OpenDevice(const char *hostname, int32_t *idx) {

	int32_t ret;
	// NOTE: idx can also be null, in that case the device index is not filled.
	CAEN_MCA_HANDLE device = CAEN_MCA_OpenDevice(hostname, &ret, idx);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return device;
}

void CAEN_MCA_EXAMPLES_CloseDevice(CAEN_MCA_HANDLE device) {
	//! [CloseDevice]
	CAEN_MCA_CloseDevice(device);
	//! [CloseDevice]
}

int32_t CAEN_MCA_EXAMPLES_QuitServer(CAEN_MCA_HANDLE device) {
	int32_t ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_RESTART, DATAMASK_CMD_NONE, DATAMASK_CMD_NONE);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_GetFamilyCode(CAEN_MCA_HANDLE device, CAEN_MCA_BoardFamilyCode_t *family) {
	int32_t f = 0;

	int32_t ret = CAEN_MCA_GetData(device, CAEN_MCA_DATA_BOARD_INFO, DATAMASK_BRDINFO_FAMCODE, &f);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	else {
		*family = f;
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_GetBoardInfo(CAEN_MCA_HANDLE device) {
	uint32_t channels, hvchannels, serialNum, nbits, tsample, pcbrev;
	char *modelName = calloc(MODEL_NAME_MAXLEN, sizeof(*modelName));

	int32_t ret = CAEN_MCA_GetData(
		device,
		CAEN_MCA_DATA_BOARD_INFO,
		DATAMASK_BRDINFO_MODELNAME |
		DATAMASK_BRDINFO_NCHANNELS |
		DATAMASK_BRDINFO_SERIALNUM |
		DATAMASK_BRDINFO_NHVCHANNELS |
		DATAMASK_BRDINFO_PCBREV |
		DATAMASK_BRDINFO_ADC_BIT_COUNT |
		DATAMASK_BRDINFO_TSAMPLE_PS,
		modelName,
		&channels,
		&serialNum,
		&hvchannels,
		&pcbrev,
		&nbits,
		&tsample
	);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	else {
		fprintf(stdout, "Device name: %s\n", modelName);
		fprintf(stdout, "PCB revision: %"PRIu32"\n", pcbrev);
		fprintf(stdout, "Number of input channels: %"PRIu32"\n", channels);
		fprintf(stdout, "Number of ADC bits: %"PRIu32"\n", nbits);
		fprintf(stdout, "Sample period (in picoseconds): %"PRIu32"\n", tsample);
		fprintf(stdout, "Number of HV channels: %"PRIu32"\n", channels);
		fprintf(stdout, "Serial number: %"PRIu32"\n", serialNum);
	}

	free(modelName);

	char* datetime;
	datetime = (char*)calloc(DATETIME_MAXLEN, sizeof(char));
	ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_RTCLOCK, DATAMASK_CMD_NONE, DATAMASK_CMD_CURRENT_DATETIME, datetime);
	if (ret != CAEN_MCA_RetCode_Success) {
		fprintf(stderr, "%s(): cannot get board datetime (need Hexagon/Red Eagle FW >= 0.99.24). Error: '%"PRIi32"'.\n", __func__, ret);
	}
	else {
		printf("Board datetime is %s\n", datetime);
	}

	/* ///////uncomment to update Board datetime to the local datetime
	time_t T;
	struct tm tm;
	T = time(NULL);
	tm = *localtime(&T);
	snprintf((char*)datetime, DATETIME_MAXLEN, "%04d-%02d-%02dT%02d:%02d:%02d", tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);////YYYY-MM-DDTHH:MM:SS
	ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_RTCLOCK, DATAMASK_CMD_CURRENT_DATETIME, DATAMASK_CMD_NONE, datetime);
	if (ret != CAEN_MCA_RetCode_Success) {
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}
	*/
	free(datetime);

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_GetChannelInfo(CAEN_MCA_HANDLE channel) {
	uint32_t nenergyspectra, nmcsspectra;

	//! [WrongOrder]
	int32_t ret = CAEN_MCA_GetData(
		channel,
		CAEN_MCA_DATA_CHANNEL_INFO,
		DATAMASK_CHANNELINFO_NMCSSPECTRA |		// 0x2	[Mask order does not matter, being | (bitwise or) operator commutative...]
		DATAMASK_CHANNELINFO_NENERGYSPECTRA,	// 0x1
		&nenergyspectra,						//		[Additional variadic argument order **does** matter!]
		&nmcsspectra							//
	);
	//! [WrongOrder]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
		return ret;
	}

	else {
		fprintf(stdout, "Number of activated energy spectra: %"PRIu32"\n", nenergyspectra);
		fprintf(stdout, "Number of activated MCS spectra: %"PRIu32"\n", nmcsspectra);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_DatabasePath(CAEN_MCA_HANDLE device, char *pathname) {
	//! [DatabasePath]
	char *path = pathname;
	int32_t ret;

	if (path == NULL) // null path not allowed
		ret = CAEN_MCA_RetCode_Argument;
	else {
		ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_CONFIGURATION_DB_PATH, DATAMASK_CMD_SAVE_DB_PATH, DATAMASK_CMD_NONE, path);

		if (ret != CAEN_MCA_RetCode_Success) {
			// Error
			fprintf(stderr, "%s(): failed setting database path. Error: '%"PRIi32"'.\n", __func__, ret);
		}
		else {
			// Get path back to check successful set
			strcpy(path, "");
			ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_CONFIGURATION_DB_PATH, DATAMASK_CMD_NONE, DATAMASK_CMD_SAVE_DB_PATH, path);

			if (ret != CAEN_MCA_RetCode_Success) {
				// Error
				fprintf(stderr, "%s(): failed getting database path. Error: '%"PRIi32"'.\n", __func__, ret);
			}
		}
	}

	//! [DatabasePath]

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_SaveConfiguration(CAEN_MCA_HANDLE device, const char *savename) {
	//! [SaveConfiguration]
	const char *name = savename;
	int32_t ret;
	
	if (name == NULL)
		// If no name is provided, the default name is a SQL CURRENT_TIMESTAMP with format "YYYY-MM-DD HH:MM:SS"
		// Note: no more than a save per second allowed!
		ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_CONFIGURATION_SAVE, DATAMASK_CMD_NONE, DATAMASK_CMD_NONE);
	else
		ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_CONFIGURATION_SAVE, DATAMASK_CMD_SAVE_NAME, DATAMASK_CMD_NONE, name);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}
	//! [SaveConfiguration]

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_LoadConfiguration(CAEN_MCA_HANDLE device, const char *savename) {
	//! [LoadConfiguration]
	const char *name = savename;
	int32_t ret;

	if (name == NULL)
		// If no name is provided, the most recent configuration is loaded
		ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_CONFIGURATION_LOAD, DATAMASK_CMD_NONE, DATAMASK_CMD_NONE);
	else
		ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_CONFIGURATION_LOAD, DATAMASK_CMD_SAVE_NAME, DATAMASK_CMD_NONE, name);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}
	//! [LoadConfiguration]

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_DeleteConfiguration(CAEN_MCA_HANDLE device, const char *savename) {
	//! [DeleteConfiguration]
	const char *name = savename;
	int32_t ret;

	if (name == NULL)
		// If no name is provided, all the saved configurations will be deleted
		ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_CONFIGURATION_DELETE, DATAMASK_CMD_NONE, DATAMASK_CMD_NONE);
	else
		ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_CONFIGURATION_DELETE, DATAMASK_CMD_SAVE_NAME, DATAMASK_CMD_NONE, name);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}
	//! [DeleteConfiguration]

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_ListConfigurations(CAEN_MCA_HANDLE device) {
	//! [ListConfigurations]
	uint32_t offset = 0;
	uint32_t cnt_found;
	char **savenames = malloc(CONFIGSAVE_LIST_MAXLEN * sizeof(*savenames));
	if (savenames != NULL) {
		for (int32_t i = 0; i < CONFIGSAVE_LIST_MAXLEN; i++) {
			savenames[i] = calloc(CONFIGSAVE_FULLPATH_MAXLEN, sizeof(*savenames[i]));
		}
	}

	int32_t ret = CAEN_MCA_SendCommand(
		device,
		CAEN_MCA_CMD_CONFIGURATION_LIST,
		DATAMASK_CMD_SAVE_LIST_OFFSET,
		DATAMASK_CMD_SAVE_LIST_COUNT |
		DATAMASK_CMD_SAVE_LIST_NAMES,
		offset,
		&cnt_found,
		savenames
	);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	else {
		for (uint32_t i = 0; i < cnt_found; i++) {
			// Use your results here
			fprintf(stdout, "%"PRIu32": ", i);
			fprintf(stdout, "Save name: '%s'\t", savenames[i]);
			fprintf(stdout, "\n");
		}
	}

	// Free data
	if (savenames != NULL) {
		for (int32_t i = 0; i < CONFIGSAVE_LIST_MAXLEN; i++) {
			free(savenames[i]);
		}
		free(savenames);
	}
	//! [ListConfigurations]

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_ReadRegister(CAEN_MCA_HANDLE device, uint32_t address) {
	//! [ReadRegister]
	uint32_t data;

	int32_t ret = CAEN_MCA_SendCommand(
		device,
		CAEN_MCA_CMD_REGISTER_READ,
		DATAMASK_CMD_REG_ADDR,
		DATAMASK_CMD_REG_DATA,
		address,
		&data);
	//! [ReadRegister]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	else {
		fprintf(stdout, "Address 0x%04"PRIx32": 0x%08"PRIx32"\t", address, data);
		fprintf(stdout, "\n");
	}

	return ret;
}
