#include "examples.h"

int32_t CAEN_MCA_EXAMPLES_GetParameterValue(CAEN_MCA_HANDLE handle, const char *codename, double *value) {
	//! [GetHandle]
	CAEN_MCA_HANDLE parameter = CAEN_MCA_GetChildHandleByName(handle, CAEN_MCA_HANDLE_PARAMETER, codename);
	//! [GetHandle]

	//! [GetValue]
	double pvalue;
	int32_t ret = CAEN_MCA_GetData(parameter, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, &pvalue);
	//! [GetValue]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	else {
		if (value != NULL)
			*value = pvalue;

		fprintf(stdout, "Get: %s = %f\n", codename, pvalue);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_SetParameterValue(CAEN_MCA_HANDLE handle, const char *codename, double value) {

	CAEN_MCA_HANDLE parameter = CAEN_MCA_GetChildHandleByName(handle, CAEN_MCA_HANDLE_PARAMETER, codename);

	//! [SetValue]
	double pvalue = value;
	int32_t ret = CAEN_MCA_SetData(parameter, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, pvalue);
	//! [SetValue]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	else {
		fprintf(stdout, "Set: %s = %f\n", codename, value);
	}

	CAEN_MCA_EXAMPLES_GetParameterValue(handle, codename, NULL);

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_GetParameterValueList(CAEN_MCA_HANDLE handle, const char *codename, char *value, size_t valuelen) {
	CAEN_MCA_HANDLE parameter = CAEN_MCA_GetChildHandleByName(handle, CAEN_MCA_HANDLE_PARAMETER, codename);

	//! [GetValueList]
	char *pvalue = malloc(PARAMINFO_NAME_MAXLEN * sizeof(*pvalue));
	int32_t ret = CAEN_MCA_GetData(parameter, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_CODENAME, pvalue); // No reference here!
	//! [GetValueList]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	else {
		if (value != NULL) {
			value[0] = '\0';
			strncat(value, pvalue, valuelen - 1);
		}

		fprintf(stdout, "Get: %s = %s\n", codename, pvalue);
	}
	
	free(pvalue);

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_SetParameterValueList(CAEN_MCA_HANDLE handle, const char *codename, const char *value) {

	CAEN_MCA_HANDLE parameter = CAEN_MCA_GetChildHandleByName(handle, CAEN_MCA_HANDLE_PARAMETER, codename);

	//! [SetValueList]
	const char *pvalue = value;
	int32_t ret = CAEN_MCA_SetData(parameter, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_CODENAME, pvalue);
	//! [SetValueList]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	else {
		fprintf(stdout, "Set: %s = %s\n", codename, value);
	}

	CAEN_MCA_EXAMPLES_GetParameterValue(handle, codename, NULL);

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_GetParameterInfo(CAEN_MCA_HANDLE parameter) {
	//! [GetInfo]
	char *parameter_name = calloc(PARAMINFO_NAME_MAXLEN, sizeof(*parameter_name));
	char *parameter_codename = calloc(PARAMINFO_NAME_MAXLEN, sizeof(*parameter_codename));
	uint32_t infobits;
	char *uom_name = calloc(PARAMINFO_NAME_MAXLEN, sizeof(*uom_name));
	char *uom_codename = calloc(PARAMINFO_NAME_MAXLEN, sizeof(*uom_codename));
	int32_t uom_power;
	CAEN_MCA_ParameterType_t type;
	double min, max, incr;
	uint32_t nallowed_values;
	double *allowed_values = calloc(PARAMINFO_LIST_MAXLEN, sizeof(*allowed_values));
	char **allowed_value_codenames = calloc(PARAMINFO_LIST_MAXLEN, sizeof(*allowed_values));
	char **allowed_value_names = calloc(PARAMINFO_LIST_MAXLEN, sizeof(*allowed_values));

	for (int32_t i = 0; i < PARAMINFO_LIST_MAXLEN; i++) {
		if (allowed_value_codenames != NULL) allowed_value_codenames[i] = calloc(PARAMINFO_NAME_MAXLEN, sizeof(char));
		if (allowed_value_names != NULL) allowed_value_names[i] = calloc(PARAMINFO_NAME_MAXLEN, sizeof(char));
	}

	int32_t ret = CAEN_MCA_GetData(
		parameter,
		CAEN_MCA_DATA_PARAMETER_INFO,
		DATAMASK_PARAMINFO_NAME |
		DATAMASK_PARAMINFO_CODENAME |
		DATAMASK_PARAMINFO_INFOMASK |
		DATAMASK_PARAMINFO_UOM_NAME |
		DATAMASK_PARAMINFO_UOM_CODENAME |
		DATAMASK_PARAMINFO_UOM_POWER |
		DATAMASK_PARAMINFO_TYPE |
		DATAMASK_PARAMINFO_MIN |
		DATAMASK_PARAMINFO_MAX |
		DATAMASK_PARAMINFO_INCR |
		DATAMASK_PARAMINFO_NALLOWED_VALUES |
		DATAMASK_PARAMINFO_ALLOWED_VALUES |
		DATAMASK_PARAMINFO_ALLOWED_VALUE_CODENAMES |
		DATAMASK_PARAMINFO_ALLOWED_VALUE_NAMES,
		parameter_name,
		parameter_codename,
		&infobits,
		uom_name,
		uom_codename,
		&uom_power,
		&type,
		&min,
		&max,
		&incr,
		&nallowed_values,
		allowed_values,
		allowed_value_codenames,
		allowed_value_names
	);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	else {
		fprintf(stdout, "Parameter: %s (%s)\n", parameter_codename, parameter_name);

		switch (type) {
		case CAEN_MCA_PARAMETER_TYPE_RANGE:
			fprintf(stdout, "\tMin: %f max: %f incr: %f\n", min, max, incr);
			break;
		case CAEN_MCA_PARAMETER_TYPE_LIST:
			for (uint32_t i = 0; i < nallowed_values; i++)
				fprintf(stdout, "\tAllowed value: %s [%s] (%f)\n", allowed_value_codenames[i], allowed_value_names[i], allowed_values[i]);
			break;
		default:
			fprintf(stderr, "\tUnknown parameter type.\n");
		}

		fprintf(stdout, "\tUnit of measurement: %s (power: %"PRIi32")\n", uom_name, uom_power);
	}

	for (int32_t i = 0; i < PARAMINFO_LIST_MAXLEN; i++) {
		if (allowed_value_names != NULL) free(allowed_value_names[i]);
		if (allowed_value_codenames != NULL) free(allowed_value_codenames[i]);
	}
	free(allowed_value_names);
	free(allowed_value_codenames);
	free(allowed_values);
	free(uom_codename);
	free(uom_name);
	free(parameter_codename);
	free(parameter_name);

	//! [GetInfo]

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_GetParameterCollectionInfo(CAEN_MCA_HANDLE parent_handle) {
	//! [GetSupportedParameters]
	CAEN_MCA_HANDLE collection = CAEN_MCA_GetChildHandle(parent_handle, CAEN_MCA_HANDLE_COLLECTION, CAEN_MCA_HANDLE_PARAMETER);

	//! [GetCollectionInfo]
	uint32_t collection_length = 0;
	CAEN_MCA_HANDLE *collection_handles = calloc(COLLECTION_MAXLEN, sizeof(*collection_handles));

	int32_t ret = CAEN_MCA_GetData(
		collection,
		CAEN_MCA_DATA_COLLECTION,
		DATAMASK_COLLECTION_LENGTH |
		DATAMASK_COLLECTION_HANDLES,
		&collection_length,
		collection_handles
	);
	//! [GetCollectionInfo]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	else {
		// Print results
		fprintf(stdout, "Number of parameters found: %"PRIu32"\n", collection_length);

		for (uint32_t i = 0; i < collection_length; i++) {
			int32_t type;
			int32_t index;
			char *name = malloc(HANDLE_NAME_MAXLEN * sizeof(*name));
			ret |= CAEN_MCA_GetData(
				collection_handles[i],
				CAEN_MCA_DATA_HANDLE_INFO,
				DATAMASK_HANDLE_TYPE |
				DATAMASK_HANDLE_INDEX |
				DATAMASK_HANDLE_NAME,
				&type,
				&index,
				name
			);

			if (ret != CAEN_MCA_RetCode_Success) {
				// Error
				fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
			}

			else {
				fprintf(stdout, "%"PRIu32": %s (type: %"PRIi32", index: %"PRIi32")\n", i, name, type, index);
			}

			free(name);
		}	
	}
	//! [GetSupportedParameters]

	for (uint32_t i = 0; i < collection_length; i++)
		ret |= CAEN_MCA_EXAMPLES_GetParameterInfo(collection_handles[i]);

	free(collection_handles);

	return ret;
}
