#include "examples.h"

int32_t CAEN_MCA_EXAMPLES_EnergySpectrum_GetSpectrum(CAEN_MCA_HANDLE device, int32_t channel_id, int32_t spectrum_id, uint32_t *data, uint32_t *nbins) {
	//! [GetHandle]
	CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, channel_id);
	CAEN_MCA_HANDLE spectrum = CAEN_MCA_GetChildHandle(channel, CAEN_MCA_HANDLE_ENERGYSPECTRUM, spectrum_id);
	//! [GetHandle]

	//! [GetSpectrum]
	int32_t ret = CAEN_MCA_RetCode_Success;

	// Initialize variables
	double numofbins;
	uint64_t realtime;
	uint64_t livetime;
	uint64_t deadtime;
	uint32_t overflows, underflows;
	uint64_t nentries;
	uint32_t nrois;
	uint32_t autosaveperiod;
	uint32_t stabilizer_status;
	char *filename = malloc(ENERGYSPECTRUM_FULLPATH_MAXLEN * sizeof(*filename));
	uint32_t *dataarray = malloc(ENERGYSPECTRUM_MAXLEN * sizeof(*dataarray));
	char *sample_id = malloc(SAMPLEID_MAXLEN * sizeof(*sample_id));
	char *Sdatetime = malloc(DATETIME_MAXLEN * sizeof(*Sdatetime));
	char* sdatetime = malloc(DATETIME_MAXLEN * sizeof(*sdatetime));

	// Get data
	ret |= CAEN_MCA_GetData(
		spectrum,
		CAEN_MCA_DATA_ENERGYSPECTRUM,
		DATAMASK_ENERGY_SPECTRUM_ARRAY |
		DATAMASK_ENERGY_SPECTRUM_RTIME |
		DATAMASK_ENERGY_SPECTRUM_LTIME |
		DATAMASK_ENERGY_SPECTRUM_DTIME |
		DATAMASK_ENERGY_SPECTRUM_OVERFLOW |
		DATAMASK_ENERGY_SPECTRUM_UNDERFLOW |
		DATAMASK_ENERGY_SPECTRUM_NENTRIES |
		DATAMASK_ENERGY_SPECTRUM_NROIS |
		DATAMASK_ENERGY_SPECTRUM_FILENAME |
		DATAMASK_ENERGY_SPECTRUM_AUTOSAVE_PERIOD |
		DATAMASK_ENERGY_SPECTRUM_STABILIZER_ENABLE |
		DATAMASK_ENERGY_SPECTRUM_SAMPLE_ID |
		DATAMASK_ENERGY_SPECTRUM_START_DATETIME |
		DATAMASK_ENERGY_SPECTRUM_STOP_DATETIME,
		dataarray,
		&realtime,
		&livetime,
		&deadtime,
		&overflows,
		&underflows,
		&nentries,
		&nrois,
		filename,
		&autosaveperiod,
		&stabilizer_status,
		sample_id,
		Sdatetime,
		sdatetime
	);

	// Get number of bins as parameter
	CAEN_MCA_HANDLE *parameter = CAEN_MCA_GetChildHandleByName(spectrum, CAEN_MCA_HANDLE_PARAMETER, "PARAM_ENERGY_SPECTRUM_NBINS");

	ret |= CAEN_MCA_GetData(parameter, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, &numofbins);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}
	//! [GetSpectrum]

	else {
		if (data != NULL && nbins != NULL) {
			*nbins = (uint32_t)numofbins;
			memcpy(data, dataarray, *nbins * sizeof(*data));
		}

		fprintf(stdout, "Number of bins: %"PRIu32"\n", (uint32_t)numofbins);
		fprintf(stdout, "Real time: %"PRIu64"\n", realtime);
		fprintf(stdout, "Live time: %"PRIu64"\n", livetime);
		fprintf(stdout, "Dead time: %"PRIu64"\n", deadtime);
		fprintf(stdout, "Overflows: %"PRIu32"\n", overflows);
		fprintf(stdout, "Underflows: %"PRIu32"\n", underflows);
		fprintf(stdout, "Number of entries %"PRIu64"\n", nentries);
		fprintf(stdout, "Number of activated ROIs %"PRIu32"\n", nrois);
		fprintf(stdout, "Remote filename %s\n", filename);
		fprintf(stdout, "Autosave period (in ms): %"PRIu32"\n", autosaveperiod);
		fprintf(stdout, "Sample ID %s\n", sample_id);
		fprintf(stdout, "Start acquisition datetime %s\n", Sdatetime);
		fprintf(stdout, "Stop acquisition datetime %s\n", sdatetime);

		for (uint32_t roi = 0; roi < nrois; roi++)
			CAEN_MCA_EXAMPLES_ROI_GetInfo(spectrum, roi);
	}

	free(filename);
	free(dataarray);
	free(sample_id);
	free(Sdatetime);
	free(sdatetime);

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_EnergySpectrum_ClearSpectrum(CAEN_MCA_HANDLE spectrum) {

	//! [ClearSpectrum]
	int32_t ret = CAEN_MCA_SendCommand(spectrum, CAEN_MCA_CMD_ENERGYSPECTRUM_CLEAR, DATAMASK_CMD_NONE, DATAMASK_CMD_NONE);
	//! [ClearSpectrum]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_EnergySpectrum_SetNBins(CAEN_MCA_HANDLE spectrum, uint32_t nbins) {

	//! [SetNBins]
	CAEN_MCA_HANDLE parameter = CAEN_MCA_GetChildHandleByName(spectrum, CAEN_MCA_HANDLE_PARAMETER, "PARAM_ENERGY_SPECTRUM_NBINS");

	int32_t ret = CAEN_MCA_SetData(parameter, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, (double)nbins);
	//! [SetNBins]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_EnergySpectrum_SetLLD(CAEN_MCA_HANDLE spectrum, uint32_t lld) {

	CAEN_MCA_HANDLE parameter = CAEN_MCA_GetChildHandleByName(spectrum, CAEN_MCA_HANDLE_PARAMETER, "PARAM_ENERGY_SPECTRUM_LLD");

	int32_t ret = CAEN_MCA_SetData(parameter, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, (double)lld);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_EnergySpectrum_SetULD(CAEN_MCA_HANDLE spectrum, uint32_t uld) {

	CAEN_MCA_HANDLE parameter = CAEN_MCA_GetChildHandleByName(spectrum, CAEN_MCA_HANDLE_PARAMETER, "PARAM_ENERGY_SPECTRUM_ULD");

	int32_t ret = CAEN_MCA_SetData(parameter, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, (double)uld);

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_EnergySpectrum_SetFilename(CAEN_MCA_HANDLE spectrum, const char *filename) {

	//! [SetFilename]
	int32_t ret = CAEN_MCA_SetData(spectrum, CAEN_MCA_DATA_ENERGYSPECTRUM, DATAMASK_ENERGY_SPECTRUM_FILENAME, filename);
	//! [SetFilename]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_EnergySpectrum_SetAutosavePeriod(CAEN_MCA_HANDLE spectrum, uint32_t period) {

	//! [SetPeriod]
	int32_t ret = CAEN_MCA_SetData(spectrum, CAEN_MCA_DATA_ENERGYSPECTRUM, DATAMASK_ENERGY_SPECTRUM_AUTOSAVE_PERIOD, period);
	//! [SetPeriod]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_GainStabilizer_EnDis(CAEN_MCA_HANDLE spectrum, uint32_t enable) {

	//! [GSEnableDisable]
	int32_t ret = CAEN_MCA_SetData(spectrum, CAEN_MCA_DATA_ENERGYSPECTRUM, DATAMASK_ENERGY_SPECTRUM_STABILIZER_ENABLE, enable);
	//! [GSEnableDisable]

	if (ret != CAEN_MCA_RetCode_Success) {
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_GainStabilizer_Reset(CAEN_MCA_HANDLE spectrum) {


	//! [GSReset]
	int32_t ret = CAEN_MCA_SendCommand(spectrum, CAEN_MCA_CMD_GAIN_STABILIZER_RESET, DATAMASK_CMD_NONE, DATAMASK_CMD_NONE);
	//! [GSReset]

	if (ret != CAEN_MCA_RetCode_Success) {
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_GainStabilizer_SetROIH(CAEN_MCA_HANDLE spectrum, uint32_t roih) {

	//! [GSSetHigh]
	CAEN_MCA_HANDLE parameter = CAEN_MCA_GetChildHandleByName(spectrum, CAEN_MCA_HANDLE_PARAMETER, "PARAM_GAIN_STABILIZER_ROIH");

	int32_t ret = CAEN_MCA_SetData(parameter, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, (double)roih);
	//! [GSSetHigh]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_GainStabilizer_SetROIL(CAEN_MCA_HANDLE spectrum, uint32_t roil) {

	//! [GSSetLow]
	CAEN_MCA_HANDLE parameter = CAEN_MCA_GetChildHandleByName(spectrum, CAEN_MCA_HANDLE_PARAMETER, "PARAM_GAIN_STABILIZER_ROIL");

	int32_t ret = CAEN_MCA_SetData(parameter, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, (double)roil);
	//! [GSSetLow]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return ret;
}

int32_t CAEN_MCA_EXAMPLES_GainStabilizer_SetTimeout(CAEN_MCA_HANDLE spectrum, uint32_t time) {

	//! [GSSetTime]
	CAEN_MCA_HANDLE parameter = CAEN_MCA_GetChildHandleByName(spectrum, CAEN_MCA_HANDLE_PARAMETER, "PARAM_GAIN_STABILIZER_TIME");

	int32_t ret = CAEN_MCA_SetData(parameter, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, (double)time);
	//! [GSSetTime]

	if (ret != CAEN_MCA_RetCode_Success) {
		// Error
		fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
	}

	return ret;
}