#ifndef _WIN32
#define _DEFAULT_SOURCE
#endif

#include <CAENMCA.h>
#include <stdlib.h>
#include <stdio.h>
#include <inttypes.h>

#include "examples.h"

#ifdef _WIN32 // Windows
#include <windows.h>
#include <conio.h>
#define popen(command, type)		_popen(command, type)

#else // Linux
#include <unistd.h>
#include <termios.h>

static char _getch(){
    char buf=0;
    struct termios old={0};
    fflush(stdout);
    if(tcgetattr(0, &old)<0)
        perror("tcsetattr()");
    old.c_lflag&=~ICANON;
    old.c_lflag&=~ECHO;
    old.c_cc[VMIN]=1;
    old.c_cc[VTIME]=0;
    if(tcsetattr(0, TCSANOW, &old)<0)
        perror("tcsetattr ICANON");
    if(read(0,&buf,1)<0)
        perror("read()");
    old.c_lflag|=ICANON;
    old.c_lflag|=ECHO;
    if(tcsetattr(0, TCSADRAIN, &old)<0)
        perror ("tcsetattr ~ICANON");
    printf("%c\n",buf);
    return buf;
 }
#endif

#ifdef _WIN32
#define c_sleep(nms)				Sleep(nms)
#else
#define c_sleep(nms)				usleep((int32_t)((nms) * 1000))
#endif

#ifdef _WIN32
#define GNUPLOT "pgnuplot.exe"
#else
#define GNUPLOT "gnuplot"
#endif

static void _printMenu(void) {
	printf("i : Get board info\n");
	printf("N : Test handles\n");
	printf("A : Set Acquisition Mode\n");
	printf("T : Set threshold\n");
	printf("S : Start acquisition\n");
	printf("s : Stop acquisition\n");
	printf("E : Enable/Disable Auto-trigger\n");
	printf("K : Start Waveform\n");
	printf("k : Stop Waveform\n");
	printf("h : Get Energy Spectrum\n");
	printf("H : Set Energy Spectrum\n");
	printf("c : Clear Energy Spectrum\n");
	printf("p : Plot Energy Spectrum from file\n");
	printf("G : Enable/Disable Gain stabilizer\n");
	printf("g : Configure Gain stabilizer\n");
	printf("l : Get List Mode\n");
	printf("L : Set List Mode\n");
	printf("r : Read Register\n");
	printf("w : Get Waveform\n");
	printf("Z : Save configuration\n");
	printf("z : Load last saved configuration\n");
	printf("C : List saved configurations\n");
	printf("I : Get Parameter Info\n");
	printf("P : Start Param AutoSearch\n");
	printf("U : Stop Param AutoSearch\n");
	printf("t : Get Channel Collection Info\n");
	printf("d : Discover devices\n");
	printf("D : Delete all saved configurations\n");
	printf("R : Get HV Range info\n");
	printf("V : Set HV status\n");
	printf("m : Get MCS Spectrum\n");
	printf("M : Set MCS Spectrum\n");
	printf("b : Send MCS Sweep signal\n");
	printf("B : Clear MCS Spectrum\n");
	printf("u : Plot MCS Spectrum from file\n");
	printf("X : Set DeltaT Spectrum\n");
	printf("x : Get DeltaT Spectrum and plot data\n");
	printf("F : Clear DeltaT Spectrum\n");
	printf("a : Change configurations database path\n");
	printf("W : Configure channel start mode\n");
	printf("q : Quit client\n");
	printf("Q : Quit server and client\n\n\n");
}

static void _printSpectrumMenu(void) {
	printf("b : Nbins\n");
	printf("m : Min\n");
	printf("M : Max\n");
	printf("F : filename\n");
	printf("r : ROIL\n");
	printf("R : ROIH\n");
	printf("G : Enable/Disable gain stabilizer\n");
	printf("w : Stabilizer ROIL\n");
	printf("W : Stabilizer ROIH\n");
	printf("T : Stabilizer update time\n");
	printf("q : Ready\n\n\n");
}

static void _printListMenu(void) {
	printf("e : Enable/disable\n");
	printf("m : Mode\n");
	printf("f : Filename\n");
	printf("s : Data to save\n");
	printf("q : Ready\n\n\n");
}

static void _printMCSMenu(void) {
	printf("C : MCS Counting Mode\n");
	printf("A : MCS Acquisition Mode\n");
	printf("b : Number of bins\n");
	printf("p : Preset passes\n");
	printf("d : Dwell time\n");
	printf("D : Dwell mode\n");
	printf("S : Sweep mode\n");
	printf("q : Ready\n\n\n");
}

static void _printDTSpectrumMenu(void) {
	printf("e : Enable/disable DT spectrum\n");
	printf("C : Set Reference channel\n");
	printf("B : Set Number of bins\n");
	printf("t : Set time interval for plot\n");
	printf("q : Ready\n\n\n");
}

static void SetSpectrumParameters(CAEN_MCA_HANDLE device, int32_t channel_id, int32_t spectrum_id) {
	bool quit = false;

	CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, channel_id);
	CAEN_MCA_HANDLE spectrum = CAEN_MCA_GetChildHandle(channel, CAEN_MCA_HANDLE_ENERGYSPECTRUM, spectrum_id);

	do {
		_printSpectrumMenu();
		switch (_getch()) {
		case 'q':
		{
			quit = true;
			break;
		}
		case 'b':
		{
			uint32_t value = 0;
			printf("Enter new NBins\t");
			scanf("%"SCNu32"", &value);
			CAEN_MCA_EXAMPLES_EnergySpectrum_SetNBins(spectrum, value);
			break;
		}
		case 'm':
		{
			uint32_t value = 0;
			printf("Enter new LLD\t");
			scanf("%"SCNu32"", &value);
			CAEN_MCA_EXAMPLES_EnergySpectrum_SetLLD(spectrum, value);
			break;
		}
		case 'M':
		{
			uint32_t value = 0;
			printf("Enter new ULD\t");
			scanf("%"SCNu32"", &value);
			CAEN_MCA_EXAMPLES_EnergySpectrum_SetULD(spectrum, value);
			break;
		}
		case 'F':
		{
			char *new_filename = malloc(ENERGYSPECTRUM_FULLPATH_MAXLEN * sizeof(*new_filename));
			printf("Enter new Filename\t");
			scanf("%s", new_filename);
			CAEN_MCA_EXAMPLES_EnergySpectrum_SetFilename(spectrum, new_filename);
			free(new_filename);
			break;
		}
		case 'r':
		{
			uint32_t value = 0;
			printf("Enter new ROIL\t");
			scanf("%"SCNu32"", &value);
			CAEN_MCA_HANDLE roi = CAEN_MCA_GetChildHandle(spectrum, CAEN_MCA_HANDLE_ROI, 0);
			CAEN_MCA_EXAMPLES_ROI_SetLow(roi, value);
			break;
		}
		case 'R':
		{
			uint32_t value = 0;
			printf("Enter new ROIH\t");
			scanf("%"SCNu32"", &value);
			CAEN_MCA_HANDLE roi = CAEN_MCA_GetChildHandle(spectrum, CAEN_MCA_HANDLE_ROI, 0);
			CAEN_MCA_EXAMPLES_ROI_SetHigh(roi, value);
			break;
		}
		case 'G':
		{
			uint32_t new_status;
			printf("Type 1 to enable the gain stabilizer, 0 to disable it.\n");
			scanf("%"SCNu32"", &new_status);
			CAEN_MCA_EXAMPLES_GainStabilizer_EnDis(spectrum, new_status);
			printf("New current status of gain stbilizer for spectrum %"PRIi32" of channel %"PRIi32" is %"PRIu32"\n", spectrum_id, channel_id, new_status);
			break;
		}
		case 'W':
		{
			uint32_t value = 0;
			printf("Enter new Stabilizer ROIH\t");
			scanf("%"SCNu32"", &value);
			CAEN_MCA_EXAMPLES_GainStabilizer_SetROIH(spectrum, value);
			break;
		}
		case 'w':
		{
			uint32_t value = 0;
			printf("Enter new Stabilizer ROIL\t");
			scanf("%"SCNu32"", &value);
			CAEN_MCA_EXAMPLES_GainStabilizer_SetROIL(spectrum, value);
			break;
		}
		case 'T':
		{
			uint32_t value = 0;
			printf("Enter new Stabilizer update time\t");
			scanf("%"SCNu32"", &value);
			CAEN_MCA_EXAMPLES_GainStabilizer_SetTimeout(spectrum, value);
			break;
		}
		}
	} while (!quit);
}

static void SetListModeParameters(CAEN_MCA_HANDLE channel) {
	bool quit = false;

	do {
		_printListMenu();
		switch (_getch()) {
		case 'q':
		{
			quit = true;
			break;
		}
		case 'e':
		{
			uint32_t enable;
			printf("Press 1 to enable list mode, 0 to disable it.\t");
			scanf("%"SCNu32"", &enable);
			CAEN_MCA_EXAMPLES_EnableLists(channel, (bool)enable);
			break;
		}
		case 'm':
		{
			int32_t mode;
			printf("Enter new List Mode:\n");
			printf("0: List mode file Ascii\n");
			printf("1: List mode file binary\n");
			printf("2: List mode memory\n");
			scanf("%"SCNi32"", &mode);
			CAEN_MCA_EXAMPLES_Lists_SetSaveMode(channel, mode);

			uint32_t nevts;
			printf("Enter number of events to store in list buffer:\n");
			scanf("%"SCNu32"", &nevts);
			CAEN_MCA_EXAMPLES_Lists_SetMaxNumberOfEvents(channel, nevts);

			break;
		}
		case 'f':
		{
			char *new_filename = malloc(LISTS_FULLPATH_MAXLEN * sizeof(*new_filename));
			printf("Enter new Filename\t");
			scanf("%s", new_filename);
			CAEN_MCA_EXAMPLES_Lists_SetFilename(channel, new_filename);
			free(new_filename);
			break;
		}
		case 's':
		{
			uint32_t timetag, energy, extras;
			printf("Timetag: (0/1)\n");
			scanf("%"SCNu32"", &timetag);

			printf("Energy: (0/1)\n");
			scanf("%"SCNu32"", &energy);

			printf("Extras: (0/1)\n");
			scanf("%"SCNu32"", &extras);

			CAEN_MCA_EXAMPLES_Lists_SetData(channel, (bool)timetag, (bool)energy, (bool)extras);
			break;
		}
		}
		
	} while (!quit);
}

static void SetMCSParameters(CAEN_MCA_HANDLE device, int32_t channel_id, int32_t spectrum_id) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	uint32_t value = 0;
	double dwell_value = 0;
	uint32_t enable = 0;
	bool enable_b = false;
	bool quit = false;

	CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, channel_id);
	CAEN_MCA_HANDLE mcs_spectrum = CAEN_MCA_GetChildHandle(channel, CAEN_MCA_HANDLE_MCSSPECTRUM, spectrum_id);

		do {
			_printMCSMenu();
			switch (_getch()) {
			case 'q':
			{
				quit = true;
				break;
			}
			case 'b':
			{
				printf("Enter new NBins (maximum value 32767)\t");
				scanf("%"SCNu32"", &value);
				ret = CAEN_MCA_EXAMPLES_MCS_SetNBins(mcs_spectrum, value);
				if (ret != CAEN_MCA_RetCode_Success)
					return;
				break;
			}
			case 'd':
			{
				printf("Enter new Dwell Time (s)\t");
				scanf("%lf", &dwell_value);
				ret = CAEN_MCA_EXAMPLES_MCS_SetDwellTime(mcs_spectrum, dwell_value*1e6); //in us 
				if (ret != CAEN_MCA_RetCode_Success)
					return;
				break;
			}
			case 'D':
			{
				printf("Press 1 to enable SW Dwell, 0 to disable it.\t");
				scanf("%"SCNu32"", &enable);
				enable_b = enable;
				ret = CAEN_MCA_EXAMPLES_MCS_EnableSWDwell(mcs_spectrum, enable_b);
				if (ret != CAEN_MCA_RetCode_Success)
					return;

				printf("Press 1 to enable HW Dwell, 0 to disable it.\t");
				scanf("%"SCNu32"", &enable);
				enable_b = enable;
				ret = CAEN_MCA_EXAMPLES_MCS_EnableHWDwell(mcs_spectrum, enable_b);
				if (ret != CAEN_MCA_RetCode_Success)
					return;
				break;
			}
			case 'S':
			{
				printf("Press 1 to enable SW Sweep, 0 to disable it.\t");
				scanf("%"SCNu32"", &enable);
				enable_b = enable;
				ret = CAEN_MCA_EXAMPLES_MCS_EnableSWSweep(mcs_spectrum, enable_b);
				if (ret != CAEN_MCA_RetCode_Success)
					return;

				printf("Press 1 to enable HW Sweep, 0 to disable it.\t");
				scanf("%"SCNu32"", &enable);
				enable_b = enable;
				ret = CAEN_MCA_EXAMPLES_MCS_EnableHWSweep(mcs_spectrum, enable_b);
				if (ret != CAEN_MCA_RetCode_Success)
					return;
				break;
			}
			case 'C':
			{
				int32_t mode;
				printf("Enter new MCS Mode:\n");
				printf("0: MCS on ICR\n");
				printf("1: MCS on external signal\n");
				printf("2: MCS on SCA\n");
				scanf("%"SCNd32"", &mode);
				ret = CAEN_MCA_EXAMPLES_MCS_SetMode(mcs_spectrum, mode);
				if (ret != CAEN_MCA_RetCode_Success)
					return;
				break;
			}
			case 'A':
			{
				int32_t mode;
				printf("Enter new MCS Acquisition Mode:\n");
				printf("0: REPLACE\n");
				printf("1: SUM\n");
				printf("2: REPLACE AND SUM\n");
				scanf("%"SCNd32"", &mode);
				ret = CAEN_MCA_EXAMPLES_MCS_SetAcqMode(mcs_spectrum, mode);
				if (ret != CAEN_MCA_RetCode_Success)
					return;
				break;
			}
			case 'p':
			{
				printf("Enter new number of preset passes:\t");
				scanf("%"SCNu32"", &value);
				ret = CAEN_MCA_EXAMPLES_MCS_SetPresetPasses(mcs_spectrum, value);
				if (ret != CAEN_MCA_RetCode_Success)
					return;
				break;
			}
		}

	} while (!quit);
}


static void SetDTSpectrumParameters(CAEN_MCA_HANDLE device, int32_t spectrum_id) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	uint32_t enable = 0;
	bool quit = false;
	bool enable_dt = false;

	CAEN_MCA_HANDLE dt_spectrum = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_DTSPECTRUM, spectrum_id);

	do {
		_printDTSpectrumMenu();
		switch (_getch()) {
		case 'q':
		{
			quit = true;
			break;
		}
		case 'e':
		{
			uint32_t restore = false;
			printf("Press 1 to enable Delta T spectrum, 0 to disable it.\t");
			scanf("%"SCNu32"", &enable);
			enable_dt = (bool)enable;
			CAEN_MCA_EXAMPLES_EnableDTSpectrum(device, spectrum_id, enable_dt);

			if (!enable_dt) {				
				printf("Do you want to restore the previous coincidence settings? 1=Yes, 0=No.\t");
				scanf("%"SCNu32"", &restore);
				if (restore) {
					ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_CONFIGURATION_LOAD, DATAMASK_CMD_SAVE_NAME, DATAMASK_CMD_NONE, "mem_config");
					if (ret != CAEN_MCA_RetCode_Success) {
						fprintf(stderr, "%s(): failed. Error: '%"PRIi32"'.\n", __func__, ret);
						return;
					}
				}
				ret = CAEN_MCA_SendCommand(device, CAEN_MCA_CMD_CONFIGURATION_DELETE, DATAMASK_CMD_SAVE_NAME, DATAMASK_CMD_NONE, "mem_config");
			}
			break;
		}
		case 'C':
		{
			uint32_t ch;
			printf("Select reference channel: 0 for ch 0, 1 for ch 1\n");
			scanf("%"SCNu32"", &ch);
			CAEN_MCA_EXAMPLES_DTSpectrum_SetRefCh(dt_spectrum, ch);
			break;
		}
		case 'B':
		{
			uint32_t nbins;
			printf("Enter new number of bins (maximum allowed value is 4096)\n");
			scanf("%"SCNu32"", &nbins);
			CAEN_MCA_EXAMPLES_DTSpectrum_SetNBins(dt_spectrum, nbins);
			break;
		}
		case 't':
		{
			uint32_t step;
			printf("Enter new time step (ns): \n");
			scanf("%"SCNu32"", &step);
			CAEN_MCA_EXAMPLES_DTSpectrum_SetDTStep(dt_spectrum, step);
			break;
		}
		
		}

	} while (!quit);
}

static int32_t SetAcquisitionMode(CAEN_MCA_HANDLE device) {
	int32_t ch, mode;

	printf("Select channel: 0 for ch 0, 1 for ch 1\n");
	scanf("%"SCNd32"", &ch);

	CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);

	uint64_t rt = 0;
	uint64_t lt = 0;
	uint64_t counts_total = 0;
	uint64_t counts_roi = 0;

	printf("Select Acq Stop Mode:\n");
	printf("0: STOP MODE MANUAL\n");
	printf("1: STOP MODE REALTIME (seconds)\n");
	printf("2: STOP MODE LIVETIME (seconds)\n");
	printf("3: STOP MODE TOTAL COUNTS\n");
	printf("4: STOP MODE ROI COUNTS\n\n");

	scanf("%"SCNd32"", &mode);

	if (mode != 0) {
		printf("Enter new Stop Value\t");
		uint64_t stop_value = 0;
		scanf("%"SCNu64"", &stop_value);		

		switch (mode) {
		case 1:
			rt = stop_value * 1000000000; //value in ns
			break;
		case 2:
			lt = stop_value * 1000000000; //value in ns
			break;
		case 3:
			counts_total = stop_value;
			break;
		case 4:
			counts_roi = stop_value;
			break;
		}
	}

	return CAEN_MCA_EXAMPLES_SetStopCriteria(channel, rt, lt, counts_total, counts_roi);
}

static void SetChStartMode(CAEN_MCA_HANDLE channel, int32_t mode, int32_t pol) {
	int32_t ret = 0;

	switch (mode) {
		case 0:
			ret = CAEN_MCA_EXAMPLES_SetParameterValueList(channel, "PARAM_CH_STARTMODE", "START_MODE_GPIO_ON_LEVEL");
			break;
		case 1:
			ret = CAEN_MCA_EXAMPLES_SetParameterValueList(channel, "PARAM_CH_STARTMODE", "START_MODE_GPIO_ON_TRG");
			break;
		case 2:
			ret = CAEN_MCA_EXAMPLES_SetParameterValueList(channel, "PARAM_CH_STARTMODE", "START_MODE_SW");
			break;
	}
	if (ret != CAEN_MCA_RetCode_Success)
		printf("Error setting start mode\n");
	switch (pol) {
		case 0:
			ret = CAEN_MCA_EXAMPLES_SetParameterValueList(channel, "PARAM_CH_START_SIN_POLARITY", "POLARITY_POSITIVE");
			break;
		case 1:
			ret = CAEN_MCA_EXAMPLES_SetParameterValueList(channel, "PARAM_CH_START_SIN_POLARITY", "POLARITY_NEGATIVE");
			break;
		default:
			break;
	}	
	if (ret != CAEN_MCA_RetCode_Success)
		printf("Error setting GPIO signal polarity\n");
}

int main(int argc, char *argv[]) {
	//init gnuplot pipe
	FILE* gnuplot_spectra = popen(GNUPLOT, "w");
	FILE* gnuplot_waveforms = popen(GNUPLOT, "w");
	FILE* gnuplot_dt_spectra = popen(GNUPLOT, "w");
	
	if (gnuplot_spectra == NULL || gnuplot_waveforms == NULL || gnuplot_dt_spectra == NULL) {
		fprintf(stderr, "Can't open " GNUPLOT ". Quitting.\n");
		return CAEN_MCA_RetCode_Generic;
	}
	
	fprintf(gnuplot_spectra, "set xlabel 'ADC channels'\n");
	fprintf(gnuplot_spectra, "set ylabel 'Counts'\n");
	fflush(gnuplot_spectra);

	fprintf(gnuplot_waveforms, "set yrange [0:32000]\n");
	fprintf(gnuplot_waveforms, "set xlabel 'sampling units (x10 ns)'\n");
	fprintf(gnuplot_waveforms, "set ylabel 'ADC counts'\n");

	fprintf(gnuplot_dt_spectra, "set xlabel 'time difference (DT steps unit)'\n");
	fprintf(gnuplot_dt_spectra, "set ylabel 'Number of events'\n");

	uint32_t *dataArray = calloc(ENERGYSPECTRUM_MAXLEN, sizeof(*dataArray));

	int32_t ret = 0;

	// path examples:
	//   - path = "eth://172.16.0.2:56342"										(Hexagon/Red Eagle via ethernet on port 56342)
	//   - path = "usb://localhost/connection?serial=20"						(connect to Hexagon/Red Eagle SN 20)
	//   - path = "usb://localhost/connection?link=0"							(BETA: direct usb connection to non-standalone boards via USB link 0)
	//   - path = "usb://localhost/connection?link=0&vmeaddress=0x32100000"		(BETA: usb connection with vme bridge to board at link 0 and vme address 0x32100000)
	//   - path = "opt://localhost/connection?link=0"							(BETA: direct optical connection to non-standalone boards via OPT link 0)
	//   - path = "opt://localhost/connection?link=0&vmeaddress=0x32100000"		(BETA: optical connection with vme bridge to board at link 0 and vme address 0x32100000)
	const char *path = "eth://172.16.0.2:56342";
	if (argc > 1) {
		path = argv[1];
	}

	int32_t devidx;
	CAEN_MCA_HANDLE device = CAEN_MCA_EXAMPLES_OpenDevice(path, &devidx);

	if (device == NULL)
		goto QuitProgram;

	CAEN_MCA_BoardFamilyCode_t famcode = CAEN_MCA_FAMILY_CODE_UNKNOWN;
	CAEN_MCA_EXAMPLES_GetFamilyCode(device, &famcode);

	uint32_t nBins = 0;
	bool auto_trg_enabled = false;
	double dwell_time=1;

	bool quit = false;
	do {
		_printMenu();
		switch (_getch()) {
		case 'Q':
		{
			printf("quitting server...\n");
			CAEN_MCA_EXAMPLES_QuitServer(device);
			//no break
		}
		case 'q':
		{
			printf("quitting test...\n");
			quit = true;
			break;
		}
		case 'N':
		{
			CAEN_MCA_EXAMPLES_GetHandles(devidx);
			break;
		}
		case 'R':
		{
			int32_t ch, r;
			printf("Select channel: 0 for ch 0, 1 for ch 1\t\n");
			scanf("%"SCNd32"", &ch);

			//get hvch handle
			CAEN_MCA_HANDLE hvchannel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_HVCHANNEL, ch);

			//get param handle from hvchannel
			ret = CAEN_MCA_EXAMPLES_HVChannel_GetExtendedStatus(hvchannel);
			if (ret != CAEN_MCA_RetCode_Success) break;

			printf("Select HV range: 0 for HPGe, 1 for PMT, 2 for SD\t\n");
			scanf("%"SCNd32"", &r);

			ret = CAEN_MCA_EXAMPLES_HVRange_GetInfo(hvchannel, r);
			if (ret != CAEN_MCA_RetCode_Success) break;

			// get vmon from currently active range
			ret = CAEN_MCA_EXAMPLES_HVRange_GetVMon(hvchannel);
			if (ret != CAEN_MCA_RetCode_Success) break;

			break;
		}
		case 'i':
		{
			printf("getting info board...\n");
			CAEN_MCA_EXAMPLES_GetBoardInfo(device);
			break;
		}
		case 't':
		{
			printf("getting channel 0 parameter collection info...\n");
			CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, 0);
			CAEN_MCA_EXAMPLES_GetParameterCollectionInfo(channel);
			break;
		}
		case 'd':
		{
			printf("discovering devices on your network...\n");
			CAEN_MCA_EXAMPLES_DiscoverDevices();
			break;
		}
		case 'T':
		{
			int32_t ch;
			int32_t value;
			printf("setting threshold...\n");
			printf("Select channel: 0 for ch 0, 1 for ch 1\n");
			scanf("%"SCNd32"", &ch);
			printf("Select value:\n");
			scanf("%"SCNd32"", &value);
			CAEN_MCA_HANDLE channel_0 = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
			CAEN_MCA_EXAMPLES_SetParameterValue(channel_0, "PARAM_CH_THRESHOLD", (double)value);
			break;
		}
		case 'I':
		{
			printf("getting parameter info...\n");
			CAEN_MCA_HANDLE parameter = CAEN_MCA_GetChildHandleByName(device, CAEN_MCA_HANDLE_PARAMETER, "PARAM_RECLEN");
			CAEN_MCA_EXAMPLES_GetParameterInfo(parameter);
			break;
		}
		case 'a':
		{
			// NOTE: string error handling not implemented to keep the code as much clear as possible
			char *newpath = calloc(CONFIGSAVE_FULLPATH_MAXLEN, sizeof(*newpath)); // should check allocation
			printf("Enter new database path (including filename): ");
			scanf("%s", newpath); // should protect overflows and check errors
			printf("changing database path to '%s'...\n", newpath);
			CAEN_MCA_EXAMPLES_DatabasePath(device, newpath);
			free(newpath);
			break;
		}
		case 'Z':
		{
			printf("saving configuration...\n");
			CAEN_MCA_EXAMPLES_SaveConfiguration(device, NULL);
			break;
		}
		case 'z':
		{
			printf("loading saved configuration...\n");
			CAEN_MCA_EXAMPLES_LoadConfiguration(device, NULL);
			break;
		}
		case 'C':
		{
			printf("getting saved configuration list...\n");
			CAEN_MCA_EXAMPLES_ListConfigurations(device);
			break;
		}
		case 'D':
		{
			printf("deleting saved configuration...\n");
			CAEN_MCA_EXAMPLES_DeleteConfiguration(device, NULL);
			break;
		}
		case 'K':
		{
			printf("enabling waveforms...\n");
			CAEN_MCA_EXAMPLES_EnableWaveforms(device, true);
			break;
		}
		case 'k':
		{
			printf("stopping waveforms...\n");
			CAEN_MCA_EXAMPLES_EnableWaveforms(device, false);
			break;
		}
		case 'A':
		{
			ret = SetAcquisitionMode(device);
			if (ret != CAEN_MCA_RetCode_Success) {
				printf("set acq mode error ret = %"PRIi32".\n", ret);
				break;
			}
			printf("New Acquisition mode set.\n\n");
			break;
		}
		case 'S':
		{
			printf("starting...\n");
			if (famcode == CAEN_MCA_FAMILY_CODE_XXHEX) {
				uint32_t chmask;
				printf("Select channel: 0 for ch 0, 1 for ch 1, 2 for both channels\t\n");
				scanf("%"SCNu32"", &chmask);
				chmask = (1 << chmask);
				if (chmask == 4) {//start acq for all channels
					CAEN_MCA_EXAMPLES_StartAcquisition(device);
				}
				else { //single channel start acq
					for (uint32_t ch = 0; ch < 2; ch++) {
						if ((chmask & (1 << ch)) != 0) {
							CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
							CAEN_MCA_EXAMPLES_StartAcquisition(channel);
						}
					}
				}
			}
			else {
				CAEN_MCA_EXAMPLES_StartAcquisition(device);
			}
			break;
		}
		case 's':
		{
			printf("stopping...\n");
			if (famcode == CAEN_MCA_FAMILY_CODE_XXHEX) {
				uint32_t chmask;
				printf("Select channel: 0 for ch 0, 1 for ch 1, 2 for both channels\t\n");
				scanf("%"SCNu32"", &chmask);
				chmask = (1 << chmask);
				if (chmask == 4) {//start acq for all channels
					CAEN_MCA_EXAMPLES_StopAcquisition(device);
				}
				else {//single channel start acq
					for (uint32_t ch = 0; ch < 2; ch++) {
						if ((chmask & (1 << ch)) != 0) {
							CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
							CAEN_MCA_EXAMPLES_StopAcquisition(channel);
						}
					}
				}
			}
			else {
				CAEN_MCA_EXAMPLES_StopAcquisition(device);
			}
			break;
		}
		case 'E':
		{
			double value;
			int32_t ch;
			if (famcode == CAEN_MCA_FAMILY_CODE_XXHEX) {
				printf("Select channel: 0 for ch 0, 1 for ch 1\t\n");
				scanf("%"SCNd32"", &ch);
				if (auto_trg_enabled) {
					printf("disabling auto-trigger...\n");
					value = 0;
					auto_trg_enabled = false;
				}
				else {
					printf("enabling auto-trigger...\n");
					value = 300;
					auto_trg_enabled = true;
				}
			}
			else {
				break;
			}
			CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
			CAEN_MCA_EXAMPLES_SetParameterValue(channel, "PARAM_CH_AUTO_TRG_TIMEOUT", (double)value);
			break;
		}
		case 'c':
		{
			int32_t ch;
			printf("Select channel: 0 for ch 0, 1 for ch 1\t\n");
			scanf("%"SCNd32"", &ch);
			printf("clearing spectrum...\n");
			CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
			CAEN_MCA_HANDLE spectrum = CAEN_MCA_GetChildHandle(channel, CAEN_MCA_HANDLE_ENERGYSPECTRUM, 0);
			CAEN_MCA_EXAMPLES_EnergySpectrum_ClearSpectrum(spectrum);
			break;
		}
		case 'h':
		{
			int32_t ch;
			printf("Select channel: 0 for ch 0, 1 for ch 1\t\n");
			scanf("%"SCNd32"", &ch);
			printf("getting spectrum...\n");
			CAEN_MCA_EXAMPLES_EnergySpectrum_GetSpectrum(device, ch, 0, dataArray, &nBins);
			break;
		}
		case 'H':
		{
			int32_t ch;
			printf("Select channel: 0 for ch 0, 1 for ch 1\t\n");
			scanf("%"SCNd32"", &ch);
			printf("editing spectrum...\n");
			// get a channel from device
			SetSpectrumParameters(device, ch, 0);
			break;
		}
		case 'L':
		{
			int32_t ch;
			printf("Select channel: 0 for ch 0, 1 for ch 1\t\n");
			scanf("%"SCNd32"", &ch);
			printf("setting list mode...\n");
			// get a channel from device
			CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
			SetListModeParameters(channel);
			break;
		}
		case 'l':
		{
			int32_t ch;
			printf("Select channel: 0 for ch 0, 1 for ch 1\t\n");
			scanf("%"SCNd32"", &ch);
			printf("getting list config...\n");
			CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
			CAEN_MCA_EXAMPLES_GetLists(channel);
			break;
		}
		case 'V':
		{
			int32_t hvch, status;
			printf("Select HV channel: 0 for ch 0, 1 for ch 1\t\n");
			scanf("%"SCNd32"", &hvch);
			printf("Select status: 0 for disable, 1 for enable\t\n");
			scanf("%"SCNd32"", &status);
			bool s = status;
			printf("setting status...\n");
			CAEN_MCA_HANDLE hvchannel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_HVCHANNEL, hvch);
			CAEN_MCA_EXAMPLES_HVChannel_SetStatus(hvchannel, s);
			break;
		}
		case 'w':
		{
			int32_t ch;
			uint32_t length;
			printf("Select channel: 0 for ch 0, 1 for ch 1\t\n");
			scanf("%"SCNd32"", &ch);

			int16_t *aw1 = calloc(WAVEFORM_MAXLEN, sizeof(*aw1));
			int16_t *aw2 = calloc(WAVEFORM_MAXLEN, sizeof(*aw2));
			uint8_t *dw1 = calloc(WAVEFORM_MAXLEN, sizeof(*dw1));
			uint8_t *dw2 = calloc(WAVEFORM_MAXLEN, sizeof(*dw2));

			if (aw1 == NULL || aw2 == NULL || dw1 == NULL || dw2 == NULL)
				break;

			ret = CAEN_MCA_EXAMPLES_GetWaveforms(device, ch, aw1, aw2, dw1, dw2, &length);

			if (ret == CAEN_MCA_RetCode_Success) {
				FILE* datafile = fopen("Wave_to_plot.txt", "w+");
				if (datafile == NULL) {
					fprintf(stderr, "Can't open file Wave_to_plot.txt for writing. Do you have write permission on current working directory?\n");
					ret = CAEN_MCA_RetCode_Generic;
				}
				else {
					for (uint32_t i = 0; i < length; i += 10) {
						fprintf(datafile, "%"PRIi16"\t", aw1[i]);
						fprintf(datafile, "%"PRIi16"\t", aw2[i]);
						uint32_t dw1ext = 100U * dw1[i];
						uint32_t dw2ext = 100U * dw2[i];
						fprintf(datafile, "%"PRIu32"\t", dw1ext);
						fprintf(datafile, "%"PRIu32"\n", dw2ext);
					}

					fclose(datafile);
					fprintf(gnuplot_waveforms, "plot 'Wave_to_plot.txt' u 1 t 'Trace1' w step, 'Wave_to_plot.txt' u 2 t 'Trace2' w step, 'Wave_to_plot.txt' u 3 t 'DTrace1' w step, 'Wave_to_plot.txt' u 4 t 'DTrace2' w step\n");
					fflush(gnuplot_waveforms);
				}
			}

			free(aw1);
			free(aw2);
			free(dw1);
			free(dw2);

			break;
		}
		case 'p':
		{
			printf("plotting spectrum...\n");

			FILE* plotdatafile = fopen("Data_to_plot.txt", "w+");
			if (plotdatafile == NULL) {
				fprintf(stderr, "Can't open file Data_to_plot.txt for writing. Do you have write permission on current working directory?\n");
				ret = CAEN_MCA_RetCode_Generic;
			}
			else {
				for (uint32_t i = 0; i < nBins; i++)
					fprintf(plotdatafile, "%"PRIu32"\n", dataArray[i]);

				fclose(plotdatafile);
				fprintf(gnuplot_spectra, "set xlabel 'ADC channels'\n");
				fprintf(gnuplot_spectra, "set ylabel 'Counts'\n");
				fprintf(gnuplot_spectra, "plot '%s' with step\n", "Data_to_plot.txt");
				fflush(gnuplot_spectra);
			}

			break;
		}
		case 'u':
		{
			printf("plotting MCS spectrum...\n");

			FILE* plotdatafile = fopen("MCS_Data.txt", "w+");
			if (plotdatafile == NULL) {
				fprintf(stderr, "Can't open file MCS_Data.txt for writing. Do you have write permission on current working directory?\n");
				ret = CAEN_MCA_RetCode_Generic;
			}
			else {
				for (uint32_t i = 0; i < nBins; i++)
					fprintf(plotdatafile, "%"PRIu32" %"PRIu32"\n", i, dataArray[i]);

				fclose(plotdatafile);
				fprintf(gnuplot_spectra, "set xlabel 'Time (s)'\n");
				fprintf(gnuplot_spectra, "set ylabel 'Counts'\n");
				fprintf(gnuplot_spectra, "plot '%s' using ($1*%f):($2) with step\n", "MCS_Data.txt", (float)dwell_time / 1e6);
				fflush(gnuplot_spectra);
			}

			break;
		}
		case 'x':
		{
			CAEN_MCA_HANDLE dt_spectrum = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_DTSPECTRUM, 0);
			CAEN_MCA_EXAMPLES_DTSpectrum_Get(dt_spectrum);

			fprintf(gnuplot_dt_spectra, "set xlabel 'time difference (DT steps unit)'\n");
			fprintf(gnuplot_dt_spectra, "set ylabel 'Number of events'\n");
			fprintf(gnuplot_dt_spectra, "plot '%s' using ($1):($2) with step\n", "DTdata_to_plot.txt");
			fflush(gnuplot_dt_spectra);
			break;
		}
		case 'F':
		{
			printf("clearing DT spectrum...\n");
			CAEN_MCA_HANDLE dt_spectrum = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_DTSPECTRUM, 0);
			CAEN_MCA_EXAMPLES_DTSpectrum_ClearSpectrum(dt_spectrum);
			break;
		}
		case 'P':
		{
			if (famcode != CAEN_MCA_FAMILY_CODE_XXHEX)
				break;

			int32_t ch;
			uint32_t enable;
			printf("Select channel: 0 for ch 0, 1 for ch 1\n");
			scanf("%"SCNd32"", &ch);
			printf("Start DCO search? (0 for NO, 1 YES\n");
			scanf("%"SCNu32"", &enable);
			uint32_t enable_dco = !!enable;
			printf("Start Thr search? (0 for NO, 1 YES\n");
			scanf("%"SCNu32"", &enable);
			uint32_t enable_thr = !!enable;
			printf("Start Tau search? (0 for NO, 1 YES\n");
			scanf("%"SCNu32"", &enable);
			uint32_t enable_tau = !!enable;

			//if all searches are enabled the board starts with DCO, then with Thr and then with Tau
			//here for preliminary tests three loops follow
			//TODO: enable stop command (tested in bGamma)
			CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
			ret = CAEN_MCA_SendCommand(channel, CAEN_MCA_CMD_PARAM_AUTOSET_START, DATAMASK_CMD_AUTOSET_DCO | DATAMASK_CMD_AUTOSET_THR | DATAMASK_CMD_AUTOSET_TAU, DATAMASK_CMD_NONE, enable_dco, enable_thr, enable_tau);
			if (ret != CAEN_MCA_RetCode_Success) {
				printf("start autosearch error ret = %"PRIi32".\n", ret);
				break;
			}

			double found_DCO = 0, found_Thr = 0, found_Tau = 0, found_value = 0, error= 0;
			if (enable_dco) {
				CAEN_MCA_HANDLE parameter_DCO = CAEN_MCA_GetChildHandleByName(device, CAEN_MCA_HANDLE_PARAMETER, "PARAM_AUTOSET_DCO_FOUND");
				CAEN_MCA_HANDLE parameter_DCO_err = CAEN_MCA_GetChildHandleByName(device, CAEN_MCA_HANDLE_PARAMETER, "PARAM_AUTOSET_DCO_FAILURE");
				for (;;) {
					//check if the value has been found
					ret = CAEN_MCA_GetData(parameter_DCO, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, &found_DCO);
					if (ret != CAEN_MCA_RetCode_Success) {
						printf("found_DCO read error ret = %"PRIi32".\n", ret);
						break;
					}
					if (found_DCO) {
						parameter_DCO = CAEN_MCA_GetChildHandleByName(device, CAEN_MCA_HANDLE_PARAMETER, "PARAM_AUTOSET_DCO_VALUE");
						ret = CAEN_MCA_GetData(parameter_DCO, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, &found_value);
						if (ret != CAEN_MCA_RetCode_Success) {
							printf("found_value read error ret = %"PRIi32".\n", ret);
							break;
						}
						printf("Found DCO value %d\n", (int)found_value);
						break;
					}
					//check if the reasearch failed
					ret = CAEN_MCA_GetData(parameter_DCO_err, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, &error);
					if (ret != CAEN_MCA_RetCode_Success) {
						printf("found_DCO_err read error ret = %"PRIi32".\n", ret);
						break;
					}
					if(error) {
						printf("DCO Autoset Failure.\n");
						break;
					}
					c_sleep(100);
				}
			}
			if (enable_thr) {
				CAEN_MCA_HANDLE parameter_thr= CAEN_MCA_GetChildHandleByName(device, CAEN_MCA_HANDLE_PARAMETER, "PARAM_AUTOSET_THR_FOUND");
				CAEN_MCA_HANDLE parameter_thr_err = CAEN_MCA_GetChildHandleByName(device, CAEN_MCA_HANDLE_PARAMETER, "PARAM_AUTOSET_THR_FAILURE");
				for (;;) {
					//check if the value has been found
					ret = CAEN_MCA_GetData(parameter_thr, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, &found_Thr);
					if (ret != CAEN_MCA_RetCode_Success) {
						printf("found_Thr read error ret = %"PRIi32".\n", ret);
						break;
					}
					if (found_Thr) {
						parameter_thr = CAEN_MCA_GetChildHandleByName(device, CAEN_MCA_HANDLE_PARAMETER, "PARAM_AUTOSET_THR_VALUE");
						ret = CAEN_MCA_GetData(parameter_thr, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, &found_value);
						if (ret != CAEN_MCA_RetCode_Success) {
							printf("found_value read error ret = %"PRIi32".\n", ret);
							break;
						}
						printf("Found Thr value %d\n", (int)found_value);
						break;
					}
					//check if the reasearch failed
					ret = CAEN_MCA_GetData(parameter_thr_err, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, &error);
					if (ret != CAEN_MCA_RetCode_Success) {
						printf("found_thr_err read error ret = %"PRIi32".\n", ret);
						break;
					}
					if (error) {
						printf("Thr Autoset Failure.\n");
						break;
					}
					c_sleep(100);
				}
			}
			if (enable_tau) {
				CAEN_MCA_HANDLE parameter_tau = CAEN_MCA_GetChildHandleByName(device, CAEN_MCA_HANDLE_PARAMETER, "PARAM_AUTOSET_TAU_FOUND");
				CAEN_MCA_HANDLE parameter_tau_err = CAEN_MCA_GetChildHandleByName(device, CAEN_MCA_HANDLE_PARAMETER, "PARAM_AUTOSET_TAU_FAILURE");
				for (;;) {
					//check if the value has been found
					ret = CAEN_MCA_GetData(parameter_tau, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, &found_Tau);
					if (ret != CAEN_MCA_RetCode_Success) {
						printf("found_Tau read error ret = %"PRIi32".\n", ret);
						break;
					}
					if (found_Tau) {
						parameter_tau = CAEN_MCA_GetChildHandleByName(device, CAEN_MCA_HANDLE_PARAMETER, "PARAM_AUTOSET_TAU_VALUE");
						ret = CAEN_MCA_GetData(parameter_tau, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, &found_value);
						if (ret != CAEN_MCA_RetCode_Success) {
							printf("found_value read error ret = %"PRIi32".\n", ret);
							break;
						}
						printf("Found Tau value %"PRIi32"\n", (int)found_value);
						break;
					}
					//check if the reasearch failed
					ret = CAEN_MCA_GetData(parameter_tau_err, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, &error);
					if (ret != CAEN_MCA_RetCode_Success) {
						printf("found_tau_err read error ret = %"PRIi32".\n", ret);
						break;
					}
					if (error) {
						printf("Tau Autoset Failure.\n");
						break;
					}
					c_sleep(100);
				}
			}

			break;
		}
		case 'U':
		{
			int32_t ch;
			uint32_t enable;

			printf("Select channel: 0 for ch 0, 1 for ch 1\n");
			scanf("%"SCNd32"", &ch);
			printf("Stop DCO search? (0 for NO, 1 YES\n");
			scanf("%"SCNu32"", &enable);
			uint32_t enable_dco = !!enable;
			printf("Stop Thr search? (0 for NO, 1 YES\n");
			scanf("%"SCNu32"", &enable);
			uint32_t enable_thr = !!enable;
			printf("Stop Tau search? (0 for NO, 1 YES\n");
			scanf("%"SCNu32"", &enable);
			uint32_t enable_tau = !!enable;

			CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
			ret = CAEN_MCA_SendCommand(channel, CAEN_MCA_CMD_PARAM_AUTOSET_STOP, DATAMASK_CMD_AUTOSET_DCO | DATAMASK_CMD_AUTOSET_THR | DATAMASK_CMD_AUTOSET_TAU, DATAMASK_CMD_NONE, enable_dco, enable_thr, enable_tau);
			if (ret != CAEN_MCA_RetCode_Success) {
				printf("stop autosearch error ret = %"PRIi32".\n", ret);
				break;
			}

			break;
		}
		case 'r':
		{
			printf("reading registers...\n");
			CAEN_MCA_EXAMPLES_ReadRegister(device, UINT32_C(0x8000)); //board configuration
			CAEN_MCA_EXAMPLES_ReadRegister(device, UINT32_C(0x8100)); //acq control
			CAEN_MCA_EXAMPLES_ReadRegister(device, UINT32_C(0x8104)); //acq status
			break;
		}
		case 'm':
		{
			int32_t ch;
			CAEN_MCA_MCS_Spectrum_Status_t status;
			uint32_t current_pass;

			printf("Select channel: 0 for ch 0, 1 for ch 1\n");
			scanf("%"SCNd32"", &ch);
			printf("getting MCS spectrum...\n");

			ret = CAEN_MCA_EXAMPLES_MCS_GetSpectrum(device, ch, 0, dataArray, &nBins, &status, &current_pass);
			if (ret != CAEN_MCA_RetCode_Success) {
				printf("Errors getting MCS spectrum for channel %"PRIi32" = %"PRIi32".\n",ch, ret);
				break;
			}

			switch (status) {
			case CAEN_MCA_MCS_SPE_ACQUIRING:
				printf("MCS spectrum is \"ACQUIRING\"\n\n");
				break;
			case CAEN_MCA_MCS_SPE_STOPPED:
				printf("MCS spectrum is \"STOPPED\"\n\n");
				break;
			case CAEN_MCA_MCS_SPE_NOT_STARTED:
				printf("MCS spectrum is \"NOT_STARTED\"\n\n");
				break;
			case CAEN_MCA_MCS_SPE_COMPLETED:
				printf("MCS spectrum is \"COMPLETED\"\n\n");
				break;
			}

			printf("Current pass number is %"PRIu32"\n", current_pass);

			//update dwell time value
			CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
			CAEN_MCA_HANDLE mcs_spectrum = CAEN_MCA_GetChildHandle(channel, CAEN_MCA_HANDLE_MCSSPECTRUM, 0);
			ret = CAEN_MCA_EXAMPLES_MCS_GetDwellTime(mcs_spectrum, &dwell_time);

			if (ret != CAEN_MCA_RetCode_Success) {
				printf("Errors getting MCS dwell time channel %"PRIi32" = %"PRIi32".\n", ch, ret);
				break;
			}
			break;
		}
		case 'M':
		{
			int32_t ch;
			printf("Select channel: 0 for ch 0, 1 for ch 1\t\n");
			scanf("%"SCNd32"", &ch);
			printf("editing MCS spectrum...\n");
			// get a channel from device
			SetMCSParameters(device, ch, 0);
			break;
		}
		case 'b':
		{
			int32_t ch;
			printf("Select channel: 0 for ch 0, 1 for ch 1\t\n");
			scanf("%"SCNd32"", &ch);
			CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
			CAEN_MCA_HANDLE mcs_spectrum = CAEN_MCA_GetChildHandle(channel, CAEN_MCA_HANDLE_MCSSPECTRUM, 0);
			CAEN_MCA_EXAMPLES_MCS_SendSweepSignal(mcs_spectrum);
			break;
		}
		case 'B':
		{
			int32_t ch;
			printf("Select channel: 0 for ch 0, 1 for ch 1\t\n");
			scanf("%"SCNd32"", &ch);
			printf("clearing MCS spectrum...\n");
			CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
			CAEN_MCA_HANDLE mcs_spectrum = CAEN_MCA_GetChildHandle(channel, CAEN_MCA_HANDLE_MCSSPECTRUM, 0);
			CAEN_MCA_EXAMPLES_MCS_ClearSpectrum(mcs_spectrum);
			break;
		}
		case 'X':
		{
			printf("editing Delta T spectrum...\n");
			// get a channel from device
			SetDTSpectrumParameters(device, 0);
			break;
		}
		case 'W':
		{
			int32_t ch, mode, pol;
			printf("Select channel: 0 for ch 0, 1 for ch 1\t\n");
			scanf("%"SCNd32"", &ch);
			// get a channel from device
			CAEN_MCA_HANDLE channel = CAEN_MCA_GetChildHandle(device, CAEN_MCA_HANDLE_CHANNEL, ch);
			printf("Select start mode: 0 for START_MODE GPIO_ON_LEVEL, 1 for START_MODE GPIO_ON_TRIGGER, 2 for START_MODE_SOFTWARE\t\n");
			scanf("%"SCNd32"", &mode);
			if (mode == 2) {
				pol = -1;
			}
			else {
				printf("Select ext. signal polarity: 0 for POLARITY_POSITIVE, 1 for POLARITY_NEGATIVE\t\n");
				scanf("%"SCNd32"", &pol);
			}
			SetChStartMode(channel, mode, pol);
			break;
		}
		default:
			break;
		}
	} while (!quit);

QuitProgram:
	if (dataArray != NULL)
		free(dataArray);
	CAEN_MCA_EXAMPLES_CloseDevice(device);
	return ret;
}

