/******************************************************************************
*
*	CAEN SpA - Front End Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*	This file is part of CAEN MCA which is released under
*	GNU Lesser General Public License. See file doc/README.md
*	or go to https://www.gnu.org/licenses/ for full license details.
*
***************************************************************************//*!
*
*	\file		CAENMCA.h
*	\brief		CAEN MCA API
*	\author		Giovanni Cerretani, Lara Palla, Francesco Pepe
*	\version	0.99.11
*	\date		2023
*	\copyright	GNU Lesser General Public License
*
******************************************************************************/
#ifndef _CAEN_MCA_LIB_H
#define _CAEN_MCA_LIB_H

#include <CAENMCATypes.h>

#ifdef __cplusplus
extern "C" {
#endif

/*!
* \defgroup VersionMacros Version macros
* \brief Macros to define the library vesion.
* \{ */
#define CAEN_MCA_VERSION_MAJOR		0																														//!< Major version
#define CAEN_MCA_VERSION_MINOR		99																														//!< Minor version
#define CAEN_MCA_VERSION_PATCH		10																														//!< Patch version
#define CAEN_MCA_VERSION			CAEN_MCA_STR(CAEN_MCA_VERSION_MAJOR) "." CAEN_MCA_STR(CAEN_MCA_VERSION_MINOR) "." CAEN_MCA_STR(CAEN_MCA_VERSION_PATCH)	//!< The version string as Major.Minor.Patch
#define CAEN_MCA_VERSION_NUMBER		((CAEN_MCA_VERSION_MAJOR) * 10000 + (CAEN_MCA_VERSION_MINOR) * 100 + (CAEN_MCA_VERSION_PATCH))							//!< The version number: for example version 1.2.3 gives 10203
/*! \} */

/*!
* \defgroup Functions API
* \brief Application programming interface.
*/

/*!
* \defgroup FunctionsVaList API with variable argument list
* \brief Application programming interface with variable argument list.
*/

/*!
* Connect to a device. The device should be closed using CAEN_MCA_CloseDevice().
* \param[in]		path		URL to device in the form \c "\<scheme\>://\<domain\>[:\<port\>][/connection?\<parameters\>]":
*                                 \arg \c scheme (required): can be \c "eth" (ethernet) \c "usb" (usb) and \c "opt" (optical)
*                                 \arg \c domain (required): host to connect (can be IP address or hostname)
*                                 \arg \c port (optional): port to use for connection. Default is 56342. Should be used only for Hexagon boards.
*                                 \arg \c parameters (optional): list of parameters ampersand-separated. Supported parameters are \c "link=<number>", \c "conetnode=<number>", \c "vmeaddress=<number>" and \c "serial=<number>". The latter is supported only for Hexagon boards. Except for \c "eth" scheme, one and only one between \c "serial" and \c "link" must be specified.
* \param[out]		retcode		::CAEN_MCA_RetCode_Success (0) in case of success. Error codes specified in #CAEN_MCA_RetCode_t. Can be `NULL`.
* \param[out]		index		the index of the opened device. Can be used as index to retrieve the returned handle later using CAEN_MCA_GetChildHandle(). Can be `NULL`.
* \return			Handle of the device. `NULL` in case of error.
* \ingroup			Functions
*/
CAEN_MCA_DLLAPI CAEN_MCA_HANDLE CAEN_MCA_API CAEN_MCA_OpenDevice(const char *path, int32_t *retcode, int32_t *index);

/*!
* Close the connection to the device opened with CAEN_MCA_OpenDevice(). Undefined behaviour when called twice on the same handle.
* \param[in]		handle		the handle returned by CAEN_MCA_OpenDevice(). If `NULL`, no operation is performed.
* \ingroup			Functions
*/
CAEN_MCA_DLLAPI void CAEN_MCA_API CAEN_MCA_CloseDevice(CAEN_MCA_HANDLE handle);

/*!
* Get an ancestor handle. Iterate over the ancestor of a given handle and return the first handle of the requested type.
* \param[in]		handle		the handle.
* \param[in]		handleType	the ancestor handle type. \sa ::DATAMASK_HANDLE_TYPE
* \return			Handle of the ancestor. `NULL` in case of error.
* \ingroup Functions
*/
CAEN_MCA_DLLAPI CAEN_MCA_HANDLE CAEN_MCA_API CAEN_MCA_GetAncestorHandle(CAEN_MCA_HANDLE handle, CAEN_MCA_HandleType_t handleType);

/*!
* Get a child handle.
* \param[in]		handle		the parent handle. Can be `NULL` if `handleType` is ::CAEN_MCA_HANDLE_LIBRARY or ::CAEN_MCA_HANDLE_DEVICE.
* \param[in]		handleType	the child handle type. \sa ::DATAMASK_HANDLE_TYPE
* \param[in]		index		the index of the child, except `handleType` is: \arg ::CAEN_MCA_HANDLE_COLLECTION: ::CAEN_MCA_HandleType_t of the collection returned. \sa ::DATAMASK_HANDLE_NAME
* \return			Handle of the child. `NULL` in case of error.
* \ingroup Functions
*/
CAEN_MCA_DLLAPI CAEN_MCA_HANDLE CAEN_MCA_API CAEN_MCA_GetChildHandle(CAEN_MCA_HANDLE handle, CAEN_MCA_HandleType_t handleType, int32_t index);

/*!
* Get a child handle.
* \param[in]		handle		the parent handle. Can be `NULL` if `handleType` is ::CAEN_MCA_HANDLE_LIBRARY or ::CAEN_MCA_HANDLE_DEVICE.
* \param[in]		handleType	the child handle type. ::CAEN_MCA_HANDLE_COLLECTION not supported. \sa ::DATAMASK_HANDLE_TYPE
* \param[in]		name		the name of the child. It is the literal version of the index (e.g. string "1" for index 1), except `handleType` is: \arg ::CAEN_MCA_HANDLE_LIBRARY: `NULL` pointer \arg ::CAEN_MCA_HANDLE_DEVICE: the hostname set in CAEN_MCA_OpenDevice() \arg ::CAEN_MCA_HANDLE_PARAMETER: the codename \sa ::DATAMASK_HANDLE_NAME
* \return			Handle of the child. `NULL` in case of error.
* \ingroup Functions
*/
CAEN_MCA_DLLAPI CAEN_MCA_HANDLE CAEN_MCA_API CAEN_MCA_GetChildHandleByName(CAEN_MCA_HANDLE handle, CAEN_MCA_HandleType_t handleType, const char *name);

/*!
* Get Data. For the variable argument list version see CAEN_MCA_GetDataV().
* \param[in]		handle		the handle.
* \param[in]		dataType	the type of data.
* \param[in]		dataMask	a bit mask for the given `dataType`. See also \ref DataMask.
* \param[out]		...			a pointer to variable for each bit set in dataMask, with the specified type. See also  \ref variadic-argument-order and \ref variadic-argument-type.
* \return			::CAEN_MCA_RetCode_Success (0) in case of success. Error codes specified in #CAEN_MCA_RetCode_t.
* \ingroup Functions
* \sa CAEN_MCA_SetData()
*/
CAEN_MCA_DLLAPI int32_t CAEN_MCA_API CAEN_MCA_GetData(CAEN_MCA_HANDLE handle, CAEN_MCA_DataType_t dataType, uint64_t dataMask, ...);

/*!
* Get Data. Identical to CAEN_MCA_GetData(), but using using variable argument list.
* \param[in]		handle		the handle.
* \param[in]		dataType	the type of data.
* \param[in]		dataMask	a bit mask for the given `dataType`. See also \ref DataMask.
* \param[out]		args		a variable arguments list initialized with va_start, containing a pointer to variable for each bit set in dataMask, with the specified type. See also  \ref variadic-argument-order and \ref variadic-argument-type.
* \return			::CAEN_MCA_RetCode_Success (0) in case of success. Error codes specified in #CAEN_MCA_RetCode_t.
* \ingroup FunctionsVaList
* \sa CAEN_MCA_SetDataV()
*/
CAEN_MCA_DLLAPI int32_t CAEN_MCA_API CAEN_MCA_GetDataV(CAEN_MCA_HANDLE handle, CAEN_MCA_DataType_t dataType, uint64_t dataMask, va_list args);

/*!
* Set Data. For the variable argument list version see CAEN_MCA_SetDataV().
* \param[in]		handle		the handle.
* \param[in]		dataType	the type of data.
* \param[in]		dataMask	a bit mask for the given `dataType`. See also \ref DataMask.
* \param[in]		...			a variable for each bit set in dataMask, with the specified type. See also \ref variadic-argument-order and \ref variadic-argument-type.
* \return			::CAEN_MCA_RetCode_Success (0) in case of success. Error codes specified in #CAEN_MCA_RetCode_t.
* \ingroup Functions
* \sa CAEN_MCA_GetData()
*/
CAEN_MCA_DLLAPI int32_t CAEN_MCA_API CAEN_MCA_SetData(CAEN_MCA_HANDLE handle, CAEN_MCA_DataType_t dataType, uint64_t dataMask, ...);

/*!
* Set Data. Identical to CAEN_MCA_SetData(), but using using variable argument list.
* \param[in]		handle		the handle.
* \param[in]		dataType	the type of data.
* \param[in]		dataMask	a bit mask for the given `dataType`. See also \ref DataMask.
* \param[in]		args		a variable arguments list initialized with va_start, containing a variable for each bit set in dataMask, with the specified type. See also \ref variadic-argument-order and \ref variadic-argument-type.
* \return			::CAEN_MCA_RetCode_Success (0) in case of success. Error codes specified in #CAEN_MCA_RetCode_t.
* \ingroup FunctionsVaList
* \sa CAEN_MCA_GetDataV()
*/
CAEN_MCA_DLLAPI int32_t CAEN_MCA_API CAEN_MCA_SetDataV(CAEN_MCA_HANDLE handle, CAEN_MCA_DataType_t dataType, uint64_t dataMask, va_list args);

/*!
* Send Command. For the variable argument list version see CAEN_MCA_SendCommandV().
* \param[in]		handle		the handle.
* \param[in]		cmdType		the type of command.
* \param[in]		cmdMaskIn 	a bit mask of input variables for the given `cmdType`. See also \ref CommandMask.
* \param[in]		cmdMaskOut 	a bit mask of output variables for the given `cmdType`. See also \ref CommandMask.
* \param[in,out]	...			a variable for each bit set first in `cmdMaskIn` and then `cmdMaskOut`, with the specified type. See also  \ref variadic-argument-order and \ref variadic-argument-type.
* \return			::CAEN_MCA_RetCode_Success (0) in case of success. Error codes specified in #CAEN_MCA_RetCode_t.
* \ingroup Functions
*/
CAEN_MCA_DLLAPI int32_t CAEN_MCA_API CAEN_MCA_SendCommand(CAEN_MCA_HANDLE handle, CAEN_MCA_CommandType_t cmdType, uint64_t cmdMaskIn, uint64_t cmdMaskOut, ...);

/*!
* Send Command. Identical to CAEN_MCA_SendCommand(), but using using variable argument list.
* \param[in]		handle		the handle.
* \param[in]		cmdType		the type of command.
* \param[in]		cmdMaskIn 	a bit mask of input variables for the given `cmdType`. See also \ref CommandMask.
* \param[in]		cmdMaskOut 	a bit mask of output variables for the given `cmdType`. See also \ref CommandMask.
* \param[in,out]	args		a variable arguments list initialized with va_start, containing a variable for each bit set first in `cmdMaskIn` and then `cmdMaskOut`, with the specified type. See also  \ref variadic-argument-order and \ref variadic-argument-type.
* \return			::CAEN_MCA_RetCode_Success (0) in case of success. Error codes specified in #CAEN_MCA_RetCode_t.
* \ingroup FunctionsVaList
*/
CAEN_MCA_DLLAPI int32_t CAEN_MCA_API CAEN_MCA_SendCommandV(CAEN_MCA_HANDLE handle, CAEN_MCA_CommandType_t cmdType, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);

/*!
* Wait for events.
* \warning			Not yet implemented.
* \param[in]		handle		the handle.
* \param[out]		eventHandle	the type of command.
* \return			::CAEN_MCA_RetCode_Success (0) in case of success. Error codes specified in #CAEN_MCA_RetCode_t.
* \ingroup Functions
*/
CAEN_MCA_DLLAPI int32_t CAEN_MCA_API CAEN_MCA_WaitEvent(CAEN_MCA_HANDLE handle, CAEN_MCA_HANDLE *eventHandle);

/*!
* \example examples.h						Definitions of example functions
* \example examples_discovery.c				Device discovery
* \example examples_board_control.c			Board control
* \example examples_parameters.c			Parameters
* \example examples_acquisition_control.c	Acquisition control
* \example examples_spectrum.c				Spectra
* \example examples_roi.c					ROIs
* \example examples_waveforms.c				Wafevorms
* \example examples_hv.c					High-Voltage
* \example examples_handles.c				Handles
* \example examples_lists.c					Lists
* \example examples_mcs.c					MCS
*/

#ifdef __cplusplus
}
#endif

#endif // _CAEN_MCA_LIB_H
