/******************************************************************************
*
*	CAEN SpA - Front End Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*	This file is part of CAEN MCA which is released under
*	GNU Lesser General Public License. See file doc/README.md
*	or go to https://www.gnu.org/licenses/ for full license details.
*
***************************************************************************//*!
*
*	\file		CAENMCATypes.h
*	\brief		Definitions and types
*	\author		Giovanni Cerretani, Lara Palla, Francesco Pepe
*	\version	0.99.11
*	\date		2023
*	\copyright	GNU Lesser General Public License
*
******************************************************************************/
#ifndef _CAEN_MCA_TYPES_H
#define _CAEN_MCA_TYPES_H

#include <stdint.h>
#include <stdarg.h>

/*!
* \defgroup ExportMacros Export macros
* \brief Functions are exported in the DLL only if the `_CAENMCA_EXPORT` macro is defined by the preprocessor
* \pre	Windows only.
* \{
* \def CAEN_MCA_API
* \brief Calling convention (Windows only)
* \def CAEN_MCA_DLLAPI
* \brief Storage-class attributes (Windows only)
*/
#ifdef _WIN32 // Windows
#define CAEN_MCA_API				__cdecl
#ifdef _CAEN_MCA_EXPORT
#define CAEN_MCA_DLLAPI				__declspec(dllexport)
#else
#define CAEN_MCA_DLLAPI				__declspec(dllimport)
#endif
#else // Linux
#define CAEN_MCA_API
#define CAEN_MCA_DLLAPI				__attribute__((visibility("default")))
#endif

#if defined(_CAEN_MCA_SERVER)
#define _MCALIB_SYMBOL_DLLAPI_ extern
#else
// Export case. Var declared as extern and exported.
// Important: there are differences in C and C++ storage duration for
// global declarations if no storage-class specifier is used. They
// are known as "tentative definitions" in C, but such thing does
// not exist in C++, where they are both declaration and definitions.
// So, we need to specify it extern, or add extern "C" in front of each
// variable.
// Surprisingly, in C++ `extern "C" int x;` (this case) is equivalent to
// `extern "C" { extern int x; }` (declaration and not definition) and is
// different from `extern "C" { int x; }`, a declaration and definition.
// See:
// - https://en.cppreference.com/w/c/language/extern
// - https://en.cppreference.com/w/cpp/language/language_linkage
// - https://stackoverflow.com/q/18544249/3287591
#define _MCALIB_SYMBOL_DLLAPI_ extern CAEN_MCA_DLLAPI
#endif
/*! \} */

//! Reference to an object
typedef void* CAEN_MCA_HANDLE;

/*!
* \defgroup StringMacros String utility macros
* \brief To convert macro to string.
* \{ */
#define _CAEN_MCA_STR_HELPER(S)						#S
#define CAEN_MCA_STR(S)								_CAEN_MCA_STR_HELPER(S)
/*! \} */

/********************************************************
*                   DATA TYPES                          *
********************************************************/
//!  Handle type definitions
typedef enum {
	CAEN_MCA_HANDLE_PARENT = -1,				//!< Current handle's parent. To be used with CAEN_MCA_GetAncestorHandle() to directly get the parent handle. \remark Special handle.
	CAEN_MCA_HANDLE_LIBRARY = 0,				//!< Library. \children ::CAEN_MCA_HANDLE_DEVICE
	CAEN_MCA_HANDLE_DEVICE = 1,					//!< Device. \children ::CAEN_MCA_HANDLE_PARAMETER, ::CAEN_MCA_HANDLE_CHANNEL, ::CAEN_MCA_HANDLE_HVCHANNEL, ::CAEN_MCA_HANDLE_LVDSGROUP, ::CAEN_MCA_HANDLE_TRACE, ::CAEN_MCA_HANDLE_MONOUT, ::CAEN_MCA_HANDLE_DTSPECTRUM
	CAEN_MCA_HANDLE_CHANNEL = 2,				//!< Channel. \children ::CAEN_MCA_HANDLE_PARAMETER, ::CAEN_MCA_HANDLE_ENERGYSPECTRUM, ::CAEN_MCA_HANDLE_MCSSPECTRUM
	CAEN_MCA_HANDLE_HVCHANNEL = 3,				//!< HV Channel \children ::CAEN_MCA_HANDLE_PARAMETER, ::CAEN_MCA_HANDLE_HVRANGE
	CAEN_MCA_HANDLE_HVRANGE = 4,				//!< HV Range \children ::CAEN_MCA_HANDLE_PARAMETER
	CAEN_MCA_HANDLE_LVDSGROUP = 5,				//!< LVDS Group \children ::CAEN_MCA_HANDLE_PARAMETER
	CAEN_MCA_HANDLE_TRACE = 6,					//!< Trace \children ::CAEN_MCA_HANDLE_PARAMETER
	CAEN_MCA_HANDLE_PARAMETER = 7,				//!< Parameter \children ::CAEN_MCA_HANDLE_PARAMETER
	CAEN_MCA_HANDLE_COLLECTION = 8,				//!< Handle Collection \remark Special handle.
	CAEN_MCA_HANDLE_ENERGYSPECTRUM = 9,			//!< Energy Spectrum \children ::CAEN_MCA_HANDLE_PARAMETER, ::CAEN_MCA_HANDLE_ROI
	CAEN_MCA_HANDLE_MCSSPECTRUM = 10,			//!< MCS Spectrum \children ::CAEN_MCA_HANDLE_PARAMETER
	CAEN_MCA_HANDLE_ROI = 11,					//!< ROI \children ::CAEN_MCA_HANDLE_PARAMETER
	CAEN_MCA_HANDLE_EVENT = 12,					//!< Board-related event (e.g: reaction to a command) \warning Currently unused.
	CAEN_MCA_HANDLE_TRIGGER = 13,				//!< Single-trigger event (list mode event) \warning Currently unused.
	CAEN_MCA_HANDLE_VALUE = 14,					//!< Parameter Value \warning Currently unused.
	CAEN_MCA_HANDLE_MONOUT = 15,				//!< Monitor Out \children ::CAEN_MCA_HANDLE_PARAMETER
	CAEN_MCA_HANDLE_DTSPECTRUM = 16,			//!< &Delta;T Spectrum \children ::CAEN_MCA_HANDLE_PARAMETER
	CAEN_MCA_HANDLE_REMOVABLE_COLLECTION = 17,	//!< Handle Collection where elements can be removed \remark Special handle.
} CAEN_MCA_HandleType_t;

//! Command type definitions
typedef enum {
	CAEN_MCA_CMD_ACQ_START = 1,					//!< Start Acquisition \remark On ::CAEN_MCA_HANDLE_DEVICE (board) or ::CAEN_MCA_HANDLE_CHANNEL (single channel) \sa NoCommandDataMask
	CAEN_MCA_CMD_ACQ_STOP = 2,					//!< Stop Acquisition \remark On ::CAEN_MCA_HANDLE_DEVICE (board) or ::CAEN_MCA_HANDLE_CHANNEL (single channel) \sa NoCommandDataMask
	CAEN_MCA_CMD_ENERGYSPECTRUM_CLEAR = 3,		//!< Clear Energy Spectrum \remark On ::CAEN_MCA_HANDLE_ENERGYSPECTRUM \sa NoCommandDataMask
	CAEN_MCA_CMD_REGISTER_READ = 4,				//!< Read Register \remark On ::CAEN_MCA_HANDLE_DEVICE \sa RegisterCommandDataMask
	CAEN_MCA_CMD_REGISTER_WRITE = 5,			//!< Write Register \remark On ::CAEN_MCA_HANDLE_DEVICE \sa RegisterCommandDataMask
	CAEN_MCA_CMD_RESTART = 6,					//!< Restart the server \remark On ::CAEN_MCA_HANDLE_DEVICE \sa NoCommandDataMask
	CAEN_MCA_CMD_CONFIGURATION_SAVE = 7,		//!< Save current configuration to a file \remark On ::CAEN_MCA_HANDLE_DEVICE \sa ConfigurationSaveCommandDataMask
	CAEN_MCA_CMD_CONFIGURATION_LOAD = 8,		//!< Load a saved configuration \remark On ::CAEN_MCA_HANDLE_DEVICE \sa ConfigurationSaveCommandDataMask
	CAEN_MCA_CMD_CONFIGURATION_LIST = 9,		//!< List all saved configuration \remark On ::CAEN_MCA_HANDLE_DEVICE \sa ConfigurationSaveCommandDataMask
	CAEN_MCA_CMD_CONFIGURATION_DELETE = 10,		//!< Delete saved configurations \remark On ::CAEN_MCA_HANDLE_DEVICE \sa ConfigurationSaveCommandDataMask
	CAEN_MCA_CMD_PARAM_AUTOSET_START = 11,		//!< Start Parameter auto-set \remark On ::CAEN_MCA_HANDLE_DEVICE \sa AutoSetCommandDataMask
	CAEN_MCA_CMD_PARAM_AUTOSET_STOP = 12,		//!< Stop Parameter auto-set \remark On ::CAEN_MCA_HANDLE_DEVICE \sa AutoSetCommandDataMask
	CAEN_MCA_CMD_HV_ON = 13,					//!< Enable HV output \remark On ::CAEN_MCA_HANDLE_HVCHANNEL \sa NoCommandDataMask
	CAEN_MCA_CMD_HV_OFF = 14,					//!< Disable HV output \remark On ::CAEN_MCA_HANDLE_HVCHANNEL \sa NoCommandDataMask
	CAEN_MCA_CMD_HV_ONOFF = 15,					//!< Enable or Disable HV output \remark On ::CAEN_MCA_HANDLE_HVCHANNEL \sa HVOutputCommandDataMask
	CAEN_MCA_CMD_MCS_SWEEP = 16,				//!< Send sweep signal for MCS Spectrum \remark On ::CAEN_MCA_HANDLE_MCSSPECTRUM \sa NoCommandDataMask
	CAEN_MCA_CMD_MCS_SPECTRUM_CLEAR = 17,		//!< Clear MCS Spectrum \remark On ::CAEN_MCA_HANDLE_MCSSPECTRUM \sa NoCommandDataMask
	CAEN_MCA_CMD_GAIN_STABILIZER_RESET = 18,	//!< Reset the gain stabilizer \remark On ::CAEN_MCA_HANDLE_ENERGYSPECTRUM \sa NoCommandDataMask
	CAEN_MCA_CMD_DTSPECTRUM_CLEAR = 19,			//!< Clear &Delta;T Spectrum \remark On ::CAEN_MCA_HANDLE_DTSPECTRUM \sa NoCommandDataMask
	CAEN_MCA_CMD_SAMPLE_ADVANCE = 20,			//!< Send Sample Advance signal \remark On ::CAEN_MCA_HANDLE_CHANNEL (single channel) \sa NoCommandDataMask
	CAEN_MCA_CMD_CONFIGURATION_DB_PATH = 21,	//!< Set or get the path of internal configrations database \remark On ::CAEN_MCA_HANDLE_DEVICE \sa ConfigurationSaveCommandDataMask
	CAEN_MCA_CMD_RTCLOCK =22,				    //!< Set or get the internal system datetime \remark On ::CAEN_MCA_HANDLE_DEVICE \sa RTCCommandDataMask
} CAEN_MCA_CommandType_t;

//! Data type definitions
typedef enum {
	CAEN_MCA_DATA_BOARD_INFO = 1,				//!< Board Information \remark On ::CAEN_MCA_HANDLE_DEVICE \sa BoardInfoDataMask
	CAEN_MCA_DATA_CHANNEL_INFO = 2,				//!< Channel Information \remark On ::CAEN_MCA_HANDLE_CHANNEL \sa ChannelInfoDataMask
	CAEN_MCA_DATA_HVCHANNEL_INFO = 3,			//!< HV Channel Information \remark On ::CAEN_MCA_HANDLE_HVCHANNEL \sa HVChannelInfoDataMask
	CAEN_MCA_DATA_HVRANGE_INFO = 4,				//!< HV Range Information \remark On ::CAEN_MCA_HANDLE_HVRANGE \sa HVRangeInfoDataMask
	CAEN_MCA_DATA_COLLECTION = 5,				//!< Handle Collection \remark On ::CAEN_MCA_HANDLE_COLLECTION \sa CollectionDataMask
	CAEN_MCA_DATA_PARAMETER_INFO = 6,			//!< Parameter Information \remark On ::CAEN_MCA_HANDLE_PARAMETER \sa ParamInfoDataMask
	CAEN_MCA_DATA_PARAMETER_VALUE = 7,			//!< Parameter Value \remark On ::CAEN_MCA_HANDLE_PARAMETER \sa ValueDataMask
	CAEN_MCA_DATA_ENERGYSPECTRUM = 8,			//!< Energy Spectrum \remark On ::CAEN_MCA_HANDLE_ENERGYSPECTRUM \sa EnergySpectrumDataMask
	CAEN_MCA_DATA_ROI = 9,						//!< ROI \remark On ::CAEN_MCA_HANDLE_ROI \sa ROIDataMask
	CAEN_MCA_DATA_WAVEFORM = 10,				//!< Waveforms \remark On ::CAEN_MCA_HANDLE_CHANNEL \sa WaveformDataMask
	CAEN_MCA_DATA_LIST_MODE = 11,				//!< List Mode \remark On ::CAEN_MCA_HANDLE_CHANNEL \sa ListDataMask
	CAEN_MCA_DATA_DISCOVEREDDEVICES = 12,		//!< Discover Hexagon devices on the local LAN \remark On ::CAEN_MCA_HANDLE_LIBRARY \sa DiscoveryFunctionDataMask
	CAEN_MCA_DATA_HANDLE_INFO = 13,				//!< Handle information \remark On any handle \sa HandleDataMask
	CAEN_MCA_DATA_MCSSPECTRUM = 14,				//!< MCS Spectrum \remark On ::CAEN_MCA_HANDLE_MCSSPECTRUM \sa MCSSpectrumDataMask
	CAEN_MCA_DATA_DTSPECTRUM = 15,				//!< &Delta;T Spectrum \remark On ::CAEN_MCA_HANDLE_DTSPECTRUM \sa DTSpectrumDataMask
} CAEN_MCA_DataType_t;

//! Form Factor types
typedef enum {
	CAEN_MCA_FORM_FACTOR_VME64 = 0,					//!< VME64 Type
	CAEN_MCA_FORM_FACTOR_VME64X = 1,				//!< VME64x Type
	CAEN_MCA_FORM_FACTOR_DESKTOP = 2,				//!< Desktop Type
	CAEN_MCA_FORM_FACTOR_NIM = 3,					//!< NIM Type
	CAEN_MCA_FORM_FACTOR_UNKNOWN = -1,				//!< Unknown Type
} CAEN_MCA_FormFactor_t;

//! Board Family types
typedef enum {
	CAEN_MCA_FAMILY_CODE_XX724 = 0,					//!< XX724 family
	CAEN_MCA_FAMILY_CODE_XX721 = 1,					//!< XX721 family
	CAEN_MCA_FAMILY_CODE_XX731 = 2,					//!< XX731 family
	CAEN_MCA_FAMILY_CODE_XX720 = 3,					//!< XX720 family
	CAEN_MCA_FAMILY_CODE_XX740 = 4,					//!< XX740 family
	CAEN_MCA_FAMILY_CODE_XX751 = 5,					//!< XX751 family
	CAEN_MCA_FAMILY_CODE_XX742 = 6,					//!< XX742 family
	CAEN_MCA_FAMILY_CODE_XX780 = 7,					//!< XX780 family
	CAEN_MCA_FAMILY_CODE_XX761 = 8,					//!< XX761 family
	CAEN_MCA_FAMILY_CODE_XX743 = 9,					//!< XX743 family
	CAEN_MCA_FAMILY_CODE_XX730 = 10,				//!< XX730 family
	CAEN_MCA_FAMILY_CODE_XX790 = 11,				//!< XX790 family
	CAEN_MCA_FAMILY_CODE_XX781 = 12,				//!< XX781 family
	CAEN_MCA_FAMILY_CODE_XX725 = 13,				//!< XX725 family
	CAEN_MCA_FAMILY_CODE_XXHEX = 14,				//!< Hexagon family
	CAEN_MCA_FAMILY_CODE_XX782 = 15,				//!< XX782 family
	CAEN_MCA_FAMILY_CODE_UNKNOWN = -1,				//!< Unknown family
} CAEN_MCA_BoardFamilyCode_t;

//! Board Model types
typedef enum {
	CAEN_MCA_MODEL_V1724 = 0,						//!< V1724
	CAEN_MCA_MODEL_V1721 = 1,						//!< V1721
	CAEN_MCA_MODEL_V1731 = 2,						//!< V1731
	CAEN_MCA_MODEL_V1720 = 3,						//!< V1720
	CAEN_MCA_MODEL_V1740 = 4,						//!< V1740
	CAEN_MCA_MODEL_V1751 = 5,						//!< V1751
	CAEN_MCA_MODEL_DT5724 = 6,						//!< DT5724
	CAEN_MCA_MODEL_DT5721 = 7,						//!< DT5721
	CAEN_MCA_MODEL_DT5731 = 8,						//!< DT5731
	CAEN_MCA_MODEL_DT5720 = 9,						//!< DT5720
	CAEN_MCA_MODEL_DT5740 = 10,						//!< DT5740
	CAEN_MCA_MODEL_DT5751 = 11,						//!< DT5751
	CAEN_MCA_MODEL_N6724 = 12,						//!< N6724
	CAEN_MCA_MODEL_N6721 = 13,						//!< N6721
	CAEN_MCA_MODEL_N6731 = 14,						//!< N6731
	CAEN_MCA_MODEL_N6720 = 15,						//!< N6720
	CAEN_MCA_MODEL_N6740 = 16,						//!< N6740
	CAEN_MCA_MODEL_N6751 = 17,						//!< N6751
	CAEN_MCA_MODEL_DT5742 = 18,						//!< DT5742
	CAEN_MCA_MODEL_N6742 = 19,						//!< N6742
	CAEN_MCA_MODEL_V1742 = 20,						//!< V1742
	CAEN_MCA_MODEL_DT5780 = 21,						//!< DT5780
	CAEN_MCA_MODEL_N6780 = 22,						//!< N6780
	CAEN_MCA_MODEL_V1780 = 23,						//!< V1780
	CAEN_MCA_MODEL_DT5761 = 24,						//!< DT5761
	CAEN_MCA_MODEL_N6761 = 25,						//!< N6761
	CAEN_MCA_MODEL_V1761 = 26,						//!< V1761
	CAEN_MCA_MODEL_DT5743 = 27,						//!< DT5743
	CAEN_MCA_MODEL_N6743 = 28,						//!< N6743
	CAEN_MCA_MODEL_V1743 = 29,						//!< V1743
	CAEN_MCA_MODEL_DT5730 = 30,						//!< DT5730
	CAEN_MCA_MODEL_N6730 = 31,						//!< N6730
	CAEN_MCA_MODEL_V1730 = 32,						//!< V1730
	CAEN_MCA_MODEL_DT5790 = 33,						//!< DT5790
	CAEN_MCA_MODEL_N6790 = 34,						//!< N6790
	CAEN_MCA_MODEL_V1790 = 35,						//!< V1790
	CAEN_MCA_MODEL_DT5781 = 36,						//!< DT5781
	CAEN_MCA_MODEL_N6781 = 37,						//!< N6781
	CAEN_MCA_MODEL_V1781 = 38,						//!< V1781
	CAEN_MCA_MODEL_DT5725 = 39,						//!< DT5725
	CAEN_MCA_MODEL_N6725 = 40,						//!< N6725
	CAEN_MCA_MODEL_V1725 = 41,						//!< V1725
	CAEN_MCA_MODEL_HEXAGON = 42,					//!< Hexagon
	CAEN_MCA_MODEL_V1782 = 43,						//!< V1782
	CAEN_MCA_MODEL_UNKNOWN = -1,					//!< Unknown model
} CAEN_MCA_BoardModel_t;

//! Parameter Range format
typedef enum {
	CAEN_MCA_PARAMETER_TYPE_RANGE = 0,				//!< Parameter accepting value in a certain range
	CAEN_MCA_PARAMETER_TYPE_LIST = 1,				//!< Parameter accepting only a fixed set of values
	CAEN_MCA_PARAMETER_TYPE_UNKNOWN = -1,			//!< Unknown format
} CAEN_MCA_ParameterType_t;

//! HV Polarity type
typedef enum {
	CAEN_MCA_POLARITY_TYPE_POSITIVE = 0,			//!< Positive polarity
	CAEN_MCA_POLARITY_TYPE_NEGATIVE = 1,			//!< Negative polarity
} CAEN_MCA_Polarity_t;

//! List save mode
typedef enum {
	CAEN_MCA_SAVEMODE_FILE_ASCII = 0,				//!< Save lists into a remote file in ASCII format
	CAEN_MCA_SAVEMODE_FILE_BINARY = 1,				//!< Save lists into a remote file in binary format
	CAEN_MCA_SAVEMODE_MEMORY = 2,					//!< Save lists only in memory
} CAEN_MCA_ListSaveMode_t;

//! HV Ranges
typedef enum {
	CAEN_MCA_HVRANGE_HPGE = 0,						//!< HV Range suited for HPGe detectors
	CAEN_MCA_HVRANGE_PMT = 1,						//!< HV Range suited for PMT detectors
	CAEN_MCA_HVRANGE_SD = 2,						//!< HV Range suited for Silicon-Diamond detectors
} CAEN_MCA_HVRangeCode_t;

//! MCS counting mode
typedef enum {
	CAEN_MCA_MCS_ON_ICR = 0,						//!< MCS counting parameter is the channel trigger
	CAEN_MCA_MCS_ON_EXT_SIGNAL = 1,					//!< MCS counting parameter is the number of external signals
	CAEN_MCA_MCS_ON_SCA = 2,						//!< MCS counting parameter is the number of events with energy inside the SCA interval 
} CAEN_MCA_MCSMode_t;

//! MCS spectrum status
typedef enum {
	CAEN_MCA_MCS_SPE_NOT_STARTED = 0,				//!< MCS is not active, the spectrum is not being filled
	CAEN_MCA_MCS_SPE_ACQUIRING = 1,					//!< MCS spectrum is being filled with MCS counts data 
	CAEN_MCA_MCS_SPE_COMPLETED = 2,					//!< MCS spectrum is ready (waiting for the sweep signal)
	CAEN_MCA_MCS_SPE_STOPPED = 3,					//!< MCS spectrum is freezed
} CAEN_MCA_MCS_Spectrum_Status_t;

//! MCS acquisition mode
typedef enum {
	CAEN_MCA_MCS_REPLACE = 0,						//!< MCS data of the current pass replace those of the previuous one (single scan mode) 
	CAEN_MCA_MCS_SUM = 1,							//!< MCS data of the current pass is added to the sum of the data from the previous passes (multi scan mode)
	CAEN_MCA_MCS_REPLACE_AND_SUM = 2,				//!< MCS data acquisition works in REPLACE mode for the first pass, in SUM mode for the next passes
} CAEN_MCA_MCS_AcqMode_t;

//! MCS acquisition mode
typedef enum {
	CAEN_MCA_GSTAB_MODE_DISABLED = 0,						//!< MCS data of the current pass replace those of the previuous one (single scan mode) 
	CAEN_MCA_GSTAB_MODE_PEAK = 1,							//!< MCS data of the current pass is added to the sum of the data from the previous passes (multi scan mode)
	CAEN_MCA_GSTAB_MODE_SENSOR = 2,				//!< MCS data acquisition works in REPLACE mode for the first pass, in SUM mode for the next passes
} CAEN_MCA_GSTAB_Mode_t;

/********************************************************
*                     RETURN CODES                     *
********************************************************/
//! Library return codes
typedef enum {
	CAEN_MCA_RetCode_Success = 0,					//!< Success
	CAEN_MCA_RetCode_Generic = -1,					//!< Generic error
	CAEN_MCA_RetCode_SockInit = -2,					//!< Socket initialization error
	CAEN_MCA_RetCode_SockConnect = -3,				//!< Socket connect error
	CAEN_MCA_RetCode_OutOfMemory = -4,				//!< Out of memory (malloc failed)
	CAEN_MCA_RetCode_Handle = -5,					//!< Invalid handle
	CAEN_MCA_RetCode_Argument = -6,					//!< Invalid argument
	CAEN_MCA_RetCode_SocketSend = -7,				//!< TCP/IP send error
	CAEN_MCA_RetCode_SocketReceive = -8,			//!< TCP/IP receive error
	CAEN_MCA_RetCode_Protocol = -9,					//!< Protocol error
	CAEN_MCA_RetCode_Serialize = -10,				//!< Serialize error
	CAEN_MCA_RetCode_Deserialize = -11,				//!< Deserialize error
	CAEN_MCA_RetCode_Parameter = -12,				//!< Parameter error
	CAEN_MCA_RetCode_ParameterValue = -13,			//!< Parameter Value error
	CAEN_MCA_RetCode_LibraryLoad = -14,				//!< Library dynamic load error (Linux only)
	CAEN_MCA_RetCode_DiscoveryFunction = -15,		//!< SSDP discovery failed
	CAEN_MCA_RetCode_NotConnected = -16,			//!< Not connected
	CAEN_MCA_RetCode_NotSupported = -17,			//!< Not supported
	CAEN_MCA_RetCode_NotYetImplemented = -18,		//!< Not yet implemented
	CAEN_MCA_RetCode_CollectionFull = -19,			//!< Handle collection full (collections support up to #COLLECTION_MAXLEN elements)
	CAEN_MCA_RetCode_Map = -20,						//!< Error in a map
	CAEN_MCA_RetCode_IndexAlreadyUsed = -21,		//!< Collection index is already in use
} CAEN_MCA_RetCode_t;

#define CONFIGSAVE_NAME_DEFAULTVALUES				"__default"

/********************************************************
*                    COMMAND MASKS                      *
********************************************************/
/*!
* \defgroup CommandMask Command masks
* \ingroup Functions
* \brief To be used in CAEN_MCA_SendCommand()
*/

/*!
* \defgroup NoCommandDataMask Command mask: None
* \ingroup CommandMask
* \brief No command data mask. To be used for commands that do not require any additional parameter.
* \{ */
#define DATAMASK_CMD_NONE							UINT64_C(0x0)		//!< Empty datamask \reqarg no data \argacc input and output
/*! \} */

/*!
* \defgroup RegisterCommandDataMask Command mask: Registers
* \ingroup CommandMask
* \brief Register-related function command data mask.
* \sa CAEN_MCA_CMD_REGISTER_READ CAEN_MCA_CMD_REGISTER_WRITE
* \{ */
#define DATAMASK_CMD_REG_ADDR						UINT64_C(0x1)		//!< Register Address \reqarg `uint32_t` \argacc input
#define DATAMASK_CMD_REG_DATA						UINT64_C(0x2)		//!< Register Data \reqarg `uint32_t*` for ::CAEN_MCA_CMD_REGISTER_READ, `uint32_t` for ::CAEN_MCA_CMD_REGISTER_WRITE \argacc input and output
#define DATAMASK_CMD_REG_MASK						UINT64_C(0x4)		//!< Register Write Mask \reqarg `uint32_t` \argacc input
/*! \} */

/*!
* \defgroup ConfigurationSaveCommandDataMask Command mask: Configuration Save
* \ingroup CommandMask
* \brief Configuration save-related function command data mask.
* \sa CAEN_MCA_CMD_CONFIGURATION_SAVE CAEN_MCA_CMD_CONFIGURATION_LOAD CAEN_MCA_CMD_CONFIGURATION_LIST CAEN_MCA_CMD_CONFIGURATION_DELETE
* \{ */
#define DATAMASK_CMD_SAVE_NAME						UINT64_C(0x1)		//!< Name of the save. \reqarg `char*` \reqsize ::CONFIGSAVE_NAME_MAXLEN \argacc input \warning If not set: \li For ::CAEN_MCA_CMD_CONFIGURATION_SAVE: the default name is a SQL `CURRENT_TIMESTAMP`. \li For ::CAEN_MCA_CMD_CONFIGURATION_LOAD: most recent saved configuration will be loaded. \li For ::CAEN_MCA_CMD_CONFIGURATION_DELETE: all the saved configurations will be deleted.
#define DATAMASK_CMD_SAVE_LIST_OFFSET				UINT64_C(0x2)		//!< Offset for list of saved configuration. If not set: 0. \reqarg `uint32_t` \argacc input
#define DATAMASK_CMD_SAVE_LIST_COUNT				UINT64_C(0x4)		//!< Number of saved configuration \reqarg `uint32_t` \argacc output
#define DATAMASK_CMD_SAVE_LIST_NAMES				UINT64_C(0x8)		//!< List of #CONFIGSAVE_LIST_MAXLEN saved configuration starting from the one speficied in the optional parameter #DATAMASK_CMD_SAVE_LIST_OFFSET \reqarg `char**` pointing to already allocated storages of size ::CONFIGSAVE_NAME_MAXLEN \reqsize ::CONFIGSAVE_LIST_MAXLEN \argacc output
#define DATAMASK_CMD_SAVE_HV_RELOAD					UINT64_C(0x10)		//!< Reload saved HV parameters. Boolean (false if not set). \reqarg `uint32_t` \argacc input
#define DATAMASK_CMD_SAVE_DB_PATH					UINT64_C(0x20)		//!< Set or get the complete path (including filename) were to look for saves database. Such database is used for both ::CAEN_MCA_CMD_CONFIGURATION_SAVE and ::CAEN_MCA_CMD_CONFIGURATION_LOAD commands. \reqarg `char*` \reqsize ::CONFIGSAVE_FULLPATH_MAXLEN \argacc input/output
/*! \} */

/*!
* \defgroup AutoSetCommandDataMask Command mask: Auto set
* \ingroup CommandMask
* \brief Auto set related commands.
* \sa CAEN_MCA_CMD_PARAM_AUTOSET_START CAEN_MCA_CMD_PARAM_AUTOSET_STOP
* \{ */
#define DATAMASK_CMD_AUTOSET_DCO					UINT64_C(0x1)		//!< Activate DC offset autoset. Boolean (false if not set). \reqarg `uint32_t` \argacc input
#define DATAMASK_CMD_AUTOSET_THR					UINT64_C(0x2)		//!< Activate Threshold autoset. Boolean (false if not set). \reqarg `uint32_t` \argacc input
#define DATAMASK_CMD_AUTOSET_TAU					UINT64_C(0x4)		//!< Activate Tau autoset. Boolean (false if not set). \reqarg `uint32_t` \argacc input
/*! \} */

/*!
* \defgroup HVOutputCommandDataMask Command mask: HV output status
* \ingroup CommandMask
* \brief HV output status related commands.
* \sa CAEN_MCA_CMD_HV_ONOFF
* \{ */
#define DATAMASK_CMD_HVOUTPUT_STATUS				UINT64_C(0x1)		//!< Represents HV Channel output status. Boolean (false if channel is OFF). \reqarg `uint32_t` \argacc input and output
/*! \} */

/*!
* \defgroup RTCCommandDataMask Command mask: System Datetime
* \ingroup CommandMask
* \brief RTC related function command data mask.
* \sa CAEN_MCA_CMD_RTCLOCK 
* \{ */
#define DATAMASK_CMD_CURRENT_DATETIME				UINT64_C(0x1)		//!< Datetime string. \reqarg `char*` \reqsize ::DATETIME_MAXLEN \argacc input/output 
/*! \} */

/********************************************************
*                   DATA MASKS                         *
********************************************************/
/*!
* \defgroup DataMask Data masks
* \ingroup Functions
* \brief To be used in CAEN_MCA_GetData() and CAEN_MCA_SetData()
*/

/*!
* \defgroup BoardInfoDataMask Data mask: Board info
* \ingroup DataMask
* \brief Information about the Board. Read only.
* \sa CAEN_MCA_DATA_BOARD_INFO
* \{ */
#define DATAMASK_BRDINFO_MODELNAME					UINT64_C(0x1)		//!< Model name \reqarg `char*` \reqsize ::MODEL_NAME_MAXLEN \argacc read only
#define DATAMASK_BRDINFO_MODEL						UINT64_C(0x2)		//!< Model type \reqarg `int32_t` \argacc read only \sa CAEN_MCA_BoardModel_t
#define DATAMASK_BRDINFO_NCHANNELS					UINT64_C(0x4)		//!< Number of input channels \reqarg `uint32_t` \argacc read only
#define DATAMASK_BRDINFO_ROC_FW_MAJ					UINT64_C(0x8)		//!< ROC firmware version (Major) \reqarg `uint16_t` \argacc read only
#define DATAMASK_BRDINFO_ROC_FW_MIN					UINT64_C(0x10)		//!< ROC firmware version (Minor) \reqarg `uint16_t` \argacc read only
#define DATAMASK_BRDINFO_ROC_FW_BUILD				UINT64_C(0x20)		//!< ROC firmware version (Build) \reqarg `uint16_t` \argacc read only
#define DATAMASK_BRDINFO_AMC_FW_MAJ					UINT64_C(0x40)		//!< AMC firmware version (Major) \reqarg `uint16_t` \argacc read only
#define DATAMASK_BRDINFO_AMC_FW_MIN					UINT64_C(0x80)		//!< AMC firmware version (Minor) \reqarg `uint16_t` \argacc read only
#define DATAMASK_BRDINFO_AMC_FW_BUILD				UINT64_C(0x100)		//!< AMC firmware version (Build) \reqarg `uint16_t` \argacc read only
#define DATAMASK_BRDINFO_LICENSE					UINT64_C(0x200)		//!< License data \reqarg `char*` \reqsize ::LICENSE_MAXLEN \argacc read only
#define DATAMASK_BRDINFO_SERIALNUM					UINT64_C(0x400)		//!< Serial number \reqarg `uint32_t` \argacc read only
#define DATAMASK_BRDINFO_FAMCODE					UINT64_C(0x800)		//!< Family code \reqarg `int32_t` \argacc read only \sa CAEN_MCA_BoardFamilyCode_t
#define DATAMASK_BRDINFO_NHVCHANNELS				UINT64_C(0x1000)	//!< Number of High-Voltage channels \reqarg `uint32_t` \argacc read only
#define DATAMASK_BRDINFO_FORMFACTOR					UINT64_C(0x2000)	//!< Form factor \reqarg `int32_t` \argacc read only \sa CAEN_MCA_FormFactor_t
#define DATAMASK_BRDINFO_PCBREV						UINT64_C(0x4000)	//!< PCB revision \reqarg `uint32_t` \argacc read only
#define DATAMASK_BRDINFO_ADC_BIT_COUNT				UINT64_C(0x8000)	//!< ADC number of bits \reqarg `uint32_t` \argacc read only
#define DATAMASK_BRDINFO_TSAMPLE_PS					UINT64_C(0x10000)	//!< Sample period in picoseconds \reqarg `uint32_t` \argacc read only
#define DATAMASK_BRDINFO_NLVDSGROUPS				UINT64_C(0x20000)	//!< Number of LVDS groups \reqarg `uint32_t` \argacc read only
#define DATAMASK_BRDINFO_NTRACES					UINT64_C(0x40000)	//!< Number of supported traces \reqarg `uint32_t` \argacc read only
#define DATAMASK_BRDINFO_NMONOUTS					UINT64_C(0x80000)	//!< Number of monitor outputs \reqarg `uint32_t` \argacc read only
#define DATAMASK_BRDINFO_ENERGY_BIT_COUNT			UINT64_C(0x100000)	//!< Energy Spectrum maximum number of bits \reqarg `uint32_t` \argacc read only
#define DATAMASK_BRDINFO_LIB_VERSION				UINT64_C(0x200000)	//!< CAENMCA library version supported by board firmware \reqarg `char*` \reqsize ::LIB_VERSION_MAXLEN \argacc read only
#define DATAMASK_BRDINFO_NDTSPECTRA					UINT64_C(0x400000)	//!< Number of DT Spectra \reqarg `uint32_t` \argacc read only
/*! \} */

/*!
* \defgroup ChannelInfoDataMask Data mask: Channel info
* \ingroup DataMask
* \brief Information about the Input Channel Channel. Read only.
* \sa CAEN_MCA_DATA_CHANNEL_INFO
* \{ */
#define DATAMASK_CHANNELINFO_NENERGYSPECTRA			UINT64_C(0x1)		//!< Number of supported Energy Spectra \reqarg `uint32_t`
#define DATAMASK_CHANNELINFO_NMCSSPECTRA			UINT64_C(0x2)		//!< Number of supported MCS Spectra \reqarg `uint32_t`
/*! \} */

/*!
* \defgroup ParamInfoDataMask Data mask: Parameter info
* \ingroup DataMask
* \brief Information about the Parameter. Read only.
* \sa CAEN_MCA_DATA_PARAMETER_INFO
* \{ */
#define DATAMASK_PARAMINFO_NAME						UINT64_C(0x1)		//!< Parameter name \reqarg `char*` \reqsize ::PARAMINFO_NAME_MAXLEN \argacc read only
#define DATAMASK_PARAMINFO_CODENAME					UINT64_C(0X2)		//!< Parameter codename (unique) \reqarg `char*` \reqsize ::PARAMINFO_NAME_MAXLEN \argacc read only
#define DATAMASK_PARAMINFO_INFOMASK					UINT64_C(0X4)		//!< Extra information mask \reqarg `uint32_t` \argacc read only \sa ParamInfoDataMaskInformationMask
#define DATAMASK_PARAMINFO_UOM_NAME					UINT64_C(0x8)		//!< Symbol of the SI unit of measurement. Metrix prefix specified by #DATAMASK_PARAMINFO_UOM_POWER as power of 10.  Example: in case of millivolts, the name will be "V" and the power "-3" \reqarg `char*` \reqsize ::PARAMINFO_NAME_MAXLEN \argacc read only
#define DATAMASK_PARAMINFO_UOM_CODENAME				UINT64_C(0x10)		//!< Codename of the unit of measurement \reqarg `char*` \reqsize ::PARAMINFO_NAME_MAXLEN \argacc read only
#define DATAMASK_PARAMINFO_UOM_POWER				UINT64_C(0x20)		//!< Power of 10 to multiply with respect to the unit of measurement. Example: in case of millivolts, the name will be "V" and the power "-3" \reqarg `int32_t` \argacc read only
#define DATAMASK_PARAMINFO_TYPE						UINT64_C(0x40)		//!< Type of parameter (range or list) \reqarg `int32_t` \argacc read only \sa CAEN_MCA_ParameterType_t
#define DATAMASK_PARAMINFO_MIN						UINT64_C(0x80)		//!< Min. value allowed \reqarg `double` \argacc read only
#define DATAMASK_PARAMINFO_MAX						UINT64_C(0x100)		//!< Max. value allowed \reqarg `double` \argacc read only
#define DATAMASK_PARAMINFO_INCR						UINT64_C(0x200)		//!< Min. step allowed \reqarg `double` \argacc read only
#define DATAMASK_PARAMINFO_NALLOWED_VALUES			UINT64_C(0x400)		//!< In case of list parameter: number of allowed values \reqarg `uint32_t` \argacc read only
#define DATAMASK_PARAMINFO_ALLOWED_VALUES			UINT64_C(0x800)		//!< In case of list parameter: allowed values \reqarg `double*` \reqsize ::PARAMINFO_LIST_MAXLEN \argacc read only
#define DATAMASK_PARAMINFO_ALLOWED_VALUE_CODENAMES	UINT64_C(0x1000)	//!< In case of list parameter: allowed codenames \reqarg `char**` pointing to already allocated storages of size ::PARAMINFO_NAME_MAXLEN \reqsize ::PARAMINFO_LIST_MAXLEN \argacc read only
#define DATAMASK_PARAMINFO_ALLOWED_VALUE_NAMES		UINT64_C(0x2000)	//!< In case of list parameter: names of allowed values \reqarg `char**` pointing to already allocated storages of size ::PARAMINFO_NAME_MAXLEN \reqsize ::PARAMINFO_LIST_MAXLEN \argacc read only
/*! \} */

/*!
* \defgroup ParamInfoDataMaskInformationMask Data mask: Parameter info (information bits)
* \ingroup ParamInfoDataMask
* \brief Bits provided by #DATAMASK_PARAMINFO_INFOMASK.
* \{ */
#define PARAMINFO_INFOMASK_IS_ACQ_ON_ALLOWED		UINT32_C(0x1)		//!< Can be set when acquisition is on
#define PARAMINFO_INFOMASK_IS_HVCONFIG_ALLOWED		UINT32_C(0x2)		//!< Can be set after HV has been configured
#define PARAMINFO_INFOMASK_IS_HVON_ALLOWED			UINT32_C(0x4)		//!< Can be set when HV is ON
#define PARAMINFO_INFOMASK_READABLE					UINT32_C(0x8)		//!< Can be read
#define PARAMINFO_INFOMASK_WRITEABLE				UINT32_C(0x10)		//!< Can be written
/*! \} */

/*!
* \defgroup HVChannelInfoDataMask Data mask: HV channel info
* \ingroup DataMask
* \brief Information about the High-Voltage Channel. Read only.
* \sa CAEN_MCA_DATA_HVCHANNEL_INFO
* \{ */
#define DATAMASK_HVCHANNELINFO_NRANGES				UINT64_C(0x1)		//!< Number of HV Ranges provided by the HV Channel \reqarg `uint32_t` \argacc read only
#define DATAMASK_HVCHANNELINFO_POLARITY				UINT64_C(0x2)		//!< Polarity of the HV Channel \reqarg `int32_t` \argacc read only \sa CAEN_MCA_Polarity_t
/*! \} */

/*!
* \defgroup HVRangeInfoDataMask Data mask: HV range info
* \ingroup DataMask
* \brief Information about the High-Voltage Range. Read only.
* \sa CAEN_MCA_DATA_HVRANGE_INFO
* \{ */
#define DATAMASK_HVRANGEINFO_VSET_MIN				UINT64_C(0x1)		//!< V<sub>set</sub> min. value in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_VSET_MAX				UINT64_C(0x2)		//!< V<sub>set</sub> max. in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_VSET_INCR				UINT64_C(0x4)		//!< V<sub>set</sub> min. step in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_ISET_MIN				UINT64_C(0x8)		//!< I<sub>set</sub> min. value in microampere \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_ISET_MAX				UINT64_C(0x10)		//!< I<sub>set</sub> max. value in microampere \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_ISET_INCR				UINT64_C(0x20)		//!< I<sub>set</sub> min. step in microampere \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_VMON_MIN				UINT64_C(0x40)		//!< V<sub>mon</sub> min. value in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_VMON_MAX				UINT64_C(0x80)		//!< V<sub>mon</sub> max. in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_VMON_INCR				UINT64_C(0x100)		//!< V<sub>mon</sub> min. step in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_IMON_MIN				UINT64_C(0x200)		//!< I<sub>mon</sub> min. value in microampere \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_IMON_MAX				UINT64_C(0x400)		//!< I<sub>mon</sub> max. value in microampere \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_IMON_INCR				UINT64_C(0x800)		//!< I<sub>mon</sub> min. step in microampere \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_VMAX_MIN				UINT64_C(0x1000)	//!< V<sub>max</sub> min. value in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_VMAX_MAX				UINT64_C(0x2000)	//!< V<sub>max</sub> max. in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_VMAX_INCR				UINT64_C(0x4000)	//!< V<sub>max</sub> min. step in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_RAMPUP_MIN				UINT64_C(0x8000)	//!< RampUp min. value in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_RAMPUP_MAX				UINT64_C(0x10000)	//!< RampUp max. in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_RAMPUP_INCR			UINT64_C(0x20000)	//!< RampUp min. step in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_RAMPDOWN_MIN			UINT64_C(0x40000)	//!< RampDown min. value in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_RAMPDOWN_MAX			UINT64_C(0x80000)	//!< RampDown max. in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_RAMPDOWN_INCR			UINT64_C(0x100000)	//!< RampDown min. step in volts \reqarg `double` \argacc read only
#define DATAMASK_HVRANGEINFO_CODE					UINT64_C(0x200000)	//!< HV Range code \reqarg `int32_t` \argacc read only \sa CAEN_MCA_HVRangeCode_t
#define DATAMASK_HVRANGEINFO_CODENAME				UINT64_C(0x400000)	//!< HV Range codename \reqarg `char*` \reqsize ::HVRANGEINFO_NAME_MAXLEN \argacc read only
#define DATAMASK_HVRANGEINFO_NAME					UINT64_C(0x800000)	//!< HV Range name \reqarg `char*` \reqsize ::HVRANGEINFO_NAME_MAXLEN \argacc read only
/*! \} */

/*!
* \defgroup CollectionDataMask Data mask: Handle collection
* \ingroup DataMask
* \brief Information about the Handle Collection. Read only.
* \sa CAEN_MCA_DATA_COLLECTION
* \{ */
#define DATAMASK_COLLECTION_LENGTH					UINT64_C(0x1)		//!< Number of handles in the collection \reqarg `uint32_t` \argacc read only
#define DATAMASK_COLLECTION_HANDLES					UINT64_C(0x2)		//!< Array of handles \reqarg `::CAEN_MCA_HANDLE*` \reqsize ::COLLECTION_MAXLEN \argacc read only
/*! \} */

/*!
* \defgroup ValueDataMask Data mask: Parameter value
* \ingroup DataMask
* \brief Information about the current value of the Parameter. Read/Write.
* \sa CAEN_MCA_DATA_PARAMETER_VALUE
* \{ */
#define DATAMASK_VALUE_NUMERIC						UINT64_C(0x1)		//!< Numeric value \reqarg `double` \argacc read/write
#define DATAMASK_VALUE_CODENAME						UINT64_C(0x2)		//!< Value codename in case of list parameter. Literal value in case or range parameter \reqarg `char*` \reqsize ::PARAMINFO_NAME_MAXLEN \argacc read/write
#define DATAMASK_VALUE_NAME							UINT64_C(0x4)		//!< Value name in case of list parameter. Literal value in case or range parameter \reqarg `char*` \reqsize ::PARAMINFO_NAME_MAXLEN \argacc read only
/*! \} */

/*!
* \defgroup EnergySpectrumDataMask Data mask: Energy spectrum
* \ingroup DataMask
* \brief Information about the Energy Spectrum. Read only except #DATAMASK_ENERGY_SPECTRUM_FILENAME and #DATAMASK_ENERGY_SPECTRUM_AUTOSAVE_PERIOD.
* \sa CAEN_MCA_DATA_ENERGYSPECTRUM
* \{ */
#define DATAMASK_ENERGY_SPECTRUM_ARRAY				UINT64_C(0x1)		//!< The spectrum \reqarg `uint32_t*` \reqsize ::ENERGYSPECTRUM_MAXLEN \argacc read only
#define DATAMASK_ENERGY_SPECTRUM_RTIME				UINT64_C(0x2)		//!< Real time \reqarg `uint64_t` \argacc read only
#define DATAMASK_ENERGY_SPECTRUM_LTIME				UINT64_C(0x4)		//!< Live time \reqarg `uint64_t` \argacc read only
#define DATAMASK_ENERGY_SPECTRUM_DTIME				UINT64_C(0x8)		//!< Dead time \reqarg `uint64_t` \argacc read only
#define DATAMASK_ENERGY_SPECTRUM_OVERFLOW			UINT64_C(0x10)		//!< Number of overflows \reqarg `uint32_t` \argacc read only
#define DATAMASK_ENERGY_SPECTRUM_UNDERFLOW			UINT64_C(0x20)		//!< Number of underflows \reqarg `uint32_t` \argacc read only
#define DATAMASK_ENERGY_SPECTRUM_NENTRIES			UINT64_C(0x40)		//!< Total number of entries in the spectrum \reqarg `uint64_t` \argacc read only
#define DATAMASK_ENERGY_SPECTRUM_NROIS				UINT64_C(0x80)		//!< Number of supported ROIs \reqarg `uint32_t` \argacc read only
#define DATAMASK_ENERGY_SPECTRUM_FILENAME			UINT64_C(0x100)		//!< Remote file where the spectrum is saved periodically and at acquisition stop \reqarg `char*` \reqsize ::ENERGYSPECTRUM_FULLPATH_MAXLEN \argacc read/write
#define DATAMASK_ENERGY_SPECTRUM_AUTOSAVE_PERIOD	UINT64_C(0x200)		//!< Period of the auto save of the spectra to file in milliseconds. Set to 0 to disable the autosave. \reqarg `uint32_t` \argacc read/write
#define DATAMASK_ENERGY_SPECTRUM_STABILIZER_ENABLE	UINT64_C(0x400)		//!< Status of the gain stabilizer. Set to 0 to disable the gain stabilizer. \reqarg `uint32_t` \argacc read/write
#define DATAMASK_ENERGY_SPECTRUM_SAMPLE_ID			UINT64_C(0x800)		//!< Identification of the sample from which the spectrum is obtained \reqarg `char*` \reqsize ::SAMPLEID_MAXLEN \argacc read/write
#define DATAMASK_ENERGY_SPECTRUM_START_DATETIME		UINT64_C(0x1000)	//!< Datetime of last data acquisition start \reqarg `char*` \reqsize ::DATETIME_MAXLEN \argacc read only
#define DATAMASK_ENERGY_SPECTRUM_STOP_DATETIME		UINT64_C(0x2000)	//!< Datetime of last data acquisition stop \reqarg `char*` \reqsize ::DATETIME_MAXLEN \argacc read only
/*! \} */

/*!
* \defgroup ROIDataMask Data mask: ROI
* \ingroup DataMask
* \brief Information about the ROI.
* \sa CAEN_MCA_DATA_ROI
* \{ */
#define DATAMASK_ROI_NENTRIES						UINT64_C(0x1)		//!< Total number of entries in the ROI \reqarg `uint64_t` \argacc read only
/*! \} */

/*!
* \defgroup DTSpectrumDataMask Data mask: &Delta;T spectrum
* \ingroup DataMask
* \brief Information about the &Delta;T Spectrum. Read only except #DATAMASK_DT_SPECTRUM_ENABLE and #DATAMASK_DT_SPECTRUM_REFCH.
* \sa CAEN_MCA_DATA_DTSPECTRUM
* \{ */
#define DATAMASK_DT_SPECTRUM_ARRAY					UINT64_C(0x1)		//!< The spectrum \reqarg `uint64_t*` \reqsize ::DTSPECTRUM_MAXLEN \argacc read only
#define DATAMASK_DT_SPECTRUM_NENTRIES				UINT64_C(0x2)		//!< Total number of entries in the spectrum \reqarg `uint64_t` \argacc read only
#define DATAMASK_DT_SPECTRUM_ENABLE					UINT64_C(0x4)		//!< &Delta;T Spectrum enabled. Boolean. \reqarg `uint32_t` \argacc read/write
#define DATAMASK_DT_SPECTRUM_REFCH					UINT64_C(0x8)		//!< Reference channel for DT calculation. \reqarg `uint32_t` \argacc read/write
/*! \} */

/*!
* \defgroup WaveformDataMask Data mask: Waveforms
* \ingroup DataMask
* \brief The last waveform acquired when the hardware is in waveform mode.
* \sa CAEN_MCA_DATA_WAVEFORM 
* \{ */
#define DATAMASK_WAVEFORM_TRACE1					UINT64_C(0x1)		//!< Analog trace #1 \reqarg `int16_t*` \reqsize ::WAVEFORM_MAXLEN \argacc read only
#define DATAMASK_WAVEFORM_TRACE2					UINT64_C(0x2)		//!< Analog trace #2 \reqarg `int16_t*` \reqsize ::WAVEFORM_MAXLEN \argacc read only
#define DATAMASK_WAVEFORM_DTRACE1					UINT64_C(0x4)		//!< Digital trace #1 \reqarg `uint8_t*` \reqsize ::WAVEFORM_MAXLEN \argacc read only
#define DATAMASK_WAVEFORM_DTRACE2					UINT64_C(0x8)		//!< Digital trace #2 \reqarg `uint8_t*` \reqsize ::WAVEFORM_MAXLEN \argacc read only
/*! \} */

/*!
* \defgroup DiscoveryFunctionDataMask Data mask: Device discovery
* \ingroup DataMask
* \brief Information about the discovery if Hexagon devices on the local LAN. Read only except #DATAMASK_DISCOVERY_TIMEOUT_MS.
* \sa CAEN_MCA_DATA_DISCOVEREDDEVICES
* \{ */
#define DATAMASK_DISCOVERY_TIMEOUT_MS				UINT64_C(0x1)		//!< Time-out of the function in milliseconds \reqarg `uint32_t` \argacc read/write
#define DATAMASK_DISCOVERY_FOUNDCOUNT				UINT64_C(0x2)		//!< Number of devices found \reqarg `uint32_t` \argacc read only
#define DATAMASK_DISCOVERY_NAME						UINT64_C(0x4)		//!< Array of found devices \reqarg `char**` pointing to already allocated storages of size #HEXAGONDISCOVERY_NAME_MAXLEN \reqsize ::HEXAGONDISCOVERY_LIST_MAXLEN \argacc read only
#define DATAMASK_DISCOVERY_SERIALNUMBER				UINT64_C(0x8)		//!< Array of found serial numbers \reqarg `uint32_t*` \reqsize ::HEXAGONDISCOVERY_LIST_MAXLEN \argacc read only
#define DATAMASK_DISCOVERY_IP						UINT64_C(0x10)		//!< Array of found IP addresses \reqarg `char**` pointing to already allocated storages of size ::HEXAGONDISCOVERY_IP_MAXLEN \reqsize ::HEXAGONDISCOVERY_LIST_MAXLEN \argacc read only
#define DATAMASK_DISCOVERY_INPUT_CHANNELS			UINT64_C(0x20)		//!< Array of found Number of Input Channels \reqarg `uint32_t*`  \reqsize ::HEXAGONDISCOVERY_LIST_MAXLEN \argacc read only
#define DATAMASK_DISCOVERY_MODEL_NAME				UINT64_C(0x40)		//!< Array of found model names \reqarg `char**` pointing to already allocated storages of size ::HEXAGONDISCOVERY_NAME_MAXLEN \reqsize ::HEXAGONDISCOVERY_LIST_MAXLEN \argacc read only
#define DATAMASK_DISCOVERY_PATH						UINT64_C(0x80)		//!< Array of found path to use as argument of CAEN_MCA_OpenDevice() \reqarg `char**` pointing to already allocated storages of size ::HEXAGONDISCOVERY_NAME_MAXLEN \reqsize ::HEXAGONDISCOVERY_LIST_MAXLEN \argacc read only
/*! \} */

/*!
* \defgroup ListDataMask Data mask: Lists
* \ingroup DataMask
* \brief Information about the list mode. Some data is read only.
* \sa CAEN_MCA_DATA_LIST_MODE 	
* \{ */
#define DATAMASK_LIST_ENABLE						UINT64_C(0x1)		//!< List enabled. Boolean. \reqarg `uint32_t` \argacc read/write
#define DATAMASK_LIST_SAVEMODE						UINT64_C(0x2)		//!< Current List mode \reqarg `int32_t` \argacc read/write \sa CAEN_MCA_ListSaveMode_t
#define DATAMASK_LIST_FILENAME						UINT64_C(0x4)		//!< In case of ASCII-Binary mode: remote file name used \reqarg `char*` \reqsize ::LISTS_FULLPATH_MAXLEN \argacc read/write
#define DATAMASK_LIST_FILE_DATAMASK					UINT64_C(0x8)		//!< Data to save \reqarg `uint32_t` \argacc read/write \sa ListDataMaskDataMask
#define DATAMASK_LIST_GETFAKEEVTS					UINT64_C(0x10)		//!< Get also fake events. Boolean. \reqarg `uint32_t` \argacc read/write
#define DATAMASK_LIST_MAXNEVTS						UINT64_C(0x20)		//!< In case of memory mode: max number of events to get in case of memory mode. Must be < ::LISTS_DATA_MAXLEN \reqarg `uint32_t` \argacc read/write
#define DATAMASK_LIST_NEVTS							UINT64_C(0x40)		//!< Number of events in the data \reqarg `uint32_t` \argacc read only
#define DATAMASK_LIST_DATA_TIMETAG					UINT64_C(0x80)		//!< Timetags of the events in picoseconds since acquisition start \reqarg `uint64_t*` \reqsize ::LISTS_DATA_MAXLEN \argacc read only \warning Should be used only when save mode is ::CAEN_MCA_SAVEMODE_MEMORY, as the internal buffer is reset when this data is read. 
#define DATAMASK_LIST_DATA_ENERGY					UINT64_C(0x100)		//!< Energies of the events in bins \reqarg `uint16_t*` \reqsize LISTS_DATA_MAXLEN \argacc read only \warning Should be used only when save mode is ::CAEN_MCA_SAVEMODE_MEMORY, as the internal buffer is reset when this data is read. 
#define DATAMASK_LIST_DATA_FLAGS_DATAMASK			UINT64_C(0x200)		//!< Flags of the events \reqarg `uint32_t*` \reqsize ::LISTS_DATA_MAXLEN \argacc read only \warning Should be used only when save mode is ::CAEN_MCA_SAVEMODE_MEMORY, as the internal buffer is reset when this data is read. \sa ListEventFlagsMask
#define DATAMASK_LIST_GETSATEVTS					UINT64_C(0x400)		//!< Get also saturated events. Boolean. \reqarg `uint32_t` \argacc read/write
#define DATAMASK_LIST_GETSKIMEVTS					UINT64_C(0x800)		//!< Get also energy skim events (whose energy is outside the spectrum [LLD,ULD] interval). Boolean. \reqarg `uint32_t` \argacc read/write
/*! \} */

/*!
* \defgroup ListDataMaskDataMask Data mask: Lists (type of data)
* \ingroup ListDataMask
* \brief Bits provided by #DATAMASK_LIST_FILE_DATAMASK.
* \{ */
#define LIST_FILE_DATAMASK_TIMETAG					UINT32_C(0x1)		//!< enable Timetag
#define LIST_FILE_DATAMASK_ENERGY					UINT32_C(0x2)		//!< enable Energy
#define LIST_FILE_DATAMASK_FLAGS					UINT32_C(0x4)		//!< enable Flags
/*! \} */

/*!
* \defgroup ListEventFlagsMask Data mask: Lists (event flags)
* \ingroup ListDataMask
* \brief Bits provided by #DATAMASK_LIST_DATA_FLAGS_DATAMASK. Copied from CAENDPPLayerTypes.h
* \{ */
#define LIST_DATA_FLAGS_DATAMASK_DEADTIME			UINT32_C(0x1)		//!< Identifies the first event after a dead time.
#define LIST_DATA_FLAGS_DATAMASK_TTROLLOVER			UINT32_C(0x2)		//!< Identifies a trigger time stamp roll-over that occurred before this event.
#define LIST_DATA_FLAGS_DATAMASK_TTRESET			UINT32_C(0x4)		//!< Identifies a trigger time stamp reset forced from external signals in S-IN (GPI for Desktop).
#define LIST_DATA_FLAGS_DATAMASK_EVTFAKE			UINT32_C(0x8)		//!< Identifies a fake event (which does not correspond to any physical event).
#define LIST_DATA_FLAGS_DATAMASK_MEMFULL			UINT32_C(0x10)		//!< Reading memory full.
#define LIST_DATA_FLAGS_DATAMASK_TRGLOST			UINT32_C(0x20)		//!< Identifies the first event after a trigger lost.
#define LIST_DATA_FLAGS_DATAMASK_NTRGLOST			UINT32_C(0x40)		//!< Every N lost events this flag is high (see board documentation to set N).
#define LIST_DATA_FLAGS_DATAMASK_OVERRNG			UINT32_C(0x80)		//!< Energy overranged.
#define LIST_DATA_FLAGS_DATAMASK_1024TRG			UINT32_C(0x100)		//!< 1024 triggers counted (every 1024 counted events this flag is high).
#define LIST_DATA_FLAGS_DATAMASK_LOSTEVT			UINT32_C(0x200)		//!< Identifies the first event after when one or more events is lost due to a memory board FULL. The memory can be FULL due to a write event.
#define LIST_DATA_FLAGS_DATAMASK_INPUTSAT			UINT32_C(0x400)		//!< Identifies an event saturating the input dynamics (clipping).
#define LIST_DATA_FLAGS_DATAMASK_NTRGTOT			UINT32_C(0x800)		//!< Every N total events this flag is high (see board documentation to set N).
#define LIST_DATA_FLAGS_DATAMASK_OLDSORT			UINT32_C(0x1000)	//!< Identifies an event not sorted but sent for waveform.
#define LIST_DATA_FLAGS_DATAMASK_EOR				UINT32_C(0x2000)	//!< Identifies a fake event occurring at the end of run.
#define LIST_DATA_FLAGS_DATAMASK_FINETT				UINT32_C(0x4000)	//!< Identifies an event with fine time stamp.
#define LIST_DATA_FLAGS_DATAMASK_PILEUP				UINT32_C(0x8000)	//!< Identifies a pile up event.
#define LIST_DATA_FLAGS_DATAMASK_TIME				UINT32_C(0x10000)	//!< Identifies a fake event occurring on a time stamp roll-over.
#define LIST_DATA_FLAGS_DATAMASK_ENERGY_SKIM		UINT32_C(0x20000)	//!< Energy skimming.
#define LIST_DATA_FLAGS_DATAMASK_SATREJ				UINT32_C(0x40000)	//!< Identifies an event occurred when detector was inibited due to saturation.
/*! \} */

/*!
* \defgroup HandleDataMask Data mask: Handles
* \ingroup DataMask
* \brief Information about the handle. Read only.
* \sa CAEN_MCA_DATA_HANDLE_INFO
* \{ */
#define DATAMASK_HANDLE_TYPE						UINT64_C(0x1)		//!< Handle type \reqarg `int32_t` \argacc read only \sa CAEN_MCA_HandleType_t
#define DATAMASK_HANDLE_INDEX						UINT64_C(0x2)		//!< Handle index \reqarg `int32_t` \argacc read only
#define DATAMASK_HANDLE_NAME						UINT64_C(0x4)		//!< Handle name \reqarg `char*` \reqsize ::HANDLE_NAME_MAXLEN \argacc read only
/*! \} */


/*!
* \defgroup MCSSpectrumDataMask Data mask: MCS spectrum
* \ingroup DataMask
* \brief Information about the MCS Spectrum. Read only.
* \sa CAEN_MCA_DATA_MCSSPECTRUM
* \{ */
#define DATAMASK_MCS_SPECTRUM_ARRAY					UINT64_C(0x1)		//!< The MCS spectrum \reqarg `uint32_t*` \reqsize ::MCSSPECTRUM_MAXLEN \argacc read only
#define DATAMASK_MCS_SPECTRUM_STATUS				UINT64_C(0x2)		//!< The MCS spectrum status \reqarg `uint8_t` \argacc read only \sa CAEN_MCA_MCS_Spectrum_Status_t
#define DATAMASK_MCS_SPECTRUM_CURRENT_PASS			UINT64_C(0x4)		//!< The MCS current pass number \reqarg `uint32_t` \argacc read only
/*! \} */

/********************************************************
*          MAX LENGTH AND DEFAULTS DEFINITIONS          *
********************************************************/

#ifdef __cplusplus
extern "C" {
#endif

//! Default port
_MCALIB_SYMBOL_DLLAPI_ int32_t CAEN_MCA_DEFAULT_TCP_PORT;

/*!
* \defgroup MaxLengthDefs Maximum length macros
* \brief Macros to define the maximum length or size for various type of data.
* \{ */
_MCALIB_SYMBOL_DLLAPI_ int32_t PARAMINFO_LIST_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t HANDLE_NAME_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t PARAMINFO_NAME_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t HVRANGEINFO_NAME_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t MODEL_NAME_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t LICENSE_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t VERSION_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t LIB_VERSION_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t ENERGYSPECTRUM_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t WAVEFORM_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t DEFINITIONARRAY_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t CONFIGSAVE_NAME_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t CONFIGSAVE_LIST_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t DISCOVERY_NAME_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t DISCOVERY_LIST_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t DISCOVERY_IP_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t LISTS_DATA_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t COLLECTION_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t MCSSPECTRUM_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t DTSPECTRUM_MAXLEN;
// For retrocompatibility (deprecated)
#define HEXAGONDISCOVERY_NAME_MAXLEN (DISCOVERY_NAME_MAXLEN)
#define HEXAGONDISCOVERY_LIST_MAXLEN (DISCOVERY_LIST_MAXLEN)
#define HEXAGONDISCOVERY_IP_MAXLEN (DISCOVERY_IP_MAXLEN)

// Filename lengths
_MCALIB_SYMBOL_DLLAPI_ int32_t STANDARD_FILE_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t BASEDIR_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t RUNSAVE_RELATIVEPATH_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t RUNSAVE_FULLPATH_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t ENERGYSPECTRUM_RELATIVEPATH_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t ENERGYSPECTRUM_FULLPATH_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t CONFIGSAVE_RELATIVEPATH_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t CONFIGSAVE_FULLPATH_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t LISTS_RELATIVEPATH_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t LISTS_FULLPATH_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t SAMPLEID_MAXLEN;
_MCALIB_SYMBOL_DLLAPI_ int32_t DATETIME_MAXLEN;
/*! \} */

#ifdef __cplusplus
}
#endif

#endif // _CAEN_MCA_TYPES_H
