/******************************************************************************
*
*	CAEN SpA - Front End Division
*	Via Vetraia, 11 - 55049 - Viareggio ITALY
*	+39 0594 388 398 - www.caen.it
*
*	This file is part of CAEN MCA which is released under
*	GNU Lesser General Public License. See file doc/README.md
*	or go to https://www.gnu.org/licenses/ for full license details.
*
***************************************************************************//*!
*
*	\file		MCALIB_Common.h
*	\brief		Internal stuff
*	\author		Giovanni Cerretani, Lara Palla, Francesco Pepe
*	\version	0.99.11
*	\date		2023
*	\copyright	GNU Lesser General Public License
*
******************************************************************************/
#ifndef _MCALIB_COMMON_H
#define _MCALIB_COMMON_H

#include <CAENUtility.h>
#include <CAENSocket.h>
#include <CAENLogger.h>
#include <CAENMap.h>
#include <CAENXMLParser.h>

#include <CAENMCATypes.h>
#include <MCALIB_SharedClientServer.h>

/*****************************************************************
*	Generic types:
*
*	- MCALIB_HandleType_t
*	- MCALIB_ObjectCommonData_t
*	- MCALIB_Handle_t
*	- MCALIB_HandleMap_t
*
******************************************************************/

//! Device info struct
typedef struct {
	char ModelName[MODEL_NAME_MAXLEN_DEF];
	int32_t Model;
	uint32_t nChannels;
	uint32_t nHVChannels;
	uint32_t nLVDSGroups;
	uint32_t nVirtualTraces;
	uint32_t nMonOuts;
	uint32_t nDTSpectra;
	uint32_t SerialNum;
	char LibVersion[LIB_VERSION_MAXLEN_DEF];
} MCALIB_DeviceInfo_t;

typedef struct {
	const CAEN_MCA_HandleType_t typeCode;
	const size_t size;
	const char * const typeName;
} MCALIB_HandleType_t;

typedef struct MCALIB_Handle_t MCALIB_Handle_t;

typedef struct {
	const MCALIB_HandleType_t *type;
	MCALIB_Handle_t *handle;
	const MCALIB_Handle_t *parentHandle;
	char name[HANDLE_NAME_MAXLEN_DEF];
	int32_t index; // Client/Server index
	int32_t collectionIndex; // Collection index
} MCALIB_ObjectCommonData_t;

struct MCALIB_Handle_t {
	uint64_t controlPattern;
	MCALIB_ObjectCommonData_t *commonData;
    void *data;
};

typedef c_map_t(MCALIB_Handle_t*) MCALIB_HandleMap_t;

/*****************************************************************
*	Object types. 
*
*	*** NOTE:
*	*** The fist element type MUST be MCALIB_ObjectCommonData_t
*	*** See MCALIB_CreateObject() and MCALIB_DeleteObject()
******************************************************************/
typedef struct {
	MCALIB_ObjectCommonData_t commonData; // for MCALIB_HandleCollection_t the "commondata.index" represents the type of handle it can contain.
	MCALIB_Handle_t **handles;
	MCALIB_HandleMap_t *map;
	uint32_t length;
} MCALIB_HandleCollection_t;

// This is similar to MCALIB_HandleCollection_t but handles can be removed.
// It is implemented using two maps, the returned index is not incremental because it
// comes from name hashing.
typedef struct {
	MCALIB_ObjectCommonData_t commonData; // for MCALIB_RemovableHandleCollection_t the "commondata.index" represents the type of handle it can contain.
	MCALIB_HandleMap_t* map_index; // the index is obtained as (hashed_name % INT_MAX), converted to string and used as key.
	MCALIB_HandleMap_t* map_name; // the key is the name of the handle.
} MCALIB_RemovableHandleCollection_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
} MCALIB_Event_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
} MCALIB_Trigger_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
} MCALIB_Value_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
	MCALIB_HandleCollection_t *parameters;
	MCALIB_HandleCollection_t *rois;
	uint32_t nROIs;
} MCALIB_EnergySpectrum_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
	MCALIB_HandleCollection_t *parameters;
} MCALIB_MCSSpectrum_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
	MCALIB_HandleCollection_t *parameters;
} MCALIB_ROI_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
	MCALIB_HandleCollection_t *parameters;
} MCALIB_DTSpectrum_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
	MCALIB_HandleCollection_t *parameters;
} MCALIB_MonOut_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
	MCALIB_HandleCollection_t *parameters;
} MCALIB_Parameter_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
    MCALIB_HandleCollection_t *parameters;
	MCALIB_HandleCollection_t *energyspectra;
	MCALIB_HandleCollection_t *mcsspectra;
	uint32_t nEnergySpectra;
	uint32_t nMCSSpectra;
} MCALIB_Channel_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
	MCALIB_HandleCollection_t *parameters;
} MCALIB_HVRange_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
	MCALIB_HandleCollection_t *parameters;
	MCALIB_HandleCollection_t *hvranges;
	uint32_t nRanges;
} MCALIB_HVChannel_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
	MCALIB_HandleCollection_t *parameters;
} MCALIB_LVDSGroup_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
	MCALIB_HandleCollection_t *parameters;
} MCALIB_Trace_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
	MCALIB_HandleCollection_t *parameters;
	MCALIB_HandleCollection_t *channels;
	MCALIB_HandleCollection_t *hvchannels;
	MCALIB_HandleCollection_t *lvdsgroups;
	MCALIB_HandleCollection_t *traces;
	MCALIB_HandleCollection_t *monouts;
	MCALIB_HandleCollection_t *dtspectra;
	MCALIB_DeviceInfo_t DeviceInfo;
	c_socket_t *socket;
	c_xmlfile_t *definitions;
	bool isInitialized;
} MCALIB_Device_t;

typedef struct {
	MCALIB_ObjectCommonData_t commonData;
	MCALIB_RemovableHandleCollection_t *devices;
	uint32_t discoverTimeout_ms;
	char libPath[1024]; // will be filled at startup with the path of the DLL
} MCALIB_Library_t;


/*****************************************************************
*	Global variables
******************************************************************/
extern const MCALIB_HandleType_t MCALIB_Handle_Library;
extern const MCALIB_HandleType_t MCALIB_Handle_Device;
extern const MCALIB_HandleType_t MCALIB_Handle_Channel;
extern const MCALIB_HandleType_t MCALIB_Handle_HVChannel;
extern const MCALIB_HandleType_t MCALIB_Handle_HVRange;
extern const MCALIB_HandleType_t MCALIB_Handle_LVDSGroup;
extern const MCALIB_HandleType_t MCALIB_Handle_Trace;
extern const MCALIB_HandleType_t MCALIB_Handle_Parameter;
extern const MCALIB_HandleType_t MCALIB_Handle_Collection;
extern const MCALIB_HandleType_t MCALIB_Handle_Event;
extern const MCALIB_HandleType_t MCALIB_Handle_Trigger;
extern const MCALIB_HandleType_t MCALIB_Handle_Value;
extern const MCALIB_HandleType_t MCALIB_Handle_EnergySpectrum;
extern const MCALIB_HandleType_t MCALIB_Handle_MCSSpectrum;
extern const MCALIB_HandleType_t MCALIB_Handle_ROI;
extern const MCALIB_HandleType_t MCALIB_Handle_MonOut;
extern const MCALIB_HandleType_t MCALIB_Handle_DTSpectrum;
extern const MCALIB_HandleType_t MCALIB_Handle_Removable_Collection;
extern MCALIB_Library_t *Library;

/*****************************************************************
*	Pattern
******************************************************************/
bool MCALIB_IsValidHandle(const MCALIB_Handle_t *handle);
uint64_t MCALIB_GetPatternGood(void);
void MCALIB_SetPatternGood(uint64_t pattern);
uint64_t MCALIB_GetPatternBad(void);
void MCALIB_SetPatternBad(uint64_t pattern);

/*****************************************************************
*	Functions
******************************************************************/
MCALIB_Handle_t* MCALIB_CreateHandle(MCALIB_ObjectCommonData_t *commonData, void *data) c_attribute_malloc;
void MCALIB_DeleteHandle(MCALIB_Handle_t *handle);

void* MCALIB_CreateObject(const MCALIB_HandleType_t *type, const MCALIB_Handle_t *parent, int32_t index) c_attribute_malloc;
void MCALIB_DeleteObject(void *object);

void* MCALIB_GetObject(const MCALIB_Handle_t *handle, CAEN_MCA_HandleType_t type);
void* MCALIB_GetAncestor(const MCALIB_Handle_t *handle, CAEN_MCA_HandleType_t type);

MCALIB_Handle_t* MCALIB_GetLibraryHandle(void);
MCALIB_Handle_t* MCALIB_GetDeviceHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name);
MCALIB_Handle_t* MCALIB_GetChannelHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name);
MCALIB_Handle_t* MCALIB_GetEnergySpectrumHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name);
MCALIB_Handle_t* MCALIB_GetMCSSpectrumHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name);
MCALIB_Handle_t* MCALIB_GetDTSpectrumHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name);
MCALIB_Handle_t* MCALIB_GetROIHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name);
MCALIB_Handle_t* MCALIB_GetHVChannelHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name);
MCALIB_Handle_t* MCALIB_GetHVRangeHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name);
MCALIB_Handle_t* MCALIB_GetLVDSGroupHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name);
MCALIB_Handle_t* MCALIB_GetTraceHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name);
MCALIB_Handle_t* MCALIB_GetMonOutHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name);
MCALIB_Handle_t* MCALIB_GetParameterHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name);
MCALIB_Handle_t* MCALIB_GetHandleCollectionHandleByType(const MCALIB_Handle_t *handle, CAEN_MCA_HandleType_t type);

MCALIB_HandleCollection_t* MCALIB_GetParameterHandleCollection(const MCALIB_Handle_t *handle);

MCALIB_HandleCollection_t* MCALIB_CreateHandleCollection(const MCALIB_Handle_t *parent, CAEN_MCA_HandleType_t type);
void MCALIB_DeleteHandleCollection(MCALIB_HandleCollection_t *coll);
int32_t MCALIB_AddHandleTo(MCALIB_HandleCollection_t* coll, MCALIB_Handle_t* handle);
MCALIB_Handle_t* MCALIB_GetHandleAt(const MCALIB_HandleCollection_t *coll, uint32_t index);
MCALIB_Handle_t* MCALIB_GetHandleByName(const MCALIB_HandleCollection_t *coll, const char *name);
int32_t MCALIB_GetHandleCollectionLength(const MCALIB_HandleCollection_t *coll, uint32_t *length);

MCALIB_RemovableHandleCollection_t* MCALIB_CreateRemovableHandleCollection(const MCALIB_Handle_t* parent, CAEN_MCA_HandleType_t type);
void MCALIB_DeleteRemovableHandleCollection(MCALIB_RemovableHandleCollection_t* object);
int32_t MCALIB_AddRemovableHandleToIdx(MCALIB_RemovableHandleCollection_t* coll, MCALIB_Handle_t* handle, int32_t* index);
MCALIB_Handle_t* MCALIB_GetRemovableHandleAt(const MCALIB_RemovableHandleCollection_t* coll, uint32_t index);
MCALIB_Handle_t* MCALIB_GetRemovableHandleByName(const MCALIB_RemovableHandleCollection_t* coll, const char* name);
int32_t MCALIB_GetRemovableHandleCollectionLength(const MCALIB_RemovableHandleCollection_t* coll, uint32_t* length);
int32_t MCALIB_RemoveHandleFrom(MCALIB_RemovableHandleCollection_t* coll, MCALIB_Handle_t* handle);

int32_t MCALIB_GetIndex(const MCALIB_Handle_t *handle);
int32_t MCALIB_GetParentIndexes(const MCALIB_Handle_t *handle, int32_t parentIndexes[NESTED_HANDLE_MAXLEN], uint32_t *length);

int32_t MCALIB_GetDefinitionFile(const MCALIB_Handle_t *handle);

int32_t MCALIB_GetBoardInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);
int32_t MCALIB_GetChannelInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);
int32_t MCALIB_GetHVChannelInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);
int32_t MCALIB_GetHVRangeInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);
int32_t MCALIB_GetCollectionInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);

int32_t MCALIB_GetParameterInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);
int32_t MCALIB_GetParameterValue(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);
int32_t MCALIB_SetParameterValue(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);

int32_t MCALIB_SetEnergySpectrumInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);
int32_t MCALIB_GetEnergySpectrumInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);
int32_t MCALIB_ClearEnergySpectrum(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);
int32_t MCALIB_ResetGainStabilizer(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);

int32_t MCALIB_GetROIInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);

int32_t MCALIB_SetDTSpectrumInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);
int32_t MCALIB_GetDTSpectrumInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);
int32_t MCALIB_ClearDTSpectrum(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);

int32_t MCALIB_GetWaveform(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);

int32_t MCALIB_SetListMode(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);
int32_t MCALIB_GetListMode(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);

int32_t MCALIB_GetMCSSpectrumInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);
int32_t MCALIB_MCSSweep(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);
int32_t MCALIB_ClearMCSSpectrum(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);

int32_t MCALIB_GetRegisterValue(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);
int32_t MCALIB_SetRegisterValue(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);

int32_t MCALIB_AcquisitionStart(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);
int32_t MCALIB_AcquisitionStop(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);

int32_t MCALIB_DatabasePath(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);
int32_t MCALIB_SaveConfiguration(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);
int32_t MCALIB_LoadConfiguration(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);
int32_t MCALIB_GetConfigurationList(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);
int32_t MCALIB_DeleteConfiguration(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);

int32_t MCALIB_ParamAutoSetStart(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);
int32_t MCALIB_ParamAutoSetStop(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);

int32_t MCALIB_HVOn(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);
int32_t MCALIB_HVOff(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);
int32_t MCALIB_HVOnOff(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);

int32_t MCALIB_SampleAdvance(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);

int32_t MCALIB_GetHandleInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args);

int32_t MCALIB_GlobalQuit(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);

int32_t MCALIB_RTClock(const MCALIB_Handle_t* handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args);

// "fmt" included in the "__VA_ARGS__" to avoid warnings in "gcc -std=c99 -pedantic" complaing about at least one variadic argument
int32_t MCALIB_extendedLogMsg(const c_locallogger_t *locallogger, int line, const MCALIB_Handle_t *handle, c_logger_Severity sev, const char *fmt, ...) c_attribute_format(5, 6);
#define MCALIB_LogMsgExt(HNDL, SEV, ...) MCALIB_extendedLogMsg(&__locallogger, __LINE__, HNDL, SEV, __VA_ARGS__)

void MCALIB_DecodeError(const MCALIB_Handle_t *handle, const char *functName, int32_t resCode);
int32_t MCALIB_TranslateError(int32_t srverr);

#endif // _MCALIB_COMMON_H
