#ifndef _WIN32
#include <dlfcn.h>
#include <fcntl.h>
#endif

#include <CAENMCA.h>

#include <MCALIB_Common.h>
#include <MCALIB_Discovery.h>

#include <CAENMultiplatform.h>
#include <CAENSerDes.h>
#include <CAENUrl.h>
#include <CAENThread.h>
#include <CAENRandom.h>

MCALIB_Library_t *Library;

INIT_C_LOGGER("CAENMCALog.txt", "CAENMCA.c");

#define MCALIB_RESOURCES_DIR "mcalib_resources"
#ifdef _WIN32
#define MCALIB_RESOURCES_ARCH_DIR MCALIB_RESOURCES_DIR
#define HEXAGONSERVERNAME "HexagonServer.exe"
#else
// NOTE: in the following define, ARCH_ABI must be defined from preprocessor AS STRING (quoted)
#define MCALIB_RESOURCES_ARCH_DIR MCALIB_RESOURCES_DIR DIRSEP ARCH_ABI
#define HEXAGONSERVERNAME "HexagonServer"
#endif

DEFINE_MCALIB_CONSTANTS();

static inline int32_t _getBoardInfo(MCALIB_Device_t *device) {
	device->DeviceInfo.nDTSpectra = 0;
	//get nDTSpectra if supported, if not it is set to 0
	CAEN_MCA_GetData(device->commonData.handle, CAEN_MCA_DATA_BOARD_INFO, DATAMASK_BRDINFO_NDTSPECTRA, &device->DeviceInfo.nDTSpectra);
		
	return CAEN_MCA_GetData(
		device->commonData.handle,
		CAEN_MCA_DATA_BOARD_INFO,
		DATAMASK_BRDINFO_MODELNAME |
		DATAMASK_BRDINFO_MODEL |
		DATAMASK_BRDINFO_NCHANNELS |
		DATAMASK_BRDINFO_SERIALNUM |
		DATAMASK_BRDINFO_NHVCHANNELS |
		DATAMASK_BRDINFO_NLVDSGROUPS |
		DATAMASK_BRDINFO_NTRACES |
		DATAMASK_BRDINFO_NMONOUTS |
		DATAMASK_BRDINFO_LIB_VERSION,
		device->DeviceInfo.ModelName,
		&device->DeviceInfo.Model,
		&device->DeviceInfo.nChannels,
		&device->DeviceInfo.SerialNum,
		&device->DeviceInfo.nHVChannels,
		&device->DeviceInfo.nLVDSGroups,
		&device->DeviceInfo.nVirtualTraces,
		&device->DeviceInfo.nMonOuts,
		device->DeviceInfo.LibVersion);
}

static inline int32_t _getChannelInfo(MCALIB_Channel_t *channel) {
	return CAEN_MCA_GetData(
		channel->commonData.handle,
		CAEN_MCA_DATA_CHANNEL_INFO,
		DATAMASK_CHANNELINFO_NENERGYSPECTRA |
		DATAMASK_CHANNELINFO_NMCSSPECTRA,
		&channel->nEnergySpectra,
		&channel->nMCSSpectra
	);
}

static inline int32_t _getEnergySpectrumInfo(MCALIB_EnergySpectrum_t *spectrum) {
	return CAEN_MCA_GetData(
		spectrum->commonData.handle,
		CAEN_MCA_DATA_ENERGYSPECTRUM,
		DATAMASK_ENERGY_SPECTRUM_NROIS,
		&spectrum->nROIs
	);
}

static inline int32_t _getHVChannelInfo(MCALIB_HVChannel_t *hvchannel) {
	return CAEN_MCA_GetData(
		hvchannel->commonData.handle,
		CAEN_MCA_DATA_HVCHANNEL_INFO,
		DATAMASK_HVCHANNELINFO_NRANGES,
		&hvchannel->nRanges
	);
}

static inline int32_t _getHVRangeInfo(MCALIB_HVRange_t *hvrange) {
	return CAEN_MCA_GetData(
		hvrange->commonData.handle,
		CAEN_MCA_DATA_HVRANGE_INFO,
		DATAMASK_HVRANGEINFO_CODENAME,
		hvrange->commonData.name
	);
}

static int32_t _checkVersion(const char *server_version) {
	uint16_t server_major, server_minor, server_patch;
	const char *library_version = CAEN_MCA_VERSION;
	uint16_t library_major = UINT16_C(CAEN_MCA_VERSION_MAJOR);
	uint16_t library_minor = UINT16_C(CAEN_MCA_VERSION_MINOR);

	int found = sscanf(server_version, "%"SCNu16".%"SCNu16".%"SCNu16, &server_major, &server_minor, &server_patch);

	if (found != 3)
		return CAEN_MCA_RetCode_Argument;

	if (library_major != server_major) {
		logMsg(c_logger_Severity_ERROR, "%s(): library version used by server (%s) not compatible with the current version (%s). Major version must be identical.", __func__, server_version, library_version);
		return CAEN_MCA_RetCode_NotSupported;
	}
	else if (library_minor > server_minor) {
		logMsg(c_logger_Severity_WARNING, "%s(): library version used by server (%s) differs from the current version (%s). Some features could be not supported by the server.", __func__, server_version, library_version);
	}
	else if (library_minor < server_minor) {
		logMsg(c_logger_Severity_WARNING, "%s(): library version used by server (%s) differs from the current version (%s). Some features could be not supported by the library.", __func__, server_version, library_version);
	}
	return CAEN_MCA_RetCode_Success;
}

static void _freeParameter(MCALIB_Parameter_t *dest) {
	if (dest != NULL) {
		c_free(dest->commonData.handle);
		c_free(dest);
	}
}

static xmlNode *_getParametersNodeOfEntity(const MCALIB_Handle_t *handle) {
	const char *entity_key = NULL;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	xmlNode *result;

	if (handle == NULL || device == NULL) {
		logMsg(c_logger_Severity_ERROR, "Invalid or NULL handle.");
		return NULL;
	}

	switch (handle->commonData->type->typeCode) {
	case CAEN_MCA_HANDLE_DEVICE:
		entity_key = XML_BOARD_PARAMS_KEY;
		break;
	case CAEN_MCA_HANDLE_CHANNEL:
		entity_key = XML_CHANNEL_PARAMS_KEY;
		break;
	case CAEN_MCA_HANDLE_ENERGYSPECTRUM:
		entity_key = XML_ENERGYSPECTRUM_PARAMS_KEY;
		break;
	case CAEN_MCA_HANDLE_MCSSPECTRUM:
		entity_key = XML_MCSSPECTRUM_PARAMS_KEY;
		break;
	case CAEN_MCA_HANDLE_ROI:
		entity_key = XML_ROI_PARAMS_KEY;
		break;
	case CAEN_MCA_HANDLE_HVCHANNEL:
		entity_key = XML_HVCHANNEL_PARAMS_KEY;
		break;
	case CAEN_MCA_HANDLE_HVRANGE:
		entity_key = XML_HVRANGE_PARAMS_KEY;
		break;
	case CAEN_MCA_HANDLE_LVDSGROUP:
		entity_key = XML_LVDSGROUP_PARAMS_KEY;
		break;
	case CAEN_MCA_HANDLE_TRACE:
		entity_key = XML_VIRTUALTRACE_PARAMS_KEY;
		break;
	case CAEN_MCA_HANDLE_MONOUT:
		entity_key = XML_MONOUT_PARAMS_KEY;
		break;
	case CAEN_MCA_HANDLE_DTSPECTRUM:
		entity_key = XML_DTSPECTRUM_PARAMS_KEY;
		break;
	default:
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): Unsupported handle type '%s'.", __func__, handle->commonData->type->typeName);
		return NULL;
	}

	const char *format = "ServerParameters/%s";

	int needed = snprintf(NULL, 0, format, entity_key);

	char *nodePath = c_malloc(needed + 1);
	if (nodePath == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): Can't create nodePath.", __func__);
		return NULL;
	}

	sprintf(nodePath, format, entity_key);

	result = c_xml_getfirstdescendantelementfromname(c_xml_getrootnode(device->definitions), nodePath);
	if (result == NULL)
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): Can't find node with path '%s'.", __func__, nodePath);

	c_free(nodePath);

	return result;
}

typedef struct {
	char codename[PARAMINFO_NAME_MAXLEN_DEF];
	int32_t code;
} MCALIB_ParamsCodenameAndCode_t;

typedef struct {
	MCALIB_ParamsCodenameAndCode_t **codenamesAndCodes;
	uint32_t length;
} MCALIB_ParamsCodenameAndCodeCollection_t;

static MCALIB_ParamsCodenameAndCode_t *_createCodenameAndCode(char *codename, int32_t code) {
	if (codename == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): 'codename' cannot be NULL.", __func__);
		return NULL;
	}

	MCALIB_ParamsCodenameAndCode_t *res = c_malloc(sizeof(*res));
	if (res == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): can't allocate a '_MCALIB_ParamsCodenameAndCode_t'.", __func__);
		return res;
	}

	res->codename[0] = '\0';
	strncat(res->codename, codename, sizeof(res->codename) - 1);
	res->code = code;
	return res;
}

static void _freeCodenameAndCode(MCALIB_ParamsCodenameAndCode_t *buff) {
	c_free(buff);
}

static MCALIB_ParamsCodenameAndCode_t *_createCodenameAndCodeFromXmlNode(xmlNode *node) {
	char *codename = (char*)(node->name);
	char *end;
	const char *code_str = c_xml_getattributevalue(node, "code", FALSE); // get attribute code

	if (code_str == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): code '%s' doesn't have attribute 'code'", __func__, codename);
		return NULL;
	}

	// parse code & check
	int32_t code = (int32_t)strtol(code_str, &end, 0);
	if (end == code_str) {
		logMsg(c_logger_Severity_ERROR, "%s(): can't parse parameter code from string '%s' for parameter '%s'", __func__, code_str, codename);
		return NULL;
	}

	return _createCodenameAndCode(codename, code);
}

static int32_t _addCodenameAndCode(MCALIB_ParamsCodenameAndCodeCollection_t *coll, MCALIB_ParamsCodenameAndCode_t *obj) {
	MCALIB_ParamsCodenameAndCode_t **tmp = c_realloc(coll->codenamesAndCodes, (coll->length + 1) * sizeof(*tmp));
	if (tmp == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): can't realloc CodenamesAndCodes.", __func__);
		return CAEN_MCA_RetCode_OutOfMemory;
	}
	tmp[coll->length] = obj;
	coll->codenamesAndCodes = tmp;
	coll->length++;
	return CAEN_MCA_RetCode_Success;
}

static void _freeAllCodenamesAndCodes(MCALIB_ParamsCodenameAndCodeCollection_t *coll) {
	if (coll != NULL) {
		if (coll->codenamesAndCodes != NULL) {
			for (uint32_t i = 0; i < coll->length; i++)
				_freeCodenameAndCode(coll->codenamesAndCodes[i]);
			c_free(coll->codenamesAndCodes);
			coll->codenamesAndCodes = NULL;
		}
		coll->length = 0;
	}
}

static int32_t _getAllEntityCodenamesAndCodes(const MCALIB_Handle_t *handle, MCALIB_ParamsCodenameAndCodeCollection_t *dest) {
	xmlNode *parametersNode = _getParametersNodeOfEntity(handle);

	if (parametersNode == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): can't get parameters node.", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	for (xmlNode *node = parametersNode->children; node != NULL; node = node->next) {
		if (node->type != XML_COMMENT_NODE) {
			MCALIB_ParamsCodenameAndCode_t *obj = _createCodenameAndCodeFromXmlNode(node);
			if (obj != NULL) {
				int32_t res = _addCodenameAndCode(dest, obj);
				if (res != CAEN_MCA_RetCode_Success) {
					_freeCodenameAndCode(obj);
					_freeAllCodenamesAndCodes(dest);
					return res;
				}
			}
		}
	}
	return CAEN_MCA_RetCode_Success;
}

static int32_t _createParameterHandle(MCALIB_Handle_t *parentHandle, int32_t code, char *codename, MCALIB_Parameter_t **dest) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	double value = 0.;

	MCALIB_Parameter_t *parameter = MCALIB_CreateObject(&MCALIB_Handle_Parameter, parentHandle, code);
	if (parameter == NULL) {
		MCALIB_LogMsgExt(parentHandle, c_logger_Severity_ERROR, "%s(): can't create parameter with code %"PRIi32".", __func__, code);
		return CAEN_MCA_RetCode_Generic;
	}

	// Initialize additional data after commonData.
	parameter->parameters = NULL;

	// Overwrite default object name with the codename.
	snprintf(parameter->commonData.name, sizeof(parameter->commonData.name), "%s", codename);

	// try get parameter from board. If success, create parameter.
	ret = CAEN_MCA_GetData(parameter->commonData.handle, CAEN_MCA_DATA_PARAMETER_VALUE, DATAMASK_VALUE_NUMERIC, &value);
	if (ret != CAEN_MCA_RetCode_Success) {
		// Set success!
		ret = CAEN_MCA_RetCode_Success;
		_freeParameter(parameter);
		parameter = NULL;
	}

	*dest = parameter;

	return ret;
}

static int32_t _createParameters(MCALIB_Handle_t *parent, MCALIB_HandleCollection_t **collection) {
	if ((*collection = MCALIB_CreateHandleCollection(parent, CAEN_MCA_HANDLE_PARAMETER)) == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create an handle collection", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	// Get all supportable parameters
	MCALIB_ParamsCodenameAndCodeCollection_t codenamesAndCodes = { NULL, 0 };
	int32_t ret = _getAllEntityCodenamesAndCodes(parent, &codenamesAndCodes);
	if (ret != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): error %"PRIi32" occurred.", __func__, ret);
		return ret;
	}

	if (codenamesAndCodes.length == 0 || codenamesAndCodes.codenamesAndCodes == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_DEBUG, "%s(): this entity doesn't seem to support any parameter. Going on.", __func__);
		goto QuitFunction;
	}

	for (uint32_t i = 0; i < codenamesAndCodes.length; i++) {
		MCALIB_ParamsCodenameAndCode_t *codenameAndCode = codenamesAndCodes.codenamesAndCodes[i];
		if (codenameAndCode == NULL)
			continue;

		// Create objects and check if the device supports the parameters.
		MCALIB_Parameter_t *parameter = NULL;
		if ((ret = _createParameterHandle((*collection)->commonData.handle, codenameAndCode->code, codenameAndCode->codename, &parameter)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): can't create parameter with codename '%s' and code %"PRIi32".", __func__, codenameAndCode->codename, codenameAndCode->code);
			goto QuitFunction;
		}

		// Add to collection only if supported
		if (parameter != NULL) {
			if ((ret = MCALIB_AddHandleTo(*collection, parameter->commonData.handle)) != CAEN_MCA_RetCode_Success) {
				MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): error adding parameter '%s' handle to collection.", __func__, codenameAndCode->codename);
				goto QuitFunction;
			}
		}
	}
QuitFunction:
	_freeAllCodenamesAndCodes(&codenamesAndCodes);
	return ret;
}

static int32_t _createROIs(MCALIB_Handle_t *parent, MCALIB_HandleCollection_t **collection, uint32_t length) {
	if ((*collection = MCALIB_CreateHandleCollection(parent, CAEN_MCA_HANDLE_ROI)) == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create an handle collection", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	for (uint32_t i = 0; i < length; i++) {
		int32_t ret;
		MCALIB_ROI_t *object = MCALIB_CreateObject(&MCALIB_Handle_ROI, (*collection)->commonData.handle, i);
		if (object == NULL) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
			return CAEN_MCA_RetCode_Generic;
		}

		// Initialize additional data after commonData.
		object->parameters = NULL;

		if ((ret = MCALIB_AddHandleTo(*collection, object->commonData.handle)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Error adding parameter handle to collection.");
			return ret;
		}

		// Create children
		if ((ret = _createParameters(object->commonData.handle, &object->parameters)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create parameters.");
			return ret;
		}
	}

	return CAEN_MCA_RetCode_Success;
}

static int32_t _createEnergySpectra(MCALIB_Handle_t *parent, MCALIB_HandleCollection_t **collection, uint32_t length) {
	if ((*collection = MCALIB_CreateHandleCollection(parent, CAEN_MCA_HANDLE_ENERGYSPECTRUM)) == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create an handle collection", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	for (uint32_t i = 0; i < length; i++) {
		int32_t ret;
		MCALIB_EnergySpectrum_t *object = MCALIB_CreateObject(&MCALIB_Handle_EnergySpectrum, (*collection)->commonData.handle, i);
		if (object == NULL) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
			return CAEN_MCA_RetCode_Generic;
		}

		// Initialize additional data after commonData.
		object->parameters = NULL;
		object->rois = NULL;
		object->nROIs = 0;

		if ((ret = MCALIB_AddHandleTo(*collection, object->commonData.handle)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Error adding parameter handle to collection.");
			return ret;
		}

		if ((ret = _getEnergySpectrumInfo(object)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(object->commonData.handle, c_logger_Severity_ERROR, "%s(): can't get channel info from device.", __func__);
			return ret;
		}

		// Create children
		if ((ret = _createROIs(object->commonData.handle, &object->rois, object->nROIs)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create MCS spectra.");
			return ret;
		}

		if ((ret = _createParameters(object->commonData.handle, &object->parameters)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create parameters.");
			return ret;
		}
	}

	return CAEN_MCA_RetCode_Success;
}

static int32_t _createMCSSpectra(MCALIB_Handle_t *parent, MCALIB_HandleCollection_t **collection, uint32_t length) {
	if ((*collection = MCALIB_CreateHandleCollection(parent, CAEN_MCA_HANDLE_MCSSPECTRUM)) == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create an handle collection", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	for (uint32_t i = 0; i < length; i++) {
		int32_t ret;
		MCALIB_MCSSpectrum_t *object = MCALIB_CreateObject(&MCALIB_Handle_MCSSpectrum, (*collection)->commonData.handle, i);
		if (object == NULL) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
			return CAEN_MCA_RetCode_Generic;
		}

		// Initialize additional data after commonData.
		object->parameters = NULL;

		if ((ret = MCALIB_AddHandleTo(*collection, object->commonData.handle)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Error adding parameter handle to collection.");
			return ret;
		}

		// Create children
		if ((ret = _createParameters(object->commonData.handle, &object->parameters)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create parameters.");
			return ret;
		}
	}

	return CAEN_MCA_RetCode_Success;
}

static int32_t _createChannels(MCALIB_Handle_t *parent, MCALIB_HandleCollection_t **collection, uint32_t length) {
	if ((*collection = MCALIB_CreateHandleCollection(parent, CAEN_MCA_HANDLE_CHANNEL)) == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create an handle collection", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	for (uint32_t i = 0; i < length; i++) {
		int32_t ret;
		MCALIB_Channel_t *object = MCALIB_CreateObject(&MCALIB_Handle_Channel, (*collection)->commonData.handle, i);
		if (object == NULL) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
			return CAEN_MCA_RetCode_Generic;
		}

		// Initialize additional data after commonData.
		object->parameters = NULL;
		object->energyspectra = NULL;
		object->mcsspectra = NULL;
		object->nEnergySpectra = 0;
		object->nMCSSpectra = 0;

		if ((ret = MCALIB_AddHandleTo(*collection, object->commonData.handle)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Error adding parameter handle to collection.");
			return ret;
		}

		if ((ret = _getChannelInfo(object)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(object->commonData.handle, c_logger_Severity_ERROR, "%s(): can't get channel info from device.", __func__);
			return ret;
		}

		// Create children
		if ((ret = _createEnergySpectra(object->commonData.handle, &object->energyspectra, object->nEnergySpectra)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create energy spectra.");
			return ret;
		}

		if ((ret = _createMCSSpectra(object->commonData.handle, &object->mcsspectra, object->nMCSSpectra)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create MCS spectra.");
			return ret;
		}

		if ((ret = _createParameters(object->commonData.handle, &object->parameters)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create parameters.");
			return ret;
		}
	}

	return CAEN_MCA_RetCode_Success;
}

static int32_t _createHVRanges(MCALIB_Handle_t *parent, MCALIB_HandleCollection_t **collection, uint32_t length) {
	if ((*collection = MCALIB_CreateHandleCollection(parent, CAEN_MCA_HANDLE_HVRANGE)) == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create an handle collection", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	for (uint32_t i = 0; i < length; i++) {
		int32_t ret;
		MCALIB_HVRange_t *object = MCALIB_CreateObject(&MCALIB_Handle_HVRange, (*collection)->commonData.handle, i);
		if (object == NULL) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
			return CAEN_MCA_RetCode_Generic;
		}

		// Initialize additional data after commonData.
		object->parameters = NULL;

		if ((ret = _getHVRangeInfo(object)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Cannot get HV range info from device.");
			return ret;
		}

		if ((ret = MCALIB_AddHandleTo(*collection, object->commonData.handle)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Error adding parameter handle to collection.");
			return ret;
		}

		// Create children
		if ((ret = _createParameters(object->commonData.handle, &object->parameters)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create parameters.");
			return ret;
		}
	}

	return CAEN_MCA_RetCode_Success;
}

static int32_t _createHVChannels(MCALIB_Handle_t *parent, MCALIB_HandleCollection_t **collection, uint32_t length) {
	if ((*collection = MCALIB_CreateHandleCollection(parent, CAEN_MCA_HANDLE_HVCHANNEL)) == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create an handle collection", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	for (uint32_t i = 0; i < length; i++) {
		int32_t ret;
		MCALIB_HVChannel_t *object = MCALIB_CreateObject(&MCALIB_Handle_HVChannel, (*collection)->commonData.handle, i);
		if (object == NULL) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
			return CAEN_MCA_RetCode_Generic;
		}

		// Initialize additional data after commonData.
		object->parameters = NULL;
		object->hvranges = NULL;
		object->nRanges = 0;

		if ((ret = MCALIB_AddHandleTo(*collection, object->commonData.handle)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Error adding parameter handle to collection.");
			return ret;
		}

		if ((ret = _getHVChannelInfo(object)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(object->commonData.handle, c_logger_Severity_ERROR, "%s(): can't get HV channel info from device.", __func__);
			return ret;
		}

		// Create children
		if ((ret = _createHVRanges(object->commonData.handle, &object->hvranges, object->nRanges)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create HV ranges.");
			return ret;
		}

		if ((ret = _createParameters(object->commonData.handle, &object->parameters)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create parameters.");
			return ret;
		}
	}

	return CAEN_MCA_RetCode_Success;
}

static int32_t _createLVDSGroups(MCALIB_Handle_t *parent, MCALIB_HandleCollection_t **collection, uint32_t length) {
	if ((*collection = MCALIB_CreateHandleCollection(parent, CAEN_MCA_HANDLE_LVDSGROUP)) == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create an handle collection", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	for (uint32_t i = 0; i < length; i++) {
		int32_t ret;
		MCALIB_LVDSGroup_t *object = MCALIB_CreateObject(&MCALIB_Handle_LVDSGroup, (*collection)->commonData.handle, i);
		if (object == NULL) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
			return CAEN_MCA_RetCode_Generic;
		}

		// Initialize additional data after commonData.
		object->parameters = NULL;

		if ((ret = MCALIB_AddHandleTo(*collection, object->commonData.handle)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Error adding parameter handle to collection.");
			return ret;
		}

		// Create children
		if ((ret = _createParameters(object->commonData.handle, &object->parameters)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create parameters.");
			return ret;
		}
	}

	return CAEN_MCA_RetCode_Success;
}

static int32_t _createTraces(MCALIB_Handle_t *parent, MCALIB_HandleCollection_t **collection, uint32_t length) {
	if ((*collection = MCALIB_CreateHandleCollection(parent, CAEN_MCA_HANDLE_TRACE)) == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create an handle collection", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	for (uint32_t i = 0; i < length; i++) {
		int32_t ret;
		MCALIB_Trace_t *object = MCALIB_CreateObject(&MCALIB_Handle_Trace, (*collection)->commonData.handle, i);
		if (object == NULL) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
			return CAEN_MCA_RetCode_Generic;
		}

		// Initialize additional data after commonData.
		object->parameters = NULL;

		if ((ret = MCALIB_AddHandleTo(*collection, object->commonData.handle)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Error adding parameter handle to collection.");
			return ret;
		}

		// Create children
		if ((ret = _createParameters(object->commonData.handle, &object->parameters)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create parameters.");
			return ret;
		}
	}

	return CAEN_MCA_RetCode_Success;
}

static int32_t _createMonOuts(MCALIB_Handle_t *parent, MCALIB_HandleCollection_t **collection, uint32_t length) {
	if ((*collection = MCALIB_CreateHandleCollection(parent, CAEN_MCA_HANDLE_MONOUT)) == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create an handle collection", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	for (uint32_t i = 0; i < length; i++) {
		int32_t ret;
		MCALIB_MonOut_t *object = MCALIB_CreateObject(&MCALIB_Handle_MonOut, (*collection)->commonData.handle, i);
		if (object == NULL) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
			return CAEN_MCA_RetCode_Generic;
		}

		// Initialize additional data after commonData.
		object->parameters = NULL;

		if ((ret = MCALIB_AddHandleTo(*collection, object->commonData.handle)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Error adding parameter handle to collection.");
			return ret;
		}

		// Create children
		if ((ret = _createParameters(object->commonData.handle, &object->parameters)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create parameters.");
			return ret;
		}
	}

	return CAEN_MCA_RetCode_Success;
}

static int32_t _createDTSpectra(MCALIB_Handle_t *parent, MCALIB_HandleCollection_t **collection, uint32_t length) {
	if ((*collection = MCALIB_CreateHandleCollection(parent, CAEN_MCA_HANDLE_DTSPECTRUM)) == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create an handle collection", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	for (uint32_t i = 0; i < length; i++) {
		int32_t ret;
		MCALIB_DTSpectrum_t *object = MCALIB_CreateObject(&MCALIB_Handle_DTSpectrum, (*collection)->commonData.handle, i);
		if (object == NULL) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
			return CAEN_MCA_RetCode_Generic;
		}

		// Initialize additional data after commonData.
		object->parameters = NULL;

		if ((ret = MCALIB_AddHandleTo(*collection, object->commonData.handle)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Error adding parameter handle to collection.");
			return ret;
		}

		if ((ret = _createParameters(object->commonData.handle, &object->parameters)) != CAEN_MCA_RetCode_Success) {
			MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "Can't create parameters.");
			return ret;
		}
	}

	return CAEN_MCA_RetCode_Success;
}

static void _freeParameters(MCALIB_HandleCollection_t *collection) {
	if (collection == NULL)
		return;
	uint32_t length;
	if (MCALIB_GetHandleCollectionLength(collection, &length) != CAEN_MCA_RetCode_Success)
		return;
	for (uint32_t i = 0; i < length; i++) {
		MCALIB_Handle_t *handle = MCALIB_GetHandleAt(collection, i);
		MCALIB_Parameter_t *parameter = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_PARAMETER);

		// Free children
		// NO CHILDREN

		MCALIB_DeleteObject(parameter);
	}
	MCALIB_DeleteHandleCollection(collection);
}

static void _freeTraces(MCALIB_HandleCollection_t *collection) {
	if (collection == NULL)
		return;
	uint32_t length;
	if (MCALIB_GetHandleCollectionLength(collection, &length) != CAEN_MCA_RetCode_Success)
		return;
	for (uint32_t i = 0; i < length; i++) {
		MCALIB_Handle_t *handle = MCALIB_GetHandleAt(collection, i);
		MCALIB_Trace_t *trace = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_TRACE);

		// Free children
		_freeParameters(trace->parameters);

		MCALIB_DeleteObject(trace);
	}
	MCALIB_DeleteHandleCollection(collection);
}

static void _freeMonOuts(MCALIB_HandleCollection_t *collection) {
	if (collection == NULL)
		return;
	uint32_t length;
	if (MCALIB_GetHandleCollectionLength(collection, &length) != CAEN_MCA_RetCode_Success)
		return;
	for (uint32_t i = 0; i < length; i++) {
		MCALIB_Handle_t *handle = MCALIB_GetHandleAt(collection, i);
		MCALIB_MonOut_t *monout = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_MONOUT);

		// Free children
		_freeParameters(monout->parameters);

		MCALIB_DeleteObject(monout);
	}
	MCALIB_DeleteHandleCollection(collection);
}

static void _freeLVDSGroups(MCALIB_HandleCollection_t *collection) {
	if (collection == NULL)
		return;
	uint32_t length;
	if (MCALIB_GetHandleCollectionLength(collection, &length) != CAEN_MCA_RetCode_Success)
		return;
	for (uint32_t i = 0; i < length; i++) {
		MCALIB_Handle_t *handle = MCALIB_GetHandleAt(collection, i);
		MCALIB_LVDSGroup_t *lvsdgroup = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_LVDSGROUP);

		// Free children
		_freeParameters(lvsdgroup->parameters);

		MCALIB_DeleteObject(lvsdgroup);
	}
	MCALIB_DeleteHandleCollection(collection);
}

static void _freeROIs(MCALIB_HandleCollection_t *collection) {
	if (collection == NULL)
		return;
	uint32_t length;
	if (MCALIB_GetHandleCollectionLength(collection, &length) != CAEN_MCA_RetCode_Success)
		return;
	for (uint32_t i = 0; i < length; i++) {
		MCALIB_Handle_t *handle = MCALIB_GetHandleAt(collection, i);
		MCALIB_ROI_t *roi = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_ROI);

		// Free children
		_freeParameters(roi->parameters);

		MCALIB_DeleteObject(roi);
	}
	MCALIB_DeleteHandleCollection(collection);
}


static void _freeEnergySpectra(MCALIB_HandleCollection_t *collection) {
	if (collection == NULL)
		return;
	uint32_t length;
	if (MCALIB_GetHandleCollectionLength(collection, &length) != CAEN_MCA_RetCode_Success)
		return;
	for (uint32_t i = 0; i < length; i++) {
		MCALIB_Handle_t *handle = MCALIB_GetHandleAt(collection, i);
		MCALIB_EnergySpectrum_t *spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_ENERGYSPECTRUM);

		// Free children
		_freeParameters(spectrum->parameters);
		_freeROIs(spectrum->rois);

		MCALIB_DeleteObject(spectrum);
	}
	MCALIB_DeleteHandleCollection(collection);
}

static void _freeMCSSpectra(MCALIB_HandleCollection_t *collection) {
	if (collection == NULL)
		return;
	uint32_t length;
	if (MCALIB_GetHandleCollectionLength(collection, &length) != CAEN_MCA_RetCode_Success)
		return;
	for (uint32_t i = 0; i < length; i++) {
		MCALIB_Handle_t *handle = MCALIB_GetHandleAt(collection, i);
		MCALIB_MCSSpectrum_t *spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_MCSSPECTRUM);

		// Free children
		_freeParameters(spectrum->parameters);

		MCALIB_DeleteObject(spectrum);
	}
	MCALIB_DeleteHandleCollection(collection);
}

static void _freeDTSpectra(MCALIB_HandleCollection_t *collection) {
	if (collection == NULL)
		return;
	uint32_t length;
	if (MCALIB_GetHandleCollectionLength(collection, &length) != CAEN_MCA_RetCode_Success)
		return;
	for (uint32_t i = 0; i < length; i++) {
		MCALIB_Handle_t *handle = MCALIB_GetHandleAt(collection, i);
		MCALIB_DTSpectrum_t *dt_spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DTSPECTRUM);

		// Free children
		_freeParameters(dt_spectrum->parameters);

		MCALIB_DeleteObject(dt_spectrum);
	}
	MCALIB_DeleteHandleCollection(collection);
}

static void _freeChannels(MCALIB_HandleCollection_t *collection) {
	if (collection == NULL)
		return;
	uint32_t length;
	if (MCALIB_GetHandleCollectionLength(collection, &length) != CAEN_MCA_RetCode_Success)
		return;
	for (uint32_t i = 0; i < length; i++) {
		MCALIB_Handle_t *handle = MCALIB_GetHandleAt(collection, i);
		MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);

		// Free children
		_freeParameters(channel->parameters);
		_freeEnergySpectra(channel->energyspectra);
		_freeMCSSpectra(channel->mcsspectra);

		MCALIB_DeleteObject(channel);
	}
	MCALIB_DeleteHandleCollection(collection);
}

static void _freeHVRanges(MCALIB_HandleCollection_t *collection) {
	if (collection == NULL)
		return;
	uint32_t length;
	if (MCALIB_GetHandleCollectionLength(collection, &length) != CAEN_MCA_RetCode_Success)
		return;
	for (uint32_t i = 0; i < length; i++) {
		MCALIB_Handle_t *handle = MCALIB_GetHandleAt(collection, i);
		MCALIB_HVRange_t *range = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_HVRANGE);

		// Free children
		_freeParameters(range->parameters);

		MCALIB_DeleteObject(range);
	}
	MCALIB_DeleteHandleCollection(collection);
}

static void _freeHVChannels(MCALIB_HandleCollection_t *collection) {
	if (collection == NULL)
		return;
	uint32_t length;
	if (MCALIB_GetHandleCollectionLength(collection, &length) != CAEN_MCA_RetCode_Success)
		return;
	for (uint32_t i = 0; i < length; i++) {
		MCALIB_Handle_t *handle = MCALIB_GetHandleAt(collection, i);
		MCALIB_HVChannel_t *hvchannel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_HVCHANNEL);

		// Free children
		_freeParameters(hvchannel->parameters);
		_freeHVRanges(hvchannel->hvranges);

		MCALIB_DeleteObject(hvchannel);
	}
	MCALIB_DeleteHandleCollection(collection);
}

static void _freeDevice(MCALIB_Device_t *device) {
	MCALIB_RemovableHandleCollection_t* collection = Library->devices;
	
	if (device == NULL)
		return;
	
	if (collection == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): device collection cannot be NULL.", __func__);
		return;
	}

	// Free children
	_freeMonOuts(device->monouts);
	_freeTraces(device->traces);
	_freeLVDSGroups(device->lvdsgroups);
	_freeChannels(device->channels);
	_freeHVChannels(device->hvchannels);
	_freeDTSpectra(device->dtspectra);
	_freeParameters(device->parameters);
	
	// Free other stuff
	c_xml_freefile(device->definitions);
	c_socket_delete(device->socket);

	MCALIB_RemoveHandleFrom(collection, device->commonData.handle);

	MCALIB_DeleteObject(device);
}

static int32_t _openDeviceEthParams(const char* host, uint16_t serverport, MCALIB_Device_t *dev) {
	if (c_socket_client_connect(&dev->socket, host, serverport) != c_Socket_ErrorCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't connect to board at host '%s:%" PRIu16 "'.", __func__, host, serverport);
		return CAEN_MCA_RetCode_SockInit;
	}

	return CAEN_MCA_RetCode_Success;
}

static int32_t _openDeviceEth(const c_url_field_t *url, MCALIB_Device_t *dev) {
	uint16_t serverport = CAEN_MCA_DEFAULT_TCP_PORT_DEF;

	// Port is optional. If not provided, the default is used.
	if (url->port != NULL)
		serverport = (uint16_t)strtoul(url->port, NULL, 0);

	return _openDeviceEthParams(url->host, serverport, dev);
}

static int32_t handleQueryKey(const MCALIB_Device_t *dev, const char *queryName, const char *queryValue, const char* key, const char** value) {
	if (strcmp(queryName, key) == 0) {
		if (*value != NULL) {
			MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): Given path contains query '%s' multiple times.", __func__, queryName);
			return CAEN_MCA_RetCode_Argument;
		}
		*value = queryValue;
	}
	return CAEN_MCA_RetCode_Success;
}

/** Returns true on success, or false if there was an error */
static bool _setSocketBlockingEnabled(int fd, bool blocking) {
	if (fd < 0) return false;

#ifdef _WIN32
	unsigned long mode = blocking ? 0 : 1;
	return (ioctlsocket(fd, FIONBIO, &mode) == 0) ? true : false;
#else
	int flags = fcntl(fd, F_GETFL, 0);
	if (flags == -1) return false;
	flags = blocking ? (flags & ~O_NONBLOCK) : (flags | O_NONBLOCK);
	return (fcntl(fd, F_SETFL, flags) == 0) ? true : false;
#endif
}

static int32_t _waitAndGetServerReply(MCALIB_Device_t *dev, c_socket_t *notify_socket, uint16_t *port) {
	// read the rescode/port from socket.
	// NOTE: we assume that notify_socket is non-blocking, so we select before on it using a timeout
	// in order not to get stuck on this function if for some reason nobody connects.
	fd_set readmask;
	struct timeval sel_to;

	FD_ZERO(&readmask);
	FD_SET(notify_socket->socket, &readmask);
	sel_to.tv_sec = 20; // 20 s
	sel_to.tv_usec = 0;

	do {
		int32_t selret = select((int32_t)(notify_socket->socket + 1), &readmask, NULL, NULL, &sel_to);
		if (selret == -1) { // ERROR
			logMsg(c_logger_Severity_ERROR, "'select' error on notify_socket.");
			return CAEN_MCA_RetCode_Generic;
		}
		else if (selret == 0) { // TIMEOUT
			logMsg(c_logger_Severity_DEBUG, "'select' timeout on notify_socket.");
			return CAEN_MCA_RetCode_Generic;
		}
	} while (FD_ISSET(notify_socket->socket, &readmask) == 0);

	// Now we should be able to accept. If between the select and the accept the connection
	// closes, accept should return an error immediately, because it is non-blocking.
	c_socket_t *client_notify_socket;
	if (c_socket_server_accept(notify_socket, &client_notify_socket) != c_Socket_ErrorCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): notification socket accept failed.", __func__);
		return CAEN_MCA_RetCode_Generic;
	}
	
	// set client_notify_socket to be non-blocking
	if (!_setSocketBlockingEnabled(client_notify_socket->socket, FALSE)) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): cannot set blocking on client notify socket.", __func__);
		c_socket_delete(client_notify_socket);
		return CAEN_MCA_RetCode_Generic;
	}

	// select with timeout for the same reason as above accept
	FD_ZERO(&readmask);
	FD_SET(client_notify_socket->socket, &readmask);
	do {
		int32_t selret = select((int32_t)(client_notify_socket->socket + 1), &readmask, NULL, NULL, &sel_to);
		if (selret == -1) { // ERROR
			logMsg(c_logger_Severity_ERROR, "'select' error on client_notify_socket.");
			c_socket_delete(client_notify_socket);
			return CAEN_MCA_RetCode_Generic;
		}
		else if (selret == 0) { // TIMEOUT
			logMsg(c_logger_Severity_DEBUG, "'select' timeout on client_notify_socket.");
			c_socket_delete(client_notify_socket);
			return CAEN_MCA_RetCode_Generic;
		}
	} while (FD_ISSET(client_notify_socket->socket, &readmask) == 0);

	// Now we should be able to recv. If between the select and the accept the connection
	// closes, recv should return an error immediately, because it is non-blocking.
	PKTHEAD_CMD_TYPE cmd;
	PKTHEAD_TOTLEN_TYPE totSize;
	PKTHEAD_NPARAMS_TYPE totParams;
	uint8_t *notify_buff = c_recv_packet(client_notify_socket, &cmd, &totSize, &totParams), *b_ptr = notify_buff;
	if (notify_buff == NULL) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't receive notification packet.", __func__);
		c_socket_delete(client_notify_socket);
		return CAEN_MCA_RetCode_Generic;
	}
	c_socket_delete(client_notify_socket);

	int16_t srv_retcode;
	if ((b_ptr = deserialize_int16_t(&srv_retcode, 1, b_ptr)) == NULL) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't receive retcode from server.", __func__);
		return CAEN_MCA_RetCode_Generic;
	}
	if (srv_retcode != HexagonServer_RetCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): server startup failed.", __func__);
		return CAEN_MCA_RetCode_Generic;
	}
	if ((b_ptr = deserialize_uint16_t(port, 1, b_ptr)) == NULL) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't receive port from server.", __func__);
		return CAEN_MCA_RetCode_Generic;
	}
	c_free(notify_buff);

	return CAEN_MCA_RetCode_Success;
}

static int32_t _openDeviceGeneric(const c_url_field_t *url, MCALIB_Device_t *dev) {
	uint32_t q;
	int32_t ret;
	const char *link = NULL;
	const char *vmeaddress = NULL;
	const char *conetnode = NULL;
	const char *serial = NULL;
	uint16_t serverport;
	bool isEthernet = (strcmp(url->schema, "eth") == 0);

	// parse queries
	for (q = 0; q < url->query_num; q++) {
		const char *name = url->query[q].name;
		const char *value = url->query[q].value;

		// TODO also warn/error of invalid queries.
		if ((ret = handleQueryKey(dev, name, value, "link", &link)) != CAEN_MCA_RetCode_Success)
			return ret;
		if ((ret = handleQueryKey(dev, name, value, "conetnode", &conetnode)) != CAEN_MCA_RetCode_Success)
			return ret;
		if ((ret = handleQueryKey(dev, name, value, "vmeaddress", &vmeaddress)) != CAEN_MCA_RetCode_Success)
			return ret;
		if ((ret = handleQueryKey(dev, name, value, "serial", &serial)) != CAEN_MCA_RetCode_Success)
			return ret;
	}

	// get port to use
	if (url->port != NULL) {
		char *endp;
		uint64_t sp = strtoul(url->port, &endp, 0);
		if (endp == url->port) {
			MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): Cannot parse port %s", __func__, url->port);
			return CAEN_MCA_RetCode_Argument;
		}
		if (sp > UINT16_MAX) {
			MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): Given port %s is a too high value", __func__, url->port);
			return CAEN_MCA_RetCode_Argument;
		}
		serverport = (uint16_t)sp;
	}
	else {
		// use 0 for local instances, so port is automatically choosen
		serverport = (isEthernet || (serial != NULL)) ? CAEN_MCA_DEFAULT_TCP_PORT_DEF : 0;
	}

	if (isEthernet) {
		return _openDeviceEthParams(url->host, serverport, dev);
	}
	else { // any other scheme
		if (serial != NULL && link != NULL) {
			MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): Given path specifies serial='%s' and link='%s', don't know what to use (only one is allowed).", __func__, serial, link);
			return CAEN_MCA_RetCode_Argument;
		}

		// for non-eth schema, <host>/<path> must be localhost/connection
		if (strcmp(url->host, "localhost") != 0) {
			// TODO check a better solution than the following to allow 127.0.0.*
			int32_t ip0, ip1, ip2, ip3;
			if (sscanf(url->host, "%" SCNd32 ".%" SCNd32 ".%" SCNd32 ".%" SCNd32, &ip0, &ip1, &ip2, &ip3) != 4) {
				MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): invalid host '%s'", __func__, url->host);
				return CAEN_MCA_RetCode_Argument;
			}
			if (ip0 != 127 || ip1 != 0 || ip2 != 0 || ip3 < 1 || ip3 > 254) {
				MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): host '%s' not supported for schema %s. Allowed hosts: 'localhost', '127.0.0.1-254'", __func__, url->host, url->schema);
				return CAEN_MCA_RetCode_Argument;
			}
		}
		if (strcmp(url->path, "connection") != 0) {
			MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): given path specifies unsupported host path '%s'", __func__, url->path);
			return CAEN_MCA_RetCode_Argument;
		}

		if (serial != NULL) {
			char *endp;
			int32_t ip0, ip1, ip2 = 24;
			char ip_address[30];
			int32_t ser = strtol(serial, &endp, 0);

			if (endp == serial) {
				MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): Cannot parse serial %s", __func__, serial);
				return CAEN_MCA_RetCode_Argument;
			}
			ip0 = (ser * 4) % 256 + 2;
			ip1 = (ser * 4) / 256;
			if (ip1 > 255) {
				ip2 += ip1 / 256;
				ip1 = ip1 % 256;
			}
			sprintf(ip_address, "172.%d.%d.%d", ip2, ip1, ip0);
			return _openDeviceEthParams(ip_address, serverport, dev);
		}
		else if (link != NULL) {
			char basepath[sizeof(Library->libPath)];
			char path[sizeof(Library->libPath)];
			char ctypeparam[10];
			char portparam[10];
			char conetnodeparam[10];
			char vaddrparam[20];
			char notifyparam[20];
			size_t reqsz;
			int spres;
			uint16_t notify_port = 0;
			c_socket_t *notify_socket = NULL;

			reqsz = sizeof(portparam) / sizeof(*portparam);
			if ((spres = snprintf(portparam, reqsz, "%" PRIu16, serverport)) >= (int)reqsz || spres < 0) {
				logMsg(c_logger_Severity_ERROR, "%s(): cannot write port %" PRIu16 " to string. snprintf result %d", __func__, serverport, spres);
				return CAEN_MCA_RetCode_Generic;
			}

			if (c_getDirName(basepath, Library->libPath) != MP_code_Success)
				return CAEN_MCA_RetCode_Generic;
			// add dirsep if not present
			char dirseps[] = DIRSEPS_STR;
			if (strchr(dirseps, basepath[strlen(basepath) - 1]) == NULL)
				strcat(basepath, DIRSEP);
			sprintf(path, "%s%s" DIRSEP "%s", basepath, MCALIB_RESOURCES_ARCH_DIR, HEXAGONSERVERNAME);

			c_Process_t *hexagonServerProcess = c_newProc();
			c_setProcExec(path, hexagonServerProcess);
			c_addProcArg("-p", hexagonServerProcess);
			c_addProcArg(portparam, hexagonServerProcess);

			reqsz = sizeof(ctypeparam) / sizeof(*ctypeparam);
			if ((spres = snprintf(ctypeparam, reqsz, "%s", url->schema)) >= (int)reqsz || spres < 0) {
				logMsg(c_logger_Severity_ERROR, "%s(): cannot write protocol %s to string. snprintf result %d", __func__, url->schema, spres);
				return CAEN_MCA_RetCode_Generic;
			}
			c_addProcArg("-t", hexagonServerProcess);
			c_addProcArg(ctypeparam, hexagonServerProcess);
			if (conetnode != NULL) {
				reqsz = sizeof(conetnodeparam) / sizeof(*conetnodeparam);
				if ((spres = snprintf(conetnodeparam, reqsz, "%s", conetnode)) >= (int)reqsz || spres < 0) {
					logMsg(c_logger_Severity_ERROR, "%s(): cannot write conetnode %s to string. snprintf result %d", __func__, conetnode, spres);
					return CAEN_MCA_RetCode_Generic;
				}
				c_addProcArg("-c", hexagonServerProcess);
				c_addProcArg(conetnodeparam, hexagonServerProcess);
			}
			if (vmeaddress != NULL) {
				reqsz = sizeof(vaddrparam) / sizeof(*vaddrparam);
				if ((spres = snprintf(vaddrparam, reqsz, "%s", vmeaddress)) >= (int)reqsz || spres < 0) {
					logMsg(c_logger_Severity_ERROR, "%s(): cannot write vmeaddress %s to string. snprintf result %d", __func__, vmeaddress, spres);
					return CAEN_MCA_RetCode_Generic;
				}
				c_addProcArg("-a", hexagonServerProcess);
				c_addProcArg(vaddrparam, hexagonServerProcess);
			}

			// Prepare notification socket
			// accept connections from localhost only
			if (c_socket_server_init(&notify_socket, INADDR_LOOPBACK, &notify_port) != c_Socket_ErrorCode_Success) {
				logMsg(c_logger_Severity_ERROR, "%s(): cannot initilize notification socket.", __func__);
				return CAEN_MCA_RetCode_Generic;
			}
			if (!_setSocketBlockingEnabled(notify_socket->socket, FALSE)) {
				logMsg(c_logger_Severity_ERROR, "%s(): cannot set non-blocking on notify socket.", __func__);
				return CAEN_MCA_RetCode_Generic;
			}
			// notify param
			reqsz = sizeof(notifyparam) / sizeof(*notifyparam);
			if ((spres = snprintf(notifyparam, reqsz, "127.0.0.1:%" PRIu16, notify_port)) >= (int)reqsz || spres < 0) {
				logMsg(c_logger_Severity_ERROR, "%s(): cannot setup notification parameter using port %" PRIu16 ". snprintf result %d", __func__, notify_port, spres);
				c_socket_delete(notify_socket);
				return CAEN_MCA_RetCode_Generic;
			}
			c_addProcArg("-N", hexagonServerProcess);
			c_addProcArg(notifyparam, hexagonServerProcess);

			// TODO other args
			sprintf(path, "%s%s" DIRSEP "xml", basepath, MCALIB_RESOURCES_DIR);
			c_addProcEnv("HEXAGONSERVERDIR", path, hexagonServerProcess);
			c_addProcEnv("CAENDPPDIGITIZERDIR", path, hexagonServerProcess);
			sprintf(path, "%s%s", basepath, MCALIB_RESOURCES_ARCH_DIR);
#ifdef LINUX
			//c_addProcEnv("CAEN_LOG_FILENAME", "/tmp/HexagonServerLog.txt", hexagonServerProcess);
			c_addProcEnv("LD_LIBRARY_PATH", path, hexagonServerProcess);
#endif
			if (c_startProcess(hexagonServerProcess) != CAENPROC_RetCode_Success) {
				MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): Cannot execute server from path '%s'", __func__, path);
				c_socket_delete(notify_socket);
				return CAEN_MCA_RetCode_Generic;
			}

			// wait server's reply, notifying startup retcode and connection port
			if ((ret = _waitAndGetServerReply(dev, notify_socket, &serverport)) != CAEN_MCA_RetCode_Success) {
				c_socket_delete(notify_socket);
				return ret;
			}

			//c_sleep(10000); // TODO improve
			int32_t retry, max_retry = 100;
			for (retry = 0; retry < max_retry; retry++) {
				ret = _openDeviceEthParams("localhost", serverport, dev);
				if (ret == CAEN_MCA_RetCode_Success)
					break;
				c_sleep(100);
			}
			return ret;
		}
		else {
			MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): Given path doesn't specify serial or link. One and only one of them is mandatory for USB/OPT connection.", __func__);
			return CAEN_MCA_RetCode_Argument;
		}
	}
}

CAEN_MCA_HANDLE CAEN_MCA_OpenDevice(const char *path, int32_t *retcode, int32_t *index) {
	int32_t ret, devidx = CAEN_MCA_INVALID_INDEX;
	MCALIB_RemovableHandleCollection_t*collection = Library->devices;
	c_url_field_t *url = NULL;
	
	if (path == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): path cannot be NULL.", __func__);
		if (retcode != NULL)
			*retcode = CAEN_MCA_RetCode_Argument;
		if (index != NULL)
			*index = CAEN_MCA_INVALID_INDEX;
		return NULL;
	}

	if (collection == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): device collection cannot be NULL.", __func__);
		if (retcode != NULL)
			*retcode = CAEN_MCA_RetCode_Argument;
		if (index != NULL)
			*index = CAEN_MCA_INVALID_INDEX;
		return NULL;
	}

	MCALIB_Device_t *dev = MCALIB_CreateObject(&MCALIB_Handle_Device, collection->commonData.handle, 0);
	if (dev == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): can't allocate memory for device.", __func__);
		ret = CAEN_MCA_RetCode_OutOfMemory;
		goto QuitFunction;
	}

	// Initialize additional data after commonData.
	dev->isInitialized = FALSE;
	dev->parameters = NULL;
	dev->channels = NULL;
	dev->hvchannels = NULL;
	dev->lvdsgroups = NULL;
	dev->traces = NULL;
	dev->monouts = NULL;
	dev->socket = NULL;
	dev->definitions = NULL;
	dev->dtspectra = NULL;
	c_zeromem(&dev->DeviceInfo, sizeof(dev->DeviceInfo));

	// Convert path to hostname and port
	url = c_url_parse(path);
	if (url == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): can't parse path '%s'.", __func__, path);
		ret = CAEN_MCA_RetCode_Argument;
		goto QuitFunction;
	}

	if (strcmp(url->schema, "eth") == 0) {
		if ((ret = _openDeviceEth(url, dev)) != CAEN_MCA_RetCode_Success)
			goto QuitFunction;
	}
	else {
		// if schema is not 'eth', use 'generic' open (which may still be via
		// 'ethernet' in case of hexagon/gammastream with serialnum
		if ((ret = _openDeviceGeneric(url, dev)) != CAEN_MCA_RetCode_Success)
			goto QuitFunction;
	}

	if ((ret = _getBoardInfo(dev)) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't get board info from device.", __func__);
		goto QuitFunction;
	}

	if ((ret = _checkVersion(dev->DeviceInfo.LibVersion)) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): server compiled with a not compatible version of CAENMCA library.", __func__);
		goto QuitFunction;
	}

	if ((ret = MCALIB_GetDefinitionFile(dev->commonData.handle)) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't get definitions file.", __func__);
		goto QuitFunction;
	}

	// Create children
	if ((ret = _createParameters(dev->commonData.handle, &dev->parameters)) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't create parameters.", __func__);
		goto QuitFunction;
	}

	if ((ret = _createChannels(dev->commonData.handle, &dev->channels, dev->DeviceInfo.nChannels)) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't create channels.", __func__);
		goto QuitFunction;
	}

	if ((ret = _createDTSpectra(dev->commonData.handle, &dev->dtspectra, dev->DeviceInfo.nDTSpectra)) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't create Delta T spectra.", __func__);
		goto QuitFunction;
	}

	if ((ret = _createHVChannels(dev->commonData.handle, &dev->hvchannels, dev->DeviceInfo.nHVChannels)) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't create HV channels.", __func__);
		goto QuitFunction;
	}

	if ((ret = _createLVDSGroups(dev->commonData.handle, &dev->lvdsgroups, dev->DeviceInfo.nLVDSGroups)) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't create LVDS Groups.", __func__);
		goto QuitFunction;
	}

	if ((ret = _createTraces(dev->commonData.handle, &dev->traces, dev->DeviceInfo.nVirtualTraces)) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't create Traces.", __func__);
		goto QuitFunction;
	}

	if ((ret = _createMonOuts(dev->commonData.handle, &dev->monouts, dev->DeviceInfo.nMonOuts)) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(dev->commonData.handle, c_logger_Severity_ERROR, "%s(): can't create Monitor Outputs.", __func__);
		goto QuitFunction;
	}

	dev->isInitialized = TRUE;

	// Overwrite default object name with the <MODEL>_<SERIAL> and add it to collection.
	snprintf(dev->commonData.name, sizeof(dev->commonData.name), "%s_%" PRIu32, dev->DeviceInfo.ModelName, dev->DeviceInfo.SerialNum);
	if ((ret = MCALIB_AddRemovableHandleToIdx(collection, dev->commonData.handle, &devidx)) != CAEN_MCA_RetCode_Success) {
		goto QuitFunction;
	}

QuitFunction:
	// Return the device index as the index of the last element of the collection.
	if (index != NULL)
		*index = devidx;

	// Return the retcode
	if (retcode != NULL)
		*retcode = ret;

	c_url_free(url);

	if (ret != CAEN_MCA_RetCode_Success) {
		_freeDevice(dev);
		logMsg(c_logger_Severity_ERROR, "%s(): failed with error code %"PRIi32".", __func__, ret);
		return NULL;
	}

	return dev->commonData.handle;
}

void CAEN_MCA_CloseDevice(CAEN_MCA_HANDLE handle) {
	MCALIB_Handle_t *h = (MCALIB_Handle_t*)handle;

	if (!MCALIB_IsValidHandle(h)) {
		logMsg(c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return;
	}

	MCALIB_Device_t *device = MCALIB_GetObject(h, CAEN_MCA_HANDLE_DEVICE);
	_freeDevice(device);
}

CAEN_MCA_HANDLE CAEN_MCA_GetAncestorHandle(CAEN_MCA_HANDLE handle, CAEN_MCA_HandleType_t handleType) {
	MCALIB_Handle_t *h = (MCALIB_Handle_t*)handle;

	if (!MCALIB_IsValidHandle(h)) {
		logMsg(c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	// The returned pointer can be casted safeli to MCALIB_ObjectCommonData_t*
	// Being the first element of the object structures.
	MCALIB_ObjectCommonData_t *ancestor = MCALIB_GetAncestor(h, handleType);

	if (ancestor != NULL)
		return ancestor->handle;
	else
		return NULL;
}

CAEN_MCA_HANDLE CAEN_MCA_GetChildHandle(CAEN_MCA_HANDLE handle, CAEN_MCA_HandleType_t handleType, int32_t index) {
	MCALIB_Handle_t *h = (MCALIB_Handle_t*)handle;

	if (!MCALIB_IsValidHandle(h)) {
		logMsg(c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	// No need to check if handle is NULL:
	// NULL allowed for CAEN_MCA_HANDLE_LIBRARY and CAEN_MCA_HANDLE_DEVICE
	switch (handleType) {
	case CAEN_MCA_HANDLE_LIBRARY:
		return MCALIB_GetLibraryHandle();
	case CAEN_MCA_HANDLE_DEVICE:
		return MCALIB_GetDeviceHandle(h, index, NULL);
	case CAEN_MCA_HANDLE_CHANNEL:
		return MCALIB_GetChannelHandle(h, index, NULL);
	case CAEN_MCA_HANDLE_ENERGYSPECTRUM:
		return MCALIB_GetEnergySpectrumHandle(h, index, NULL);
	case CAEN_MCA_HANDLE_MCSSPECTRUM:
		return MCALIB_GetMCSSpectrumHandle(h, index, NULL);
	case CAEN_MCA_HANDLE_DTSPECTRUM:
		return MCALIB_GetDTSpectrumHandle(h, index, NULL);
	case CAEN_MCA_HANDLE_ROI:
		return MCALIB_GetROIHandle(h, index, NULL);
	case CAEN_MCA_HANDLE_HVCHANNEL:
		return MCALIB_GetHVChannelHandle(h, index, NULL);
	case CAEN_MCA_HANDLE_HVRANGE:
		return MCALIB_GetHVRangeHandle(h, index, NULL);
	case CAEN_MCA_HANDLE_LVDSGROUP:
		return MCALIB_GetLVDSGroupHandle(h, index, NULL);
	case CAEN_MCA_HANDLE_TRACE:
		return MCALIB_GetTraceHandle(h, index, NULL);
	case CAEN_MCA_HANDLE_MONOUT:
		return MCALIB_GetMonOutHandle(h, index, NULL);
	case CAEN_MCA_HANDLE_PARAMETER:
		return MCALIB_GetParameterHandle(h, index, NULL);
	case CAEN_MCA_HANDLE_COLLECTION: {
		// For collection, index represents the type of collection.
		CAEN_MCA_HandleType_t type = index;
		return MCALIB_GetHandleCollectionHandleByType(h, type);
	}
	default:
		MCALIB_LogMsgExt(h, c_logger_Severity_ERROR, "%s(): unsupported handle type '%d'.", __func__, handleType);
		return NULL;
	}
}

CAEN_MCA_HANDLE CAEN_MCA_GetChildHandleByName(CAEN_MCA_HANDLE handle, CAEN_MCA_HandleType_t handleType, const char *name) {
	MCALIB_Handle_t *h = (MCALIB_Handle_t*)handle;

	if (!MCALIB_IsValidHandle(h)) {
		logMsg(c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	// No need to check if handle is NULL:
	// NULL allowed for CAEN_MCA_HANDLE_LIBRARY and CAEN_MCA_HANDLE_DEVICE
	switch (handleType) {
	case CAEN_MCA_HANDLE_LIBRARY:
		return MCALIB_GetLibraryHandle();
	case CAEN_MCA_HANDLE_DEVICE:
		return MCALIB_GetDeviceHandle(h, CAEN_MCA_INVALID_INDEX, name);
	case CAEN_MCA_HANDLE_CHANNEL:
		return MCALIB_GetChannelHandle(h, CAEN_MCA_INVALID_INDEX, name);
	case CAEN_MCA_HANDLE_ENERGYSPECTRUM:
		return MCALIB_GetEnergySpectrumHandle(h, CAEN_MCA_INVALID_INDEX, name);
	case CAEN_MCA_HANDLE_MCSSPECTRUM:
		return MCALIB_GetMCSSpectrumHandle(h, CAEN_MCA_INVALID_INDEX, name);
	case CAEN_MCA_HANDLE_DTSPECTRUM:
		return MCALIB_GetDTSpectrumHandle(h, CAEN_MCA_INVALID_INDEX, name);
	case CAEN_MCA_HANDLE_ROI:
		return MCALIB_GetROIHandle(h, CAEN_MCA_INVALID_INDEX, name);
	case CAEN_MCA_HANDLE_HVCHANNEL:
		return MCALIB_GetHVChannelHandle(h, CAEN_MCA_INVALID_INDEX, name);
	case CAEN_MCA_HANDLE_HVRANGE:
		return MCALIB_GetHVRangeHandle(h, CAEN_MCA_INVALID_INDEX, name);
	case CAEN_MCA_HANDLE_LVDSGROUP:
		return MCALIB_GetLVDSGroupHandle(h, CAEN_MCA_INVALID_INDEX, name);
	case CAEN_MCA_HANDLE_TRACE:
		return MCALIB_GetTraceHandle(h, CAEN_MCA_INVALID_INDEX, name);
	case CAEN_MCA_HANDLE_MONOUT:
		return MCALIB_GetMonOutHandle(h, CAEN_MCA_INVALID_INDEX, name);
	case CAEN_MCA_HANDLE_PARAMETER:
		return MCALIB_GetParameterHandle(h, CAEN_MCA_INVALID_INDEX, name);
	default:
		MCALIB_LogMsgExt(h, c_logger_Severity_ERROR, "%s(): unsupported handle type '%d'.", __func__, handleType);
		return NULL;
	}
}

int32_t CAEN_MCA_GetData(CAEN_MCA_HANDLE handle, CAEN_MCA_DataType_t dataType, uint64_t dataMask, ...) {
	va_list args;
	va_start(args, dataMask);
	int32_t ret = CAEN_MCA_GetDataV(handle, dataType, dataMask, args);
	va_end(args);
	return ret;
}

int32_t CAEN_MCA_GetDataV(CAEN_MCA_HANDLE handle, CAEN_MCA_DataType_t dataType, uint64_t dataMask, va_list args) {
	MCALIB_Handle_t *h = (MCALIB_Handle_t*)handle;

	if (!MCALIB_IsValidHandle(h)) {
		logMsg(c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return CAEN_MCA_RetCode_Handle;
	}

	if (dataMask == DATAMASK_CMD_NONE) {
		MCALIB_LogMsgExt(h, c_logger_Severity_ERROR, "%s(): dataMask cannot be 0x%"PRIx64".", __func__, DATAMASK_CMD_NONE);
		return CAEN_MCA_RetCode_Argument;
	}

	switch (dataType) {
	case CAEN_MCA_DATA_BOARD_INFO:
		return MCALIB_GetBoardInfo(h, dataMask, args);
	case CAEN_MCA_DATA_PARAMETER_INFO:
		return MCALIB_GetParameterInfo(h, dataMask, args);
	case CAEN_MCA_DATA_CHANNEL_INFO:
		return MCALIB_GetChannelInfo(h, dataMask, args);
	case CAEN_MCA_DATA_HVCHANNEL_INFO:
		return MCALIB_GetHVChannelInfo(h, dataMask, args);
	case CAEN_MCA_DATA_HVRANGE_INFO:
		return MCALIB_GetHVRangeInfo(h, dataMask, args);
	case CAEN_MCA_DATA_COLLECTION:
		return MCALIB_GetCollectionInfo(h, dataMask, args);
	case CAEN_MCA_DATA_PARAMETER_VALUE:
		return MCALIB_GetParameterValue(h, dataMask, args);
	case CAEN_MCA_DATA_ENERGYSPECTRUM:
		return MCALIB_GetEnergySpectrumInfo(h, dataMask, args);
	case CAEN_MCA_DATA_ROI:
		return MCALIB_GetROIInfo(h, dataMask, args);
	case CAEN_MCA_DATA_WAVEFORM:
		return MCALIB_GetWaveform(h, dataMask, args);
	case CAEN_MCA_DATA_LIST_MODE:
		return MCALIB_GetListMode(h, dataMask, args);
	case CAEN_MCA_DATA_MCSSPECTRUM:
		return MCALIB_GetMCSSpectrumInfo(h, dataMask, args);
	case CAEN_MCA_DATA_DISCOVEREDDEVICES:
		return MCALIB_GetDiscoveredDevices(h, dataMask, args);
	case CAEN_MCA_DATA_HANDLE_INFO:
		return MCALIB_GetHandleInfo(h, dataMask, args);
	case CAEN_MCA_DATA_DTSPECTRUM:
		return MCALIB_GetDTSpectrumInfo(h, dataMask, args);
	default:
		MCALIB_LogMsgExt(h, c_logger_Severity_ERROR, "%s(): unsupported dataType: %d", __func__, dataType);
		return CAEN_MCA_RetCode_Argument;
	}
}

int32_t CAEN_MCA_SetData(CAEN_MCA_HANDLE handle, CAEN_MCA_DataType_t dataType, uint64_t dataMask, ...) {
	va_list args;
	va_start(args, dataMask);
	int32_t ret = CAEN_MCA_SetDataV(handle, dataType, dataMask, args);
	va_end(args);
	return ret;
}

int32_t CAEN_MCA_SetDataV(CAEN_MCA_HANDLE handle, CAEN_MCA_DataType_t dataType, uint64_t dataMask, va_list args) {
	MCALIB_Handle_t *h = (MCALIB_Handle_t*)handle;

	if (!MCALIB_IsValidHandle(h)) {
		logMsg(c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return CAEN_MCA_RetCode_Handle;
	}

	if (dataMask == DATAMASK_CMD_NONE) {
		MCALIB_LogMsgExt(h, c_logger_Severity_ERROR, "%s(): dataMask cannot be 0.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}

	switch (dataType) {
	case CAEN_MCA_DATA_PARAMETER_VALUE:
		return MCALIB_SetParameterValue(h, dataMask, args);
	case CAEN_MCA_DATA_ENERGYSPECTRUM:
		return MCALIB_SetEnergySpectrumInfo(h, dataMask, args);
	case CAEN_MCA_DATA_LIST_MODE:
		return MCALIB_SetListMode(h, dataMask, args);
	case CAEN_MCA_DATA_DISCOVEREDDEVICES:
		return MCALIB_SetDiscoveredDevices(h, dataMask, args);
	case CAEN_MCA_DATA_DTSPECTRUM:
		return MCALIB_SetDTSpectrumInfo(h, dataMask, args);
	default:
		MCALIB_LogMsgExt(h, c_logger_Severity_ERROR, "%s(): unsupported dataType: %d", __func__, dataType);
		return CAEN_MCA_RetCode_Argument;
	}
}

// TODO
int32_t CAEN_MCA_WaitEvent(CAEN_MCA_HANDLE handle, CAEN_MCA_HANDLE *eventHandle) {
	c_unused_parameter(handle);
	c_unused_parameter(eventHandle);
	return CAEN_MCA_RetCode_NotYetImplemented;
}

int32_t CAEN_MCA_SendCommand(CAEN_MCA_HANDLE handle, CAEN_MCA_CommandType_t cmdType, uint64_t cmdMaskIn, uint64_t cmdMaskOut, ...) {
	va_list args;
	va_start(args, cmdMaskOut);
	int32_t ret = CAEN_MCA_SendCommandV(handle, cmdType, cmdMaskIn, cmdMaskOut, args);
	va_end(args);
	return ret;
}

int32_t CAEN_MCA_SendCommandV(CAEN_MCA_HANDLE handle, CAEN_MCA_CommandType_t cmdType, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	MCALIB_Handle_t *h = (MCALIB_Handle_t*)handle;

	if (!MCALIB_IsValidHandle(h)) {
		logMsg(c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return CAEN_MCA_RetCode_Handle;
	}

	switch (cmdType) {
	case CAEN_MCA_CMD_ACQ_START:
		return MCALIB_AcquisitionStart(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_ACQ_STOP:
		return MCALIB_AcquisitionStop(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_ENERGYSPECTRUM_CLEAR:
		return MCALIB_ClearEnergySpectrum(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_DTSPECTRUM_CLEAR:
		return MCALIB_ClearDTSpectrum(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_REGISTER_READ:
		return MCALIB_GetRegisterValue(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_REGISTER_WRITE:
		return MCALIB_SetRegisterValue(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_CONFIGURATION_SAVE:
		return MCALIB_SaveConfiguration(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_CONFIGURATION_LOAD:
		return MCALIB_LoadConfiguration(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_CONFIGURATION_LIST:
		return MCALIB_GetConfigurationList(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_CONFIGURATION_DELETE:
		return MCALIB_DeleteConfiguration(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_PARAM_AUTOSET_START:
		return MCALIB_ParamAutoSetStart(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_PARAM_AUTOSET_STOP:
		return MCALIB_ParamAutoSetStop(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_HV_ON:
		return MCALIB_HVOn(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_HV_OFF:
		return MCALIB_HVOff(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_HV_ONOFF:
		return MCALIB_HVOnOff(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_RESTART:
		return MCALIB_GlobalQuit(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_MCS_SWEEP:
		return MCALIB_MCSSweep(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_MCS_SPECTRUM_CLEAR:
		return MCALIB_ClearMCSSpectrum(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_GAIN_STABILIZER_RESET:
		return MCALIB_ResetGainStabilizer(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_SAMPLE_ADVANCE:
		return MCALIB_SampleAdvance(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_CONFIGURATION_DB_PATH:
		return MCALIB_DatabasePath(h, cmdMaskIn, cmdMaskOut, args);
	case CAEN_MCA_CMD_RTCLOCK:
		return MCALIB_RTClock(h, cmdMaskIn, cmdMaskOut, args);
	default:
		MCALIB_LogMsgExt(h, c_logger_Severity_ERROR, "%s(): unsupported cmdType: %d", __func__, cmdType);
		return CAEN_MCA_RetCode_Argument;
	}
}

#ifndef _WIN32
static void fillLibPath(void) {
	// Fill library path. NOTE: for windows, it is done in DLLMain.
	Dl_info dlinfo;
	if (dladdr(fillLibPath, &dlinfo) == 0)
		logMsg(c_logger_Severity_ERROR, "Cannot get this library path. Local server instances may not work.");
	strcpy(Library->libPath, dlinfo.dli_fname);
}
#endif

// Perform here any Library initialization.
static int32_t _initMCALib(void) {

	// Generate random control pattern.
	// This needs to be done before the first MCALIB_Handle_t is created.
	uint64_t pattern = c_rand64_int();
	MCALIB_SetPatternGood(pattern);
	MCALIB_SetPatternBad(~pattern);

	// Initialize Library object
	Library = MCALIB_CreateObject(&MCALIB_Handle_Library, NULL, 0);
	if (Library == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

	// Initialize additional data after commonData.
	Library->devices = NULL;
	Library->discoverTimeout_ms = SSDP_DEFAULT_TIMEOUT_MS;

	// Create children
	// For library, it is just an empty collection for devices.
	if ((Library->devices = MCALIB_CreateRemovableHandleCollection(Library->commonData.handle, CAEN_MCA_HANDLE_DEVICE)) == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): cannot create an handle collection.", __func__);
		return CAEN_MCA_RetCode_Generic;
	}

#ifndef _WIN32
	fillLibPath();
#endif

	return CAEN_MCA_RetCode_Success;
}

// Perform here any Library initialization.
static int32_t _closeLibrary(void) {
	// Free devices collection
	MCALIB_DeleteRemovableHandleCollection(Library->devices);

	// Free Library handle
	MCALIB_DeleteObject(Library);

	return CAEN_MCA_RetCode_Success;
}

#ifdef _CAEN_MCA_EXPORT
#ifdef _WIN32
/**********************************************************************\
								DllMain
\**********************************************************************/
BOOL WINAPI DllMain(HINSTANCE hModule, DWORD ul_reason_for_call, LPVOID lpReserved) {
	DWORD sz, max_sz = sizeof(Library->libPath);

	// Get the library path
	UNREFERENCED_PARAMETER(lpReserved);
	switch (ul_reason_for_call) {
	case DLL_PROCESS_ATTACH:
		_initMCALib();
		
		// Fill libPath. This may be done in _initMCALib, but we need to pass hModule that is WIN-dependent.
		sz = GetModuleFileName(hModule, Library->libPath, max_sz);
		if (sz == 0 || sz == max_sz)
			logMsg(c_logger_Severity_ERROR, "Cannot get this library path. Local server instances may not work.");
		break;
	case DLL_PROCESS_DETACH:
		_closeLibrary();
		break;
	case DLL_THREAD_ATTACH:
		break;
	case DLL_THREAD_DETACH:
		break;
	}
	return TRUE;
}
#else
/**********************************************************************\
						__attribute__
\**********************************************************************/
void __attribute__((constructor)) CAEN_MCA_init(void) {
	_initMCALib();
	MCALIB_LoadGSSDPLibraries();
}
void __attribute__((destructor)) CAEN_MCA_close(void) {
	MCALIB_CloseGSSDPLibraries();
	_closeLibrary();
}
#endif // _CAEN_MCA_EXPORT
#endif
