#include <MCALIB_Common.h>

#include <stdio.h>

#include <CAENMultiplatform.h>
#include <CAENSerDes.h>

INIT_C_LOGGER("MCALIB_CommonLog.txt", "MCALIB_Common.c");

const MCALIB_HandleType_t MCALIB_Handle_Library =				{ CAEN_MCA_HANDLE_LIBRARY,				sizeof(MCALIB_Library_t),					"Library"						};
const MCALIB_HandleType_t MCALIB_Handle_Device =				{ CAEN_MCA_HANDLE_DEVICE,				sizeof(MCALIB_Device_t),					"Device"						};
const MCALIB_HandleType_t MCALIB_Handle_Channel =				{ CAEN_MCA_HANDLE_CHANNEL,				sizeof(MCALIB_Channel_t),					"Channel"						};
const MCALIB_HandleType_t MCALIB_Handle_HVChannel =				{ CAEN_MCA_HANDLE_HVCHANNEL,			sizeof(MCALIB_HVChannel_t),					"HV Channel"					};
const MCALIB_HandleType_t MCALIB_Handle_HVRange =				{ CAEN_MCA_HANDLE_HVRANGE,				sizeof(MCALIB_HVRange_t),					"HV Range"						};
const MCALIB_HandleType_t MCALIB_Handle_LVDSGroup =				{ CAEN_MCA_HANDLE_LVDSGROUP,			sizeof(MCALIB_LVDSGroup_t),					"LVDS Group"					};
const MCALIB_HandleType_t MCALIB_Handle_Trace =					{ CAEN_MCA_HANDLE_TRACE,				sizeof(MCALIB_Trace_t),						"Trace"							};
const MCALIB_HandleType_t MCALIB_Handle_Parameter =				{ CAEN_MCA_HANDLE_PARAMETER,			sizeof(MCALIB_Parameter_t),					"Parameter"						};
const MCALIB_HandleType_t MCALIB_Handle_Collection =			{ CAEN_MCA_HANDLE_COLLECTION,			sizeof(MCALIB_HandleCollection_t),			"Handle Collection"				};
const MCALIB_HandleType_t MCALIB_Handle_Event =					{ CAEN_MCA_HANDLE_EVENT,				sizeof(MCALIB_Event_t),						"Event"							};
const MCALIB_HandleType_t MCALIB_Handle_Trigger =				{ CAEN_MCA_HANDLE_TRIGGER,				sizeof(MCALIB_Trigger_t),					"Trigger"						};
const MCALIB_HandleType_t MCALIB_Handle_Value =					{ CAEN_MCA_HANDLE_VALUE,				sizeof(MCALIB_Value_t),						"Value"							};
const MCALIB_HandleType_t MCALIB_Handle_EnergySpectrum =		{ CAEN_MCA_HANDLE_ENERGYSPECTRUM,		sizeof(MCALIB_EnergySpectrum_t),			"Energy Spectrum"				};
const MCALIB_HandleType_t MCALIB_Handle_MCSSpectrum =			{ CAEN_MCA_HANDLE_MCSSPECTRUM,			sizeof(MCALIB_MCSSpectrum_t),				"MCS Spectrum"					};
const MCALIB_HandleType_t MCALIB_Handle_ROI =					{ CAEN_MCA_HANDLE_ROI,					sizeof(MCALIB_ROI_t),						"ROI"							};
const MCALIB_HandleType_t MCALIB_Handle_MonOut =				{ CAEN_MCA_HANDLE_MONOUT,				sizeof(MCALIB_MonOut_t),					"Monitor Output"				};
const MCALIB_HandleType_t MCALIB_Handle_DTSpectrum =			{ CAEN_MCA_HANDLE_DTSPECTRUM,			sizeof(MCALIB_DTSpectrum_t),				"DT Spectrum"					};
const MCALIB_HandleType_t MCALIB_Handle_Removable_Collection =	{ CAEN_MCA_HANDLE_REMOVABLE_COLLECTION,	sizeof(MCALIB_RemovableHandleCollection_t),	"Removalbe Handle Collection"	};

static uint64_t _patternGood;
static uint64_t _patternBad;

static int32_t djb2a(const char* str) {
	uint32_t hash = UINT32_C(5381);
	char c;
	while ((c = *str++) != '\0')
		hash = (UINT32_C(33) * hash) ^ c;
	return (hash % INT32_MAX);
}

static size_t getNumDigits(int32_t i) {
	if (i >= 100000) {
		if (i >= 10000000) {
			if (i >= 1000000000)
				return 10;
			if (i >= 100000000)
				return 9;
			return 8;
		}
		else {
			if (i >= 1000000)
				return 7;
			return 6;
		}
	}
	else {
		if (i >= 1000) {
			if (i >= 10000)
				return 5;
			return 4;
		}
		else {
			if (i >= 100)
				return 3;
			if (i >= 10)
				return 2;
			return 1;
		}
	}
}

static size_t fastPrintInteger(int32_t num, char* buffer) {
	size_t nd = 1;
	if (num == 0) {
		*buffer++ = '0';
		*buffer = '\0';
	}
	else if (num < 0) {
		*buffer++ = '-';
		return fastPrintInteger(-num, buffer) + 1;
	}
	else {
		nd = getNumDigits(num);
		buffer += nd;
		*buffer-- = '\0';

		do {
			*buffer-- = '0' + (num % 10);
		} while (num /= 10);
	}
	return nd;
}


static char* idxToString(char* dest, int32_t idx) {
	fastPrintInteger(idx, dest);
	return dest;
}

uint64_t MCALIB_GetPatternGood(void) {
	return _patternGood;
}

void MCALIB_SetPatternGood(uint64_t pattern) {
	_patternGood = pattern;
}

uint64_t MCALIB_GetPatternBad(void) {
	return _patternBad;
}

void MCALIB_SetPatternBad(uint64_t pattern) {
	_patternBad = pattern;
}

bool MCALIB_IsValidHandle(const MCALIB_Handle_t *handle) {
	// NULL is considered valid, as could be a valid argument of some functions.
	if (handle == NULL)
		return TRUE;

	return (handle->controlPattern == MCALIB_GetPatternGood());
}

MCALIB_Handle_t *MCALIB_CreateHandle(MCALIB_ObjectCommonData_t *commonData, void *data) {
	MCALIB_Handle_t *handle = c_malloc(sizeof(*handle));
	if (handle == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): can't allocate handle.", __func__);
		return NULL;
	}

	// Set controlPattern to a fixed string, to be used to check if a CAEN_MCA_HANDLE passed
	// to the API functions are valid or not.
	handle->controlPattern = MCALIB_GetPatternGood();

	// When 'commonData' is the first element of the object in 'data'
	// the two address are identical.
	handle->commonData = commonData;
	handle->data = data;

	return handle;
}

void MCALIB_DeleteHandle(MCALIB_Handle_t *handle) {
	c_free(handle);
}

void *MCALIB_CreateObject(const MCALIB_HandleType_t *type, const MCALIB_Handle_t *parent, int32_t index) {
	if (type == NULL)
		return NULL;

	if (type->size == 0)
		return NULL;

	void *object = c_malloc(type->size);
	if (object == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): can't allocate object.", __func__);
		return NULL;
	}

	// Cast the pointer to allocated memory to MCALIB_ObjectCommonData_t,
	// that MUST be the first element of the Object. It is safe, according to the C standard.
	// This allow to set the CommonData, despite of its type.
	MCALIB_ObjectCommonData_t *commonData = object;

	// Pointer to the global variable representing the type.
	commonData->type = type;

	// A generic index of the object. Its meaning depends on the type.
	commonData->index = index;

	// Set collectionIndex to CAEN_MCA_INVALID_INDEX.
	// It will be set by MCALIB_AddHandleTo().
	commonData->collectionIndex = CAEN_MCA_INVALID_INDEX;

	// The parent creating the new object. Can be NULL.
	commonData->parentHandle = parent;
	
	// The name is used to insert the object Handled in a map.
	// The default name is the index, can be overwritten later (e.g. for Parameters with codename).
	snprintf(commonData->name, sizeof(commonData->name), "%"PRIi32, index);

	// The Handle to be passed to the client.
	commonData->handle = MCALIB_CreateHandle(commonData, object);
	if (commonData->handle == NULL) {
		MCALIB_DeleteObject(object);
		return NULL;
	}

	return object;
}

void MCALIB_DeleteObject(void *object) {
	if (object == NULL) {
		return;
	}

	// Cast the pointer to allocated memory to MCALIB_ObjectCommonData_t,
	// that MUST be the first element of the Object. It is safe, according to the C standard.
	// This allow to set the CommonData, despite of its type.
	MCALIB_ObjectCommonData_t *commonData = object;

	MCALIB_DeleteHandle(commonData->handle);
	c_free(object);
}

void *MCALIB_GetObject(const MCALIB_Handle_t *handle, CAEN_MCA_HandleType_t type) {
	if (handle == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): NULL handle.", __func__);
		return NULL;
	}

	if (handle->commonData->type->typeCode == type)
		return handle->data;
	else
		return NULL;
}

void *MCALIB_GetAncestor(const MCALIB_Handle_t *handle, CAEN_MCA_HandleType_t type) {
	if (handle == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): NULL handle.", __func__);
		return NULL;
	}

	if (type == CAEN_MCA_HANDLE_PARENT) {
		const MCALIB_Handle_t *parent = handle->commonData->parentHandle;
		// skip parent collections
		if (parent->commonData->type->typeCode == CAEN_MCA_HANDLE_COLLECTION)
			return MCALIB_GetAncestor(parent, type);
		else
			return parent->data;
	}
	else {
		if (handle->commonData->type->typeCode == type)
			return handle->data;
		else
			return MCALIB_GetAncestor(handle->commonData->parentHandle, type);
	}
}

MCALIB_HandleCollection_t *MCALIB_GetParameterHandleCollection(const MCALIB_Handle_t *handle) {
	if (handle == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): NULL handle.", __func__);
		return NULL;
	}

	CAEN_MCA_HandleType_t type = handle->commonData->type->typeCode;

	switch (type) {
	case CAEN_MCA_HANDLE_DEVICE: {
		MCALIB_Device_t *object = MCALIB_GetObject(handle, type);
		if (object != NULL)
			return object->parameters;
		break;
	}
	case CAEN_MCA_HANDLE_CHANNEL: {
		MCALIB_Channel_t *object = MCALIB_GetObject(handle, type);
		if (object != NULL)
			return object->parameters;
		break;
	}
	case CAEN_MCA_HANDLE_ENERGYSPECTRUM: {
		MCALIB_EnergySpectrum_t *object = MCALIB_GetObject(handle, type);
		if (object != NULL)
			return object->parameters;
		break;
	}
	case CAEN_MCA_HANDLE_MCSSPECTRUM: {
		MCALIB_MCSSpectrum_t *object = MCALIB_GetObject(handle, type);
		if (object != NULL)
			return object->parameters;
		break;
	}
	case CAEN_MCA_HANDLE_DTSPECTRUM: {
		MCALIB_DTSpectrum_t *object = MCALIB_GetObject(handle, type);
		if (object != NULL)
			return object->parameters;
		break;
	}
	case CAEN_MCA_HANDLE_ROI: {
		MCALIB_ROI_t *object = MCALIB_GetObject(handle, type);
		if (object != NULL)
			return object->parameters;
		break;
	}
	case CAEN_MCA_HANDLE_HVCHANNEL: {
		MCALIB_HVChannel_t *object = MCALIB_GetObject(handle, type);
		if (object != NULL)
			return object->parameters;
		break;
	}
	case CAEN_MCA_HANDLE_LVDSGROUP: {
		MCALIB_LVDSGroup_t *object = MCALIB_GetObject(handle, type);
		if (object != NULL)
			return object->parameters;
		break;
	}
	case CAEN_MCA_HANDLE_TRACE: {
		MCALIB_Trace_t *object = MCALIB_GetObject(handle, type);
		if (object != NULL)
			return object->parameters;
		break;
	}
	case CAEN_MCA_HANDLE_MONOUT: {
		MCALIB_MonOut_t *object = MCALIB_GetObject(handle, type);
		if (object != NULL)
			return object->parameters;
		break;
	}
	case CAEN_MCA_HANDLE_HVRANGE: {
		MCALIB_HVRange_t *object = MCALIB_GetObject(handle, type);
		if (object != NULL)
			return object->parameters;
		break;
	}
	case CAEN_MCA_HANDLE_PARAMETER: {
		MCALIB_Parameter_t *object = MCALIB_GetObject(handle, type);
		if (object != NULL)
			return object->parameters;
		break;
	}
	default:
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid type '%s'", __func__, handle->commonData->type->typeName);
	}

	MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): can't get params collection from handle '%s'", __func__, handle->commonData->type->typeName);
	return NULL;
}

MCALIB_Handle_t* MCALIB_GetLibraryHandle(void) {
	return Library->commonData.handle;
}

MCALIB_Handle_t* MCALIB_GetDeviceHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name) {
	const MCALIB_Library_t *library = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_LIBRARY);
	if (library == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	const MCALIB_RemovableHandleCollection_t *collection = library->devices;

	if (name == NULL)
		return MCALIB_GetRemovableHandleAt(collection, (uint32_t)index);
	else
		return MCALIB_GetRemovableHandleByName(collection, name);
}

MCALIB_Handle_t* MCALIB_GetChannelHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name) {
	const MCALIB_Device_t *dev = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);
	if (dev == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	const MCALIB_HandleCollection_t *collection = dev->channels;

	if (name == NULL)
		return MCALIB_GetHandleAt(collection, (uint32_t)index);
	else
		return MCALIB_GetHandleByName(collection, name);
}

MCALIB_Handle_t* MCALIB_GetHVChannelHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name) {
	const MCALIB_Device_t *dev = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);
	if (dev == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	const MCALIB_HandleCollection_t *collection = dev->hvchannels;

	if (name == NULL)
		return MCALIB_GetHandleAt(collection, (uint32_t)index);
	else
		return MCALIB_GetHandleByName(collection, name);
}

MCALIB_Handle_t* MCALIB_GetHVRangeHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name) {
	const MCALIB_HVChannel_t *hvchannel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_HVCHANNEL);
	if (hvchannel == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	const MCALIB_HandleCollection_t *collection = hvchannel->hvranges;

	if (name == NULL)
		return MCALIB_GetHandleAt(collection, (uint32_t)index);
	else
		return MCALIB_GetHandleByName(collection, name);
}

MCALIB_Handle_t* MCALIB_GetLVDSGroupHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name) {
	const MCALIB_Device_t *dev = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);
	if (dev == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	const MCALIB_HandleCollection_t *collection = dev->lvdsgroups;

	if (name == NULL)
		return MCALIB_GetHandleAt(collection, (uint32_t)index);
	else
		return MCALIB_GetHandleByName(collection, name);
}

MCALIB_Handle_t* MCALIB_GetTraceHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name) {
	const MCALIB_Device_t *dev = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);
	if (dev == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	const MCALIB_HandleCollection_t *collection = dev->traces;

	if (name == NULL)
		return MCALIB_GetHandleAt(collection, (uint32_t)index);
	else
		return MCALIB_GetHandleByName(collection, name);
}

MCALIB_Handle_t* MCALIB_GetMonOutHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name) {
	const MCALIB_Device_t *dev = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);
	if (dev == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	const MCALIB_HandleCollection_t *collection = dev->monouts;

	if (name == NULL)
		return MCALIB_GetHandleAt(collection, (uint32_t)index);
	else
		return MCALIB_GetHandleByName(collection, name);
}

MCALIB_Handle_t* MCALIB_GetEnergySpectrumHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name) {
	const MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);
	if (channel == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	const MCALIB_HandleCollection_t *collection = channel->energyspectra;

	if (name == NULL)
		return MCALIB_GetHandleAt(collection, (uint32_t)index);
	else
		return MCALIB_GetHandleByName(collection, name);
}

MCALIB_Handle_t* MCALIB_GetMCSSpectrumHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name) {
	const MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);
	if (channel == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	const MCALIB_HandleCollection_t *collection = channel->mcsspectra;

	if (name == NULL)
		return MCALIB_GetHandleAt(collection, (uint32_t)index);
	else
		return MCALIB_GetHandleByName(collection, name);
}

MCALIB_Handle_t* MCALIB_GetDTSpectrumHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name) {
	const MCALIB_Device_t *device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);
	if (device == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	const MCALIB_HandleCollection_t *collection = device->dtspectra;

	if (name == NULL)
		return MCALIB_GetHandleAt(collection, (uint32_t)index);
	else
		return MCALIB_GetHandleByName(collection, name);
}

MCALIB_Handle_t* MCALIB_GetROIHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name) {
	const MCALIB_EnergySpectrum_t *spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_ENERGYSPECTRUM);
	if (spectrum == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}

	const MCALIB_HandleCollection_t *collection = spectrum->rois;

	if (name == NULL)
		return MCALIB_GetHandleAt(collection, (uint32_t)index);
	else
		return MCALIB_GetHandleByName(collection, name);
}

MCALIB_Handle_t* MCALIB_GetParameterHandle(const MCALIB_Handle_t *handle, int32_t index, const char *name) {
	// For parameter, we start from the generic collection, as multiple types has parameters as children.
	const MCALIB_HandleCollection_t *collection = MCALIB_GetParameterHandleCollection(handle);
	if (collection == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return NULL;
	}
	
	if (name == NULL)
		return MCALIB_GetHandleAt(collection, (uint32_t)index);
	else
		return MCALIB_GetHandleByName(collection, name);
}

MCALIB_Handle_t *MCALIB_GetHandleCollectionHandleByType(const MCALIB_Handle_t *handle, CAEN_MCA_HandleType_t type) {
	if (handle == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): NULL handle.", __func__);
		return NULL;
	}

	MCALIB_Handle_t *ret = NULL;

	switch (type) {
	case CAEN_MCA_HANDLE_DEVICE: {
		MCALIB_Library_t *lib = Library;
		if (lib != NULL && lib->devices != NULL)
			ret = lib->devices->commonData.handle;
		break;
	}
	case CAEN_MCA_HANDLE_CHANNEL: {
		MCALIB_Device_t *dev = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);
		if (dev != NULL && dev->channels != NULL)
			ret = dev->channels->commonData.handle;
		break;
	}
	case CAEN_MCA_HANDLE_HVCHANNEL: {
		MCALIB_Device_t *dev = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);
		if (dev != NULL && dev->hvchannels != NULL)
			ret = dev->hvchannels->commonData.handle;
		break;
	}
	case CAEN_MCA_HANDLE_LVDSGROUP: {
		MCALIB_Device_t *dev = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);
		if (dev != NULL && dev->lvdsgroups != NULL)
			ret = dev->lvdsgroups->commonData.handle;
		break;
	}
	case CAEN_MCA_HANDLE_TRACE: {
		MCALIB_Device_t *dev = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);
		if (dev != NULL && dev->traces != NULL)
			ret = dev->traces->commonData.handle;
		break;
	}
	case CAEN_MCA_HANDLE_MONOUT: {
		MCALIB_Device_t *dev = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);
		if (dev != NULL && dev->monouts != NULL)
			ret = dev->monouts->commonData.handle;
		break;
	}
	case CAEN_MCA_HANDLE_HVRANGE: {
		MCALIB_HVChannel_t *hvchannel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_HVCHANNEL);
		if (hvchannel != NULL && hvchannel->hvranges != NULL)
			ret = hvchannel->hvranges->commonData.handle;
		break;
	}
	case CAEN_MCA_HANDLE_ENERGYSPECTRUM: {
		MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);
		if (channel != NULL && channel->energyspectra != NULL)
			ret = channel->energyspectra->commonData.handle;
		break;
	}
	case CAEN_MCA_HANDLE_MCSSPECTRUM: {
		MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);
		if (channel != NULL && channel->mcsspectra != NULL)
			ret = channel->mcsspectra->commonData.handle;
		break;
	}
	case CAEN_MCA_HANDLE_DTSPECTRUM: {
		MCALIB_Device_t *device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);
		if (device != NULL && device->dtspectra != NULL)
			ret = device->dtspectra->commonData.handle;
		break;
	}
	case CAEN_MCA_HANDLE_ROI: {
		MCALIB_EnergySpectrum_t *spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_ENERGYSPECTRUM);
		if (spectrum != NULL && spectrum->rois != NULL)
			ret = spectrum->rois->commonData.handle;
		break;
	}
	case CAEN_MCA_HANDLE_PARAMETER: {
		MCALIB_HandleCollection_t *collection = MCALIB_GetParameterHandleCollection(handle);
		if (collection != NULL)
			ret = collection->commonData.handle;
		break;
	}
	default:
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid collection type (%d).", __func__, type);
		ret = NULL;
	}

	return ret;
}

int32_t MCALIB_TranslateError(int32_t srverr) {
	switch (srverr) {
	case HexagonServer_RetCode_Success:				return CAEN_MCA_RetCode_Success;
	case HexagonServer_RetCode_Generic:				return CAEN_MCA_RetCode_Generic;
	case HexagonServer_RetCode_SockInit:			return CAEN_MCA_RetCode_SockInit;
	case HexagonServer_RetCode_Serialize:			return CAEN_MCA_RetCode_Serialize;
	case HexagonServer_RetCode_Deserialize:			return CAEN_MCA_RetCode_Deserialize;
	case HexagonServer_RetCode_Parameter:			return CAEN_MCA_RetCode_Parameter;
	case HexagonServer_RetCode_ParameterValue:		return CAEN_MCA_RetCode_ParameterValue;
	case HexagonServer_RetCode_Handle:				return CAEN_MCA_RetCode_Handle;
	case HexagonServer_RetCode_Argument:			return CAEN_MCA_RetCode_Argument;
	case HexagonServer_RetCode_NotYetImplemented:	return CAEN_MCA_RetCode_NotYetImplemented;
	case HexagonServer_RetCode_OutOfMemory:			return CAEN_MCA_RetCode_OutOfMemory;
	default:										return CAEN_MCA_RetCode_Generic;
	}
}

int32_t MCALIB_GetDefinitionFile(const MCALIB_Handle_t *handle) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	const uint16_t cmd = CMD_GLOBAL_DEFINITIONS_GET;
	uint16_t recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);
	uint16_t numChunks;
	int8_t *ptr, *content = NULL;
	uint32_t maxsize;
	uint32_t size = 0;

	if (device == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}

	sendBuff = c_createheader(0, cmd, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

	numChunks = totParams - 1;
	maxsize = numChunks * DEFINITIONARRAY_MAXLEN_DEF + 1;
	content = c_malloc(maxsize);
	if (content == NULL) {
		ret = CAEN_MCA_RetCode_OutOfMemory;
		goto QuitFunction;
	}
	ptr = content;
	for (uint16_t i = 0; i < numChunks; i++) {
		uint32_t count;
		if ((pU8 = deserialize_int8_t_array(ptr, DEFINITIONARRAY_MAXLEN_DEF, &count, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
		ptr += count;
		size += count;
	}

	device->definitions = c_xml_newfile_from_memory((char*)content, size);
	if (device->definitions == NULL) {
		ret = CAEN_MCA_RetCode_Generic;
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	c_free(content);
	return ret;
}

int32_t MCALIB_GetBoardInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	const uint16_t cmd = CMD_BOARD_INFO_GET;
	uint16_t recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);

	if (device == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

	if (dataMask & DATAMASK_BRDINFO_MODELNAME) {
		if ((pU8 = deserialize_char(va_arg(args, char*), MODEL_NAME_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_MODEL) {
		if ((pU8 = deserialize_int32_t(va_arg(args, int32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_NCHANNELS) {
		if ((pU8 = deserialize_uint32_t(va_arg(args, uint32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_ROC_FW_MAJ) {
		if ((pU8 = deserialize_uint16_t(va_arg(args, uint16_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_ROC_FW_MIN) {
		if ((pU8 = deserialize_uint16_t(va_arg(args, uint16_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_ROC_FW_BUILD) {
		if ((pU8 = deserialize_uint16_t(va_arg(args, uint16_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_AMC_FW_MAJ) {
		if ((pU8 = deserialize_uint16_t(va_arg(args, uint16_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_AMC_FW_MIN) {
		if ((pU8 = deserialize_uint16_t(va_arg(args, uint16_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_AMC_FW_BUILD) {
		if ((pU8 = deserialize_uint16_t(va_arg(args, uint16_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_LICENSE) {
		if ((pU8 = deserialize_char(va_arg(args, char*), LICENSE_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_SERIALNUM) {
		if ((pU8 = deserialize_uint32_t(va_arg(args, uint32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_FAMCODE) {
		if ((pU8 = deserialize_int32_t(va_arg(args, int32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_NHVCHANNELS) {
		uint32_t *hvchannels = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(hvchannels, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_FORMFACTOR) {
		if ((pU8 = deserialize_int32_t(va_arg(args, int32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_PCBREV) {
		if ((pU8 = deserialize_uint32_t(va_arg(args, uint32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_ADC_BIT_COUNT) {
		if ((pU8 = deserialize_uint32_t(va_arg(args, uint32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_TSAMPLE_PS) {
		if ((pU8 = deserialize_uint32_t(va_arg(args, uint32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_NLVDSGROUPS) {
		if ((pU8 = deserialize_uint32_t(va_arg(args, uint32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_NTRACES) {
		if ((pU8 = deserialize_uint32_t(va_arg(args, uint32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_NMONOUTS) {
		if ((pU8 = deserialize_uint32_t(va_arg(args, uint32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_ENERGY_BIT_COUNT) {
		if ((pU8 = deserialize_uint32_t(va_arg(args, uint32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_LIB_VERSION) {
		if ((pU8 = deserialize_char(va_arg(args, char*), LIB_VERSION_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_BRDINFO_NDTSPECTRA) {
		if ((pU8 = deserialize_uint32_t(va_arg(args, uint32_t*), 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetParameterInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	const uint16_t cmd = CMD_PARAMETER_INFO_GET;
	uint16_t recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Parameter_t *param = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_PARAMETER);

	if (param == NULL || device == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(param->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}

	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

	if (dataMask & DATAMASK_PARAMINFO_NAME) {
		char *name = va_arg(args, char*);
		if ((pU8 = deserialize_char(name, PARAMINFO_NAME_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_CODENAME) {
		char *codename = va_arg(args, char*);
		if ((pU8 = deserialize_char(codename, PARAMINFO_NAME_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_INFOMASK) {
		uint32_t *infobits = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(infobits, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_UOM_NAME) {
		char *name = va_arg(args, char*);
		if ((pU8 = deserialize_char(name, PARAMINFO_NAME_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_UOM_CODENAME) {
		char *codename = va_arg(args, char*);
		if ((pU8 = deserialize_char(codename, PARAMINFO_NAME_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_UOM_POWER) {
		int32_t *uom = va_arg(args, int32_t*);
		if ((pU8 = deserialize_int32_t(uom, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_TYPE) {
		int32_t *type = va_arg(args, int32_t*);
		if ((pU8 = deserialize_int32_t(type, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_MIN) {
		double *param_min = va_arg(args, double*);
		if ((pU8 = deserialize_double(param_min, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_MAX) {
		double *param_max = va_arg(args, double*);
		if ((pU8 = deserialize_double(param_max, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_INCR) {
		double *param_incr = va_arg(args, double*);
		if ((pU8 = deserialize_double(param_incr, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_NALLOWED_VALUES) {
		uint32_t *param_incr = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(param_incr, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_ALLOWED_VALUES) {
		uint32_t count;
		double *allowed_values = va_arg(args, double*);
		if ((pU8 = deserialize_double_array(allowed_values, PARAMINFO_LIST_MAXLEN_DEF, &count, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_ALLOWED_VALUE_CODENAMES) {
		char **value_codenames = va_arg(args, char**);
		uint32_t cnt;
		if ((pU8 = deserialize_uint32_t(&cnt, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
		for (uint32_t i = 0; i < cnt; i++) {
			if ((pU8 = deserialize_char(value_codenames[i], PARAMINFO_NAME_MAXLEN_DEF, pU8)) == NULL) {
				ret = CAEN_MCA_RetCode_Deserialize;
				goto QuitFunction;
			}
		}
	}
	if (dataMask & DATAMASK_PARAMINFO_ALLOWED_VALUE_NAMES) {
		char **value_codenames = va_arg(args, char**);
		uint32_t cnt;
		if ((pU8 = deserialize_uint32_t(&cnt, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
		for (uint32_t i = 0; i < cnt; i++) {
			if ((pU8 = deserialize_char(value_codenames[i], PARAMINFO_NAME_MAXLEN_DEF, pU8)) == NULL) {
				ret = CAEN_MCA_RetCode_Deserialize;
				goto QuitFunction;
			}
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetChannelInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	const uint16_t cmd = CMD_CHANNEL_INFO_GET;
	uint16_t recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_HVChannel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);

	if (channel == NULL || device == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);
	sendBuff = serialize_int32_t(&channel->commonData.index, 1, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

	if (dataMask & DATAMASK_CHANNELINFO_NENERGYSPECTRA) {
		uint32_t* nes = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(nes, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_CHANNELINFO_NMCSSPECTRA) {
		int32_t *nmcss = va_arg(args, int32_t*);
		if ((pU8 = deserialize_int32_t(nmcss, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetHVChannelInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	const uint16_t cmd = CMD_HVCHANNEL_INFO_GET;
	uint16_t recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_HVChannel_t *hvchannel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_HVCHANNEL);

	if (hvchannel == NULL || device == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);
	sendBuff = serialize_int32_t(&hvchannel->commonData.index, 1, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

	if (dataMask & DATAMASK_HVCHANNELINFO_NRANGES) {
		uint32_t* nranges = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(nranges, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVCHANNELINFO_POLARITY) {
		int32_t *polarity = va_arg(args, int32_t*);
		if ((pU8 = deserialize_int32_t(polarity, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetHVRangeInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	const uint16_t cmd = CMD_HVRANGE_INFO_GET;
	uint16_t recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_HVChannel_t *hvchannel = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_HVCHANNEL);
	MCALIB_HVRange_t *range = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_HVRANGE);

	if (range == NULL || hvchannel == NULL || device == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(range->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

	if (dataMask & DATAMASK_HVRANGEINFO_VSET_MIN) {
		double* vset = va_arg(args, double*);
		if ((pU8 = deserialize_double(vset, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_VSET_MAX) {
		double* vset = va_arg(args, double*);
		if ((pU8 = deserialize_double(vset, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_VSET_INCR) {
		double* vset = va_arg(args, double*);
		if ((pU8 = deserialize_double(vset, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_ISET_MIN) {
		double* iset = va_arg(args, double*);
		if ((pU8 = deserialize_double(iset, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_ISET_MAX) {
		double* iset = va_arg(args, double*);
		if ((pU8 = deserialize_double(iset, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_ISET_INCR) {
		double* iset = va_arg(args, double*);
		if ((pU8 = deserialize_double(iset, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_VMON_MIN) {
		double* vmon = va_arg(args, double*);
		if ((pU8 = deserialize_double(vmon, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_VMON_MAX) {
		double* vmon = va_arg(args, double*);
		if ((pU8 = deserialize_double(vmon, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_VMON_INCR) {
		double* vmon = va_arg(args, double*);
		if ((pU8 = deserialize_double(vmon, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_IMON_MIN) {
		double* imon = va_arg(args, double*);
		if ((pU8 = deserialize_double(imon, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_IMON_MAX) {
		double* imon = va_arg(args, double*);
		if ((pU8 = deserialize_double(imon, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_IMON_INCR) {
		double* imon = va_arg(args, double*);
		if ((pU8 = deserialize_double(imon, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_VMAX_MIN) {
		double* vmax = va_arg(args, double*);
		if ((pU8 = deserialize_double(vmax, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_VMAX_MAX) {
		double* vmax = va_arg(args, double*);
		if ((pU8 = deserialize_double(vmax, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_VMAX_INCR) {
		double* vmax = va_arg(args, double*);
		if ((pU8 = deserialize_double(vmax, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_RAMPUP_MIN) {
		double* rampup = va_arg(args, double*);
		if ((pU8 = deserialize_double(rampup, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_RAMPUP_MAX) {
		double* rampup = va_arg(args, double*);
		if ((pU8 = deserialize_double(rampup, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_RAMPUP_INCR) {
		double* rampup = va_arg(args, double*);
		if ((pU8 = deserialize_double(rampup, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_RAMPDOWN_MIN) {
		double* rampdown = va_arg(args, double*);
		if ((pU8 = deserialize_double(rampdown, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_RAMPDOWN_MAX) {
		double* rampdown = va_arg(args, double*);
		if ((pU8 = deserialize_double(rampdown, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_RAMPDOWN_INCR) {
		double* rampdown = va_arg(args, double*);
		if ((pU8 = deserialize_double(rampdown, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_CODE) {
		int32_t* rangeCode = va_arg(args, int32_t*);
		if ((pU8 = deserialize_int32_t(rangeCode, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_CODENAME) {
		char *name = va_arg(args, char*);
		if ((pU8 = deserialize_char(name, HVRANGEINFO_NAME_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_HVRANGEINFO_NAME) {
		char *name = va_arg(args, char*);
		if ((pU8 = deserialize_char(name, HVRANGEINFO_NAME_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetCollectionInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	const MCALIB_HandleCollection_t *collection = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_COLLECTION);
	if (collection == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}

	if (dataMask & DATAMASK_COLLECTION_LENGTH) {
		uint32_t *length = va_arg(args, uint32_t*);
		*length = collection->length;
	}
	if (dataMask & DATAMASK_COLLECTION_HANDLES) {
		CAEN_MCA_HANDLE *par = va_arg(args, CAEN_MCA_HANDLE*);
		for (uint32_t p = 0; p < collection->length; p++) {
			par[p] = (CAEN_MCA_HANDLE)collection->handles[p];
		}
	}

	return CAEN_MCA_RetCode_Success;
}

static int _fillLogMsgPrefix(const MCALIB_Handle_t *handle, char *prefix, size_t maxsize) {
	if (prefix == NULL)
		return -1;

	// If it is called on a NULL pointer
	if (handle == NULL) {
		return snprintf(prefix, maxsize, "[NULL HANDLE]");
	}

	// If it is called on a device or higher level handle
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	if (device != NULL && device->isInitialized) {
		return snprintf(prefix, maxsize, "%s_%" PRIu32 " [%s %s]", device->DeviceInfo.ModelName, device->DeviceInfo.SerialNum, handle->commonData->type->typeName, handle->commonData->name);
	}

	// If it is called on a library handle
	MCALIB_Library_t *lib = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_LIBRARY);
	if (lib != NULL) {
		return snprintf(prefix, maxsize, "%s_%" PRIi32 " [%s %s]", lib->commonData.type->typeName, lib->commonData.index, handle->commonData->type->typeName, handle->commonData->name);
	}

	return -1;
}

int32_t MCALIB_extendedLogMsg(const c_locallogger_t *locallogger, int line, const MCALIB_Handle_t *handle, c_logger_Severity sev, const char *fmt, ...) {
	int32_t ret;
	va_list args;
	char prefix[100];
	int prefixlen = _fillLogMsgPrefix(handle, prefix, sizeof(prefix));
	if (prefixlen < 0)
		return CAEN_MCA_RetCode_Generic;

	size_t maxsize = prefixlen + strlen(fmt) + 3;
	char *fmtExtended = c_malloc(maxsize);
	if (fmtExtended == NULL) {
		c_lprintf(locallogger, sev, line, "Can't allocate string buffer to print message.");
		return CAEN_MCA_RetCode_Generic;
	}
	strcpy(fmtExtended, prefix);
	strcat(fmtExtended, ": ");
	strcat(fmtExtended, fmt);

	va_start(args, fmt);
	ret = c_lvprintf(locallogger, sev, line, fmtExtended, args);
	va_end(args);

	c_free(fmtExtended);

	if (ret == c_logger_Success)
		return CAEN_MCA_RetCode_Success;
	else
		return CAEN_MCA_RetCode_Generic;
}

void MCALIB_DecodeError(const MCALIB_Handle_t *handle, const char *functName, int32_t resCode) {
	const char *ErrMsg;

	switch (resCode) {
	case CAEN_MCA_RetCode_Success:
		return;
	case CAEN_MCA_RetCode_Generic:
		ErrMsg = "Generic error.";
		break;
	case CAEN_MCA_RetCode_SockInit:
		ErrMsg = "Socket initialization error.";
		break;
	case CAEN_MCA_RetCode_SockConnect:
		ErrMsg = "Socket connect error.";
		break;
	case CAEN_MCA_RetCode_OutOfMemory:
		ErrMsg = "Out of memory.";
		break;
	case CAEN_MCA_RetCode_Handle:
		ErrMsg = "Invalid handle.";
		break;
	case CAEN_MCA_RetCode_Argument:
		ErrMsg = "Invalid argument.";
		break;
	case CAEN_MCA_RetCode_SocketSend:
		ErrMsg = "TCP/IP send error.";
		break;
	case CAEN_MCA_RetCode_SocketReceive:
		ErrMsg = "TCP/IP receive error.";
		break;
	case CAEN_MCA_RetCode_Protocol:
		ErrMsg = "Protocol error.";
		break;
	case CAEN_MCA_RetCode_Serialize:
		ErrMsg = "Serialize error.";
		break;
	case CAEN_MCA_RetCode_Deserialize:
		ErrMsg = "Deserialize error.";
		break;
	case CAEN_MCA_RetCode_Parameter:
		ErrMsg = "Parameter error.";
		break;
	case CAEN_MCA_RetCode_ParameterValue:
		ErrMsg = "Parameter value error.";
		break;
	case CAEN_MCA_RetCode_LibraryLoad:
		ErrMsg = "Library dynamic load error.";
		break;
	case CAEN_MCA_RetCode_DiscoveryFunction:
		ErrMsg = "SSDP discovery failed.";
		break;
	case CAEN_MCA_RetCode_NotConnected:
		ErrMsg = "Not connected.";
		break;
	case CAEN_MCA_RetCode_NotSupported:
		ErrMsg = "Not supported.";
		break;
	case CAEN_MCA_RetCode_NotYetImplemented:
		ErrMsg = "Not yet implemented.";
		break;
	case CAEN_MCA_RetCode_CollectionFull:
		ErrMsg = "Handle collection full.";
		break;
	default:
		ErrMsg = "Generic error.";
		break;
	}
	MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): %s", functName, ErrMsg);
}

int32_t MCALIB_GetParameterValue(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	const uint16_t cmd = CMD_PARAMETER_GET;
	uint16_t recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Parameter_t *param = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_PARAMETER);

	if (param == NULL || device == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): invalid handle.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(param->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}
	if (dataMask & DATAMASK_VALUE_NUMERIC) {
		double *numeric = va_arg(args, double*);
		if ((pU8 = deserialize_double(numeric, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_VALUE_CODENAME) {
		char *literal = va_arg(args, char*);
		if ((pU8 = deserialize_char(literal, PARAMINFO_NAME_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_VALUE_NAME) {
		char *name = va_arg(args, char*);
		if ((pU8 = deserialize_char(name, PARAMINFO_NAME_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success) {
		// if board is not initialized and retcode is
		// CAEN_MCA_RetCode_Parameter, do not log the error (we are
		// building the supported parameters list).
		MCALIB_Device_t *dev = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
		if ((dev != NULL && dev->isInitialized) || (ret != CAEN_MCA_RetCode_Parameter))
			MCALIB_DecodeError(handle, __func__, ret);
	}
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_SetParameterValue(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_PARAMETER_SET, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Parameter_t *param = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_PARAMETER);

	if (param == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(param->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);

	if (dataMask & DATAMASK_VALUE_NUMERIC) {
		double numeric = va_arg(args, double);
		if ((sendBuff = serialize_double(&numeric, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_VALUE_CODENAME) {
		char *literal = va_arg(args, char*);
		if ((sendBuff = serialize_char(literal, PARAMINFO_NAME_MAXLEN_DEF, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_ClearEnergySpectrum(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	c_unused_parameter(args);
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_ENERGY_SPECTRUM_CLEAR, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_CHANNEL);
	MCALIB_EnergySpectrum_t *spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_ENERGYSPECTRUM);

	if (spectrum == NULL || channel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(spectrum->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_SetEnergySpectrumInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_ENERGY_SPECTRUM_SET, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_CHANNEL);
	MCALIB_EnergySpectrum_t *spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_ENERGYSPECTRUM);

	if (spectrum == NULL || channel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(spectrum->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);
	
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_FILENAME) {
		char *filename = va_arg(args, char*);
		if ((sendBuff = serialize_char(filename, ENERGYSPECTRUM_FULLPATH_MAXLEN_DEF, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_AUTOSAVE_PERIOD) {
		uint32_t period = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&period, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_STABILIZER_ENABLE) {
		uint32_t enable = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&enable, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_SAMPLE_ID) {
		char *sample_id = va_arg(args, char*);
		if ((sendBuff = serialize_char(sample_id, SAMPLEID_MAXLEN_DEF, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetEnergySpectrumInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_ENERGY_SPECTRUM_GET, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_CHANNEL);
	MCALIB_EnergySpectrum_t *spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_ENERGYSPECTRUM);

	if (spectrum == NULL || channel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(spectrum->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);
	
	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_ARRAY) {
		uint32_t *dataArray = va_arg(args, uint32_t*);
		uint32_t count;
		if ((pU8 = deserialize_uint32_t_array(dataArray, ENERGYSPECTRUM_MAXLEN_DEF, &count, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_RTIME) {
		uint64_t* rTime = va_arg(args, uint64_t*);
		if ((pU8 = deserialize_uint64_t(rTime, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_LTIME) {
		uint64_t* lTime = va_arg(args, uint64_t*);
		if ((pU8 = deserialize_uint64_t(lTime, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_DTIME) {
		uint64_t* dTime = va_arg(args, uint64_t*);
		if ((pU8 = deserialize_uint64_t(dTime, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_OVERFLOW) {
		uint32_t* ovflows = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(ovflows, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_UNDERFLOW) {
		uint32_t* undflows = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(undflows, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_NENTRIES) {
		uint64_t* nentries = va_arg(args, uint64_t*);
		if ((pU8 = deserialize_uint64_t(nentries, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_NROIS) {
		uint32_t* nrois = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(nrois, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_FILENAME) {
		char *filename = va_arg(args, char*);
		if ((pU8 = deserialize_char(filename, ENERGYSPECTRUM_FULLPATH_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_AUTOSAVE_PERIOD) {
		uint32_t* period = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(period, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_STABILIZER_ENABLE) {
		uint32_t* status = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(status, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_SAMPLE_ID) {
		char *sample_id = va_arg(args, char*);
		if ((pU8 = deserialize_char(sample_id, SAMPLEID_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_START_DATETIME) {
		char* datetime = va_arg(args, char*);
		if ((pU8 = deserialize_char(datetime, DATETIME_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_ENERGY_SPECTRUM_STOP_DATETIME) {
		char* datetime = va_arg(args, char*);
		if ((pU8 = deserialize_char(datetime, DATETIME_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetROIInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_ROI_INFO_GET, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_CHANNEL);
	MCALIB_EnergySpectrum_t *spectrum = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_ENERGYSPECTRUM);
	MCALIB_ROI_t *roi = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_ROI);

	if (spectrum == NULL || channel == NULL || device == NULL || roi == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(roi->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}
	if (dataMask & DATAMASK_ROI_NENTRIES) {
		uint64_t* nentries = va_arg(args, uint64_t*);
		if ((pU8 = deserialize_uint64_t(nentries, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

//get index from index (the wanted index/traces is/are selected through datamask)
int32_t MCALIB_GetWaveform(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_WAVEFORM_GET, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);

	if (channel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);
	sendBuff = serialize_int32_t(&channel->commonData.index, 1, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}
	if (dataMask & DATAMASK_WAVEFORM_TRACE1) {
		int16_t *SampleData = va_arg(args, int16_t*);
		uint32_t nsamples;
		if ((pU8 = deserialize_int16_t_array(SampleData, WAVEFORM_MAXLEN_DEF, &nsamples, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_WAVEFORM_TRACE2) {
		int16_t *SampleData = va_arg(args, int16_t*);
		uint32_t count;
		if ((pU8 = deserialize_int16_t_array(SampleData, WAVEFORM_MAXLEN_DEF, &count, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_WAVEFORM_DTRACE1) {
		uint8_t *SampleData = va_arg(args, uint8_t*);
		uint32_t count;
		if ((pU8 = deserialize_uint8_t_array(SampleData, WAVEFORM_MAXLEN_DEF, &count, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_WAVEFORM_DTRACE2) {
		uint8_t *SampleData = va_arg(args, uint8_t*);
		uint32_t count;
		if ((pU8 = deserialize_uint8_t_array(SampleData, WAVEFORM_MAXLEN_DEF, &count, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_SetListMode(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_LIST_SET, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);

	if (channel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);
	sendBuff = serialize_int32_t(&channel->commonData.index, 1, sendBuff, &sendSize);

	if (dataMask & DATAMASK_LIST_ENABLE) {
		uint32_t enable = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&enable, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_SAVEMODE) {
		int32_t mode = va_arg(args, int32_t);
		if ((sendBuff = serialize_int32_t(&mode, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_FILENAME) {
		char *filename = va_arg(args, char*);
		if ((sendBuff = serialize_char(filename, LISTS_FULLPATH_MAXLEN_DEF, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_FILE_DATAMASK) {
		uint32_t mask = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&mask, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_GETFAKEEVTS) {
		uint32_t getfake = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&getfake, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_MAXNEVTS) {
		uint32_t size = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&size, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_GETSATEVTS) {
		uint32_t getsat = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&getsat, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_GETSKIMEVTS) {
		uint32_t getskim = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&getskim, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetListMode(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_LIST_GET, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);

	if (channel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);
	sendBuff = serialize_int32_t(&channel->commonData.index, 1, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}
	if (dataMask & DATAMASK_LIST_ENABLE) {
		uint32_t *enable = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(enable, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_SAVEMODE) {
		int32_t *mode = va_arg(args, int32_t*);
		if ((pU8 = deserialize_int32_t(mode, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_FILENAME) {
		char *filename = va_arg(args, char*);
		if ((pU8 = deserialize_char(filename, LISTS_FULLPATH_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_FILE_DATAMASK) {
		uint32_t* mask = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(mask, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_GETFAKEEVTS) {
		uint32_t* getfake = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(getfake, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_MAXNEVTS) {
		uint32_t* maxnevts = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(maxnevts, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_NEVTS) {
		uint32_t* nevts = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(nevts, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_DATA_TIMETAG) {
		uint64_t *data = va_arg(args, uint64_t*);
		uint32_t count;
		if ((pU8 = deserialize_uint64_t_array(data, LISTS_DATA_MAXLEN_DEF, &count, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_DATA_ENERGY) {
		uint16_t *data = va_arg(args, uint16_t*);
		uint32_t count;
		if ((pU8 = deserialize_uint16_t_array(data, LISTS_DATA_MAXLEN_DEF, &count, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_DATA_FLAGS_DATAMASK) {
		uint32_t *data = va_arg(args, uint32_t*);
		uint32_t count;
		if ((pU8 = deserialize_uint32_t_array(data, LISTS_DATA_MAXLEN_DEF, &count, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_GETSATEVTS) {
		uint32_t* getsat = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(getsat, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_LIST_GETSKIMEVTS) {
		uint32_t* getskim = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(getskim, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetMCSSpectrumInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_MCS_SPECTRUM_GET, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_CHANNEL);
	MCALIB_MCSSpectrum_t *MCSspectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_MCSSPECTRUM);

	if (MCSspectrum == NULL || channel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(MCSspectrum->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}
	if (dataMask & DATAMASK_MCS_SPECTRUM_ARRAY) {
		uint32_t *dataArray = va_arg(args, uint32_t*);
		uint32_t count;
		if ((pU8 = deserialize_uint32_t_array(dataArray, MCSSPECTRUM_MAXLEN_DEF, &count, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_MCS_SPECTRUM_STATUS) {
		uint8_t* status = va_arg(args, uint8_t*);
		if ((pU8 = deserialize_uint8_t(status, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_MCS_SPECTRUM_CURRENT_PASS) {
		uint32_t* pass = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(pass, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_MCSSweep(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	c_unused_parameter(args);
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_MCS_SWEEP, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_CHANNEL);

	if (channel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);
	sendBuff = serialize_int32_t(&channel->commonData.index, 1, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_ClearMCSSpectrum(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	c_unused_parameter(args);
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_MCS_SPECTRUM_CLEAR, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_CHANNEL);
	MCALIB_EnergySpectrum_t *mcs_spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_MCSSPECTRUM);

	if (mcs_spectrum == NULL || channel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(mcs_spectrum->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_ResetGainStabilizer(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	c_unused_parameter(args);
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_GAIN_STABILIZER_RESET, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_CHANNEL);
	MCALIB_EnergySpectrum_t *spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_ENERGYSPECTRUM);

	if (channel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(spectrum->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;

}


int32_t MCALIB_ClearDTSpectrum(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	c_unused_parameter(args);
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_DT_SPECTRUM_CLEAR, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_DTSpectrum_t *dt_spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DTSPECTRUM);

	if (dt_spectrum == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(dt_spectrum->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_SetDTSpectrumInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_DT_SPECTRUM_SET, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_DTSpectrum_t *dt_spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DTSPECTRUM);

	if (dt_spectrum == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(dt_spectrum->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);

	if (dataMask & DATAMASK_DT_SPECTRUM_ENABLE) {
		uint32_t enable = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&enable, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_DT_SPECTRUM_REFCH) {
		uint32_t ch = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&ch, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetDTSpectrumInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_DT_SPECTRUM_GET, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_DTSpectrum_t *dt_spectrum = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DTSPECTRUM);

	if (dt_spectrum == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&dataMask, 1, sendBuff, &sendSize);

	// Insert indexes
	int32_t parentIndexes[NESTED_HANDLE_MAXLEN] = { 0 };
	uint32_t nIndexes = 0;
	if (MCALIB_GetParentIndexes(dt_spectrum->commonData.handle, parentIndexes, &nIndexes) != CAEN_MCA_RetCode_Success) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "%s(): parent indexes.", __func__);
		ret = CAEN_MCA_RetCode_Handle;
		goto QuitFunction;
	}
	sendBuff = serialize_int32_t(parentIndexes, nIndexes, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}
	if (dataMask & DATAMASK_DT_SPECTRUM_ARRAY) {
		uint64_t *dataArray = va_arg(args, uint64_t*);
		uint32_t count;
		if ((pU8 = deserialize_uint64_t_array(dataArray, DTSPECTRUM_MAXLEN_DEF, &count, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_DT_SPECTRUM_NENTRIES) {
		uint64_t* nentries = va_arg(args, uint64_t*);
		if ((pU8 = deserialize_uint64_t(nentries, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_DT_SPECTRUM_ENABLE) {
		uint32_t* enable = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(enable, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (dataMask & DATAMASK_DT_SPECTRUM_REFCH) {
		uint32_t* ch = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(ch, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_SampleAdvance(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	c_unused_parameter(args);
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_SAMPLE_ADVANCE, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);

	if (device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);
	sendBuff = serialize_int32_t(&channel->commonData.index, 1, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;

}

int32_t MCALIB_GetRegisterValue(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_BOARD_REGISTER_READ, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);

	if (device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	if (cmdMaskIn & DATAMASK_CMD_REG_ADDR) {
		uint32_t addr = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&addr, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	else {
		ret = CAEN_MCA_RetCode_Argument;
		goto QuitFunction;
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}
	if (cmdMaskOut & DATAMASK_CMD_REG_DATA) {
		uint32_t *data = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(data, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	else {
		ret = CAEN_MCA_RetCode_Argument;
		goto QuitFunction;
	}


QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_SetRegisterValue(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_BOARD_REGISTER_WRITE, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);

	if (device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	if (cmdMaskIn & DATAMASK_CMD_REG_ADDR) {
		uint32_t addr = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&addr, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	else {
		ret = CAEN_MCA_RetCode_Argument;
		goto QuitFunction;
	}
	if (cmdMaskIn & DATAMASK_CMD_REG_DATA) {
		uint32_t data = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&data, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	else {
		ret = CAEN_MCA_RetCode_Argument;
		goto QuitFunction;
	}
	if (cmdMaskIn & DATAMASK_CMD_REG_MASK) {
		uint32_t mask = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&mask, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetIndex(const MCALIB_Handle_t *handle) {
	if (handle == NULL) {
		logMsg(c_logger_Severity_ERROR, "Invalid or NULL handle.");
		return CAEN_MCA_INVALID_INDEX;
	}
	return handle->commonData->index;
}

int32_t MCALIB_GetParentIndexes(const MCALIB_Handle_t *handle, int32_t parentIndexes[NESTED_HANDLE_MAXLEN], uint32_t *length) {
	if (handle == NULL) {
		logMsg(c_logger_Severity_ERROR, "Invalid or NULL handle.");
		return CAEN_MCA_RetCode_Handle;
	}

	const MCALIB_Handle_t *parentHandle = handle;
	uint32_t i = 0;

	while (parentHandle != NULL) {
		// Write the parent indexes up to CAEN_MCA_HANDLE_DEVICE (excluded).
		if (parentHandle->commonData->type->typeCode == CAEN_MCA_HANDLE_DEVICE)
			break;
		if (i >= NESTED_HANDLE_MAXLEN) {
			MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "Too many parents (%"PRIu32"). Maximum allowed: '%d'", i, NESTED_HANDLE_MAXLEN);
			return CAEN_MCA_RetCode_Handle;
		}
		parentIndexes[i++] = MCALIB_GetIndex(parentHandle);
		parentHandle = parentHandle->commonData->parentHandle;
	}

	*length = i;
	return CAEN_MCA_RetCode_Success;
}

MCALIB_HandleCollection_t* MCALIB_CreateHandleCollection(const MCALIB_Handle_t* parent, CAEN_MCA_HandleType_t type) {

	// For MCALIB_HandleCollection_t the index is the value of the CAEN_MCA_HandleType_t enumerator.
	MCALIB_HandleCollection_t* object = MCALIB_CreateObject(&MCALIB_Handle_Collection, parent, (int32_t)type);
	if (object == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
		return NULL;
	}

	// Initialize additional data after commonData.
	object->handles = NULL;
	object->map = NULL;
	object->length = 0;

	// Initialize map
	object->map = c_malloc(sizeof(*object->map));
	if (object->map == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot allocate map.", __func__);
		MCALIB_DeleteHandleCollection(object);
		return NULL;
	}

	const uint32_t initial_nbuckets = UINT32_C(128);

	if (c_map_init(object->map, initial_nbuckets) != c_Utility_ErrorCode_Success) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot initialize map.", __func__);
		MCALIB_DeleteHandleCollection(object);
		return NULL;
	}

	return object;
}

void MCALIB_DeleteHandleCollection(MCALIB_HandleCollection_t* object) {
	if (object == NULL)
		return;

	c_map_deinit(object->map);
	c_free(object->map);
	c_free(object->handles);
	MCALIB_DeleteObject(object);
}

MCALIB_Handle_t* MCALIB_GetHandleAt(const MCALIB_HandleCollection_t* coll, uint32_t index) {
	if (coll == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): NULL collection.", __func__);
		return NULL;
	}

	if (index < coll->length) {
		return coll->handles[index];
	}
	else {
		MCALIB_LogMsgExt(coll->commonData.handle, c_logger_Severity_ERROR, "%s(): invalid index %"PRIu32".", __func__, index);
		return NULL;
	}
}

MCALIB_Handle_t* MCALIB_GetHandleByName(const MCALIB_HandleCollection_t* coll, const char* name) {
	if (coll == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): NULL collection.", __func__);
		return NULL;
	}

	if (name == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): NULL name.", __func__);
		return NULL;
	}

	MCALIB_Handle_t** phandle = c_map_get(coll->map, name);
	if (phandle != NULL) {
		return *phandle;
	}
	else {
		MCALIB_LogMsgExt(coll->commonData.handle, c_logger_Severity_ERROR, "%s(): key %s not found in map.", __func__, name);
		return NULL;
	}
}

int32_t MCALIB_GetHandleCollectionLength(const MCALIB_HandleCollection_t* coll, uint32_t* length) {
	if (coll == NULL || length == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): invalid or NULL pointer.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}

	*length = coll->length;
	return CAEN_MCA_RetCode_Success;
}

int32_t MCALIB_AddHandleTo(MCALIB_HandleCollection_t* coll, MCALIB_Handle_t* handle) {
	if (coll == NULL || handle == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): error adding handle to collection.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}

	// Set a max size to collection.
	if (coll->length > COLLECTION_MAXLEN_DEF) {
		logMsg(c_logger_Severity_ERROR, "%s(): collection full. Max size allowed: %"PRIu32".", __func__, COLLECTION_MAXLEN_DEF);
		return CAEN_MCA_RetCode_CollectionFull;
	}

	// Check the if the Handle type is correct.
	// For MCALIB_HandleCollection_t, the index represent the type.
	if (coll->commonData.index != (int32_t)handle->commonData->type->typeCode) {
		logMsg(c_logger_Severity_ERROR, "%s(): incorrect handle type '%s' for this collection.", __func__, handle->commonData->type->typeName);
		return CAEN_MCA_RetCode_Argument;
	}

	// Set collectionIndex to the handle
	handle->commonData->collectionIndex = coll->length;

	MCALIB_Handle_t** newHandles = c_realloc(coll->handles, (coll->length + 1) * sizeof(*coll->handles));
	if (newHandles == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): error reallocating memory for handles.", __func__);
		return CAEN_MCA_RetCode_OutOfMemory;
	}

	coll->handles = newHandles;
	coll->handles[coll->length] = handle;
	coll->length++;

	// Insert the object pointer into the collection map.
	if (c_map_set(coll->map, handle->commonData->name, handle) != c_Utility_ErrorCode_Success) {
		logMsg(c_logger_Severity_ERROR, "%s(): error inserting value in the map.", __func__);
		return CAEN_MCA_RetCode_Map;
	}

	return CAEN_MCA_RetCode_Success;
}

MCALIB_RemovableHandleCollection_t* MCALIB_CreateRemovableHandleCollection(const MCALIB_Handle_t* parent, CAEN_MCA_HandleType_t type) {
	// For MCALIB_HandleCollection_t the index is the value of the CAEN_MCA_HandleType_t enumerator.
	MCALIB_RemovableHandleCollection_t* object = MCALIB_CreateObject(&MCALIB_Handle_Removable_Collection, parent, (int32_t)type);
	if (object == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot create object.", __func__);
		return NULL;
	}
	object->map_name = NULL;
	object->map_index = NULL;

	// Initialize maps
	const uint32_t initial_nbuckets = UINT32_C(128);
	// map_name
	object->map_name = c_malloc(sizeof(*object->map_name));
	if (object->map_name == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot allocate map_name.", __func__);
		MCALIB_DeleteRemovableHandleCollection(object);
		return NULL;
	}
	if (c_map_init(object->map_name, initial_nbuckets) != c_Utility_ErrorCode_Success) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot initialize map.", __func__);
		MCALIB_DeleteRemovableHandleCollection(object);
		return NULL;
	}
	// map_index
	object->map_index = c_malloc(sizeof(*object->map_index));
	if (object->map_index == NULL) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot allocate map_index.", __func__);
		MCALIB_DeleteRemovableHandleCollection(object);
		return NULL;
	}
	if (c_map_init(object->map_index, initial_nbuckets) != c_Utility_ErrorCode_Success) {
		MCALIB_LogMsgExt(parent, c_logger_Severity_ERROR, "%s(): cannot initialize map.", __func__);
		MCALIB_DeleteRemovableHandleCollection(object);
		return NULL;
	}

	return object;
}

void MCALIB_DeleteRemovableHandleCollection(MCALIB_RemovableHandleCollection_t* object) {
	if (object == NULL)
		return;

	if (object->map_name != NULL) {
		c_map_deinit(object->map_name);
		c_free(object->map_name);
	}
	if (object->map_index != NULL) {
		c_map_deinit(object->map_index);
		c_free(object->map_index);
	}
	MCALIB_DeleteObject(object);
}

int32_t MCALIB_AddRemovableHandleToIdx(MCALIB_RemovableHandleCollection_t* coll, MCALIB_Handle_t* handle, int32_t* index) {
	char buf[12];
	if (coll == NULL || handle == NULL || index == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): invalid or NULL pointer.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}

	*index = djb2a(handle->commonData->name);
	fastPrintInteger(*index, buf);
	// Check if we have a clash
	if (c_map_get(coll->map_index, buf) != NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): index already occupied, cannot add this digitizer.", __func__);
		return CAEN_MCA_RetCode_IndexAlreadyUsed;
	}

	// Insert the object pointer into the collection map of indexes.
	if (c_map_set(coll->map_index, buf, handle) != c_Utility_ErrorCode_Success) {
		logMsg(c_logger_Severity_ERROR, "%s(): error inserting value in the map.", __func__);
		return CAEN_MCA_RetCode_Map;
	}

	// Insert the object pointer into the collection map of names.
	if (c_map_set(coll->map_name, handle->commonData->name, handle) != c_Utility_ErrorCode_Success) {
		logMsg(c_logger_Severity_ERROR, "%s(): error inserting value in the map.", __func__);
		return CAEN_MCA_RetCode_Map;
	}
	return CAEN_MCA_RetCode_Success;
}

MCALIB_Handle_t* MCALIB_GetRemovableHandleAt(const MCALIB_RemovableHandleCollection_t* coll, uint32_t index) {
	char buf[12];
	if (coll == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): NULL collection.", __func__);
		return NULL;
	}

	MCALIB_Handle_t** phandle = c_map_get(coll->map_index, idxToString(buf, (int32_t)index));
	if (phandle != NULL) {
		return *phandle;
	}
	else {
		MCALIB_LogMsgExt(coll->commonData.handle, c_logger_Severity_ERROR, "%s(): index %" PRIu32 " not found in map.", __func__, index);
		return NULL;
	}
}


MCALIB_Handle_t* MCALIB_GetRemovableHandleByName(const MCALIB_RemovableHandleCollection_t* coll, const char* name) {
	if (coll == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): NULL collection.", __func__);
		return NULL;
	}

	if (name == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): NULL name.", __func__);
		return NULL;
	}

	MCALIB_Handle_t** phandle = c_map_get(coll->map_name, name);
	if (phandle != NULL) {
		return *phandle;
	}
	else {
		MCALIB_LogMsgExt(coll->commonData.handle, c_logger_Severity_ERROR, "%s(): key %s not found in map.", __func__, name);
		return NULL;
	}
}

int32_t MCALIB_GetRemovableHandleCollectionLength(const MCALIB_RemovableHandleCollection_t* coll, uint32_t* length) {
	size_t sz;
	sz = c_map_size(coll->map_name);
	*length = (uint32_t)sz;
	return CAEN_MCA_RetCode_Success;
}

// No need to add index here, the caller can get it from handle->commonData->collectionIndex
int32_t MCALIB_RemoveHandleFrom(MCALIB_RemovableHandleCollection_t* coll, MCALIB_Handle_t* handle) {
	char buf[12];
	if (coll == NULL || handle == NULL) {
		logMsg(c_logger_Severity_ERROR, "%s(): NULL collection or handle provided.", __func__);
		return CAEN_MCA_RetCode_Argument;
	}
	// Remove object pointer from collection map
	c_map_remove(coll->map_name, handle->commonData->name);
	c_map_remove(coll->map_index, idxToString(buf, djb2a(handle->commonData->name)));
	// check
	if (c_map_size(coll->map_name) != c_map_size(coll->map_index)) {
		logMsg(c_logger_Severity_ERROR, "%s(): ERROR! Inconsistency in device map, library may misbehave!", __func__);
		return CAEN_MCA_RetCode_Generic;
	}
	return CAEN_MCA_RetCode_Success;
}

int32_t MCALIB_AcquisitionStart(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	c_unused_parameter(args);
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_CHANNEL_ACQ_START, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	uint32_t chmask = 0;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);

	if (device == NULL)
		return CAEN_MCA_RetCode_Argument;

	// set the chmask
	if (channel == NULL) {
		chmask = ~UINT32_C(0);
	}
	else {
		chmask |= (1 << channel->commonData.index);
	}

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint32_t(&chmask, 1, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_AcquisitionStop(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	c_unused_parameter(args);
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_CHANNEL_ACQ_STOP, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	uint32_t chmask = 0;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);

	if (device == NULL)
		return CAEN_MCA_RetCode_Argument;

	// set the chmask
	if (channel == NULL) {
		chmask = ~UINT32_C(0);
	}
	else {
		chmask |= (1 << channel->commonData.index);
	}

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint32_t(&chmask, 1, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_HVOn(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	c_unused_parameter(args);
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_HVCHANNEL_ON, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_HVChannel_t *hvchannel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_HVCHANNEL);

	if (hvchannel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);
	sendBuff = serialize_int32_t(&hvchannel->commonData.index, 1, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_HVOff(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	c_unused_parameter(args);
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_HVCHANNEL_OFF, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_HVChannel_t *hvchannel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_HVCHANNEL);

	if (hvchannel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);
	sendBuff = serialize_int32_t(&hvchannel->commonData.index, 1, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_HVOnOff(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_HVCHANNEL_ONOFF, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_HVChannel_t *hvchannel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_HVCHANNEL);

	if (hvchannel == NULL || device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);
	sendBuff = serialize_int32_t(&hvchannel->commonData.index, 1, sendBuff, &sendSize);

	if (cmdMaskIn & DATAMASK_CMD_HVOUTPUT_STATUS) {
		uint32_t newStatus = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&newStatus, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

	if (cmdMaskOut & DATAMASK_CMD_HVOUTPUT_STATUS) {
		uint32_t *status = va_arg(args, uint32_t*);
		if ((pU8 = deserialize_uint32_t(status, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GlobalQuit(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	c_unused_parameter(args);
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_GLOBAL_QUIT, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);

	if (device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_DatabasePath(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_GLOBAL_DATABASE_PATH, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);

	if (device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	if (cmdMaskIn & DATAMASK_CMD_SAVE_DB_PATH) {
		char *pathname = va_arg(args, char*);
		if ((sendBuff = serialize_char(pathname, CONFIGSAVE_FULLPATH_MAXLEN_DEF, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}
	if (cmdMaskOut & DATAMASK_CMD_SAVE_DB_PATH) {
		char *pathname = va_arg(args, char*);
		if ((pU8 = deserialize_char(pathname, CONFIGSAVE_FULLPATH_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_SaveConfiguration(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_GLOBAL_CONFIG_SAVE, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);

	if (device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	if (cmdMaskIn & DATAMASK_CMD_SAVE_NAME) {
		char *savename = va_arg(args, char*);
		if ((sendBuff = serialize_char(savename, CONFIGSAVE_NAME_MAXLEN_DEF, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_LoadConfiguration(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_GLOBAL_CONFIG_LOAD, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);

	if (device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	if (cmdMaskIn & DATAMASK_CMD_SAVE_NAME) {
		char *savename = va_arg(args, char*);
		if ((sendBuff = serialize_char(savename, CONFIGSAVE_NAME_MAXLEN_DEF, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (cmdMaskIn & DATAMASK_CMD_SAVE_HV_RELOAD) {
		uint32_t loadHV = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&loadHV, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetConfigurationList(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_GLOBAL_CONFIG_LIST, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);

	if (device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	if (cmdMaskIn & DATAMASK_CMD_SAVE_LIST_OFFSET) {
		uint32_t offset = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&offset, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}

	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

	if (cmdMaskOut & DATAMASK_CMD_SAVE_LIST_COUNT) {
		int32_t *cnt = va_arg(args, int32_t*);
		if ((pU8 = deserialize_int32_t(cnt, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}
	if (cmdMaskOut & DATAMASK_CMD_SAVE_LIST_NAMES) {
		char **savenames = va_arg(args, char**);
		uint32_t cnt;
		if ((pU8 = deserialize_uint32_t(&cnt, 1, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
		for (uint32_t i = 0; i < cnt; i++) {
			if ((pU8 = deserialize_char(savenames[i], CONFIGSAVE_NAME_MAXLEN_DEF, pU8)) == NULL) {
				ret = CAEN_MCA_RetCode_Deserialize;
				goto QuitFunction;
			}
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_DeleteConfiguration(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_GLOBAL_CONFIG_DELETE, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);

	if (device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	if (cmdMaskIn & DATAMASK_CMD_SAVE_NAME) {
		char *savename = va_arg(args, char*);
		if ((sendBuff = serialize_char(savename, CONFIGSAVE_NAME_MAXLEN_DEF, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}


QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_ParamAutoSetStart(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_PARAM_AUTOSET_START, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);

	if (device == NULL || channel == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);
	sendBuff = serialize_int32_t(&channel->commonData.index, 1, sendBuff, &sendSize);

	if (cmdMaskIn & DATAMASK_CMD_AUTOSET_DCO) {
		uint32_t enable = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&enable, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (cmdMaskIn & DATAMASK_CMD_AUTOSET_THR) {
		uint32_t enable = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&enable, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (cmdMaskIn & DATAMASK_CMD_AUTOSET_TAU) {
		uint32_t enable = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&enable, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_ParamAutoSetStop(const MCALIB_Handle_t *handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_PARAM_AUTOSET_STOP, recvCmd;
	uint8_t *sendBuff = NULL;
	uint8_t *pU8 = NULL, *recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t *device = MCALIB_GetAncestor(handle, CAEN_MCA_HANDLE_DEVICE);
	MCALIB_Channel_t *channel = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_CHANNEL);

	if (device == NULL || channel == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);
	sendBuff = serialize_int32_t(&channel->commonData.index, 1, sendBuff, &sendSize);

	if (cmdMaskIn & DATAMASK_CMD_AUTOSET_DCO) {
		uint32_t enable = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&enable, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (cmdMaskIn & DATAMASK_CMD_AUTOSET_THR) {
		uint32_t enable = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&enable, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}
	if (cmdMaskIn & DATAMASK_CMD_AUTOSET_TAU) {
		uint32_t enable = va_arg(args, uint32_t);
		if ((sendBuff = serialize_uint32_t(&enable, 1, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}

int32_t MCALIB_GetHandleInfo(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;

	if (handle == NULL) {
		ret = CAEN_MCA_RetCode_Argument;
		goto QuitFunction;
	}

	if (dataMask & DATAMASK_HANDLE_TYPE) {
		int32_t *type = va_arg(args, int32_t*);
		*type = (int32_t)handle->commonData->type->typeCode;
	}
	if (dataMask & DATAMASK_HANDLE_INDEX) {
		int32_t *index = va_arg(args, int32_t*);
		*index = handle->commonData->collectionIndex;
	}
	if (dataMask & DATAMASK_HANDLE_NAME) {
		char *name = va_arg(args, char*);
		c_memcpy(name, handle->commonData->name, sizeof(handle->commonData->name));
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	return ret;
}

int32_t MCALIB_RTClock(const MCALIB_Handle_t* handle, uint64_t cmdMaskIn, uint64_t cmdMaskOut, va_list args) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	size_t sendSize;
	uint16_t cmd = CMD_RTCLOCK_DATETIME, recvCmd;
	uint8_t* sendBuff = NULL;
	uint8_t* pU8 = NULL, * recvBuff = NULL;
	uint32_t totSize;
	uint16_t totParams;
	int16_t srvret;
	MCALIB_Device_t* device = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_DEVICE);

	if (device == NULL)
		return CAEN_MCA_RetCode_Argument;

	sendBuff = c_createheader(0, cmd, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskIn, 1, sendBuff, &sendSize);
	sendBuff = serialize_uint64_t(&cmdMaskOut, 1, sendBuff, &sendSize);

	if (cmdMaskIn & DATAMASK_CMD_CURRENT_DATETIME) {
		char* datetime = va_arg(args, char*);
		if ((sendBuff = serialize_char(datetime, DATETIME_MAXLEN_DEF, sendBuff, &sendSize)) == NULL) {
			ret = CAEN_MCA_RetCode_Serialize;
			goto QuitFunction;
		}
	}

	if (c_send_lock(device->socket, sendBuff, sendSize) != (c_ssize_t)sendSize) {
		ret = CAEN_MCA_RetCode_SocketSend;
		goto QuitFunction;
	}
	pU8 = recvBuff = c_recv_packet_unlock(device->socket, &recvCmd, &totSize, &totParams);
	if (pU8 == NULL) {
		ret = CAEN_MCA_RetCode_SocketReceive;
		goto QuitFunction;
	}
	else if (recvCmd != cmd) {
		ret = CAEN_MCA_RetCode_Protocol;
		goto QuitFunction;
	}
	// Deserialize the result code from buffer
	if ((pU8 = deserialize_int16_t(&srvret, 1, pU8)) == NULL) {
		ret = CAEN_MCA_RetCode_Deserialize;
		goto QuitFunction;
	}
	if (srvret != HexagonServer_RetCode_Success) {
		ret = MCALIB_TranslateError(srvret);
		goto QuitFunction;
	}
	if (cmdMaskOut & DATAMASK_CMD_CURRENT_DATETIME) {
		char* pathname = va_arg(args, char*);
		if ((pU8 = deserialize_char(pathname, DATETIME_MAXLEN_DEF, pU8)) == NULL) {
			ret = CAEN_MCA_RetCode_Deserialize;
			goto QuitFunction;
		}
	}

QuitFunction:
	if (ret != CAEN_MCA_RetCode_Success)
		MCALIB_DecodeError(handle, __func__, ret);
	c_free(recvBuff);
	c_free(sendBuff);
	return ret;
}
