#include <MCALIB_Discovery.h>

#include <stdio.h>

#include <CAENMultiplatform.h>
#include <CAENXMLParser.h>
#include <CAENThread.h>
#include <CAENUrl.h>

#ifdef _WIN32 // Windows

#ifndef COBJMACROS
#define COBJMACROS	(1) // Enable the C declarations of functiondiscoveryapi.h
#endif

#include <functiondiscovery.h>
#pragma comment(lib, "ole32.lib") // TODO: add to Visual Studio settings

typedef struct CMyFDHelper {
	IFunctionDiscovery *m_pFunDisc;
	IFunctionDiscoveryNotification *m_pFunDiscNot;
} CMyFDHelper;

CMyFDHelper * NewCMyFDHelper(void);
void DeleteCMyFDHelper(CMyFDHelper *fd);

HRESULT ListFunctionInstances(CMyFDHelper *fd, const WCHAR *pszCategory, uint32_t timeout_ms);

#else // Linux

#include <dlfcn.h>
#include <libgssdp/gssdp.h>
#include <gio/gio.h>

#endif

INIT_C_LOGGER("MCALIB_DiscoveryLog.txt", "MCALIB_Discovery.c");

static c_mutex_t ssdpMutex;
static char **gXmlList;
static uint32_t gXmlListCnt;

#ifdef _WIN32 // Windows only functions

// This method is the main notification sink
// In this implementation it displays the name of the
// added removed or modify device
static HRESULT STDMETHODCALLTYPE OnUpdate(__RPC__in IFunctionDiscoveryNotification *m_pFunDiscNot, QueryUpdateAction enumQueryUpdateAction, FDQUERYCONTEXT fdqcQueryContext, __RPC__in_opt IFunctionInstance *pIFunctionInstance) {
	UNREFERENCED_PARAMETER(fdqcQueryContext);
	UNREFERENCED_PARAMETER(m_pFunDiscNot);
	if (NULL == pIFunctionInstance)
		return E_INVALIDARG;

	IPropertyStore *pStore;

	PROPVARIANT pvName, pvSerial, pvLocation;
	PropVariantInit(&pvName);
	PropVariantInit(&pvSerial);
	PropVariantInit(&pvLocation);

	// Open the property store
	HRESULT hr = IFunctionInstance_OpenPropertyStore(pIFunctionInstance, STGM_READ, &pStore);

	// In PnP the device's friendly name could be in one of the
	// following property keys. Check each.
	if (SUCCEEDED(hr))
		hr = IPropertyStore_GetValue(pStore, &PKEY_Device_FriendlyName, &pvName);
	if (SUCCEEDED(hr))
		hr = IPropertyStore_GetValue(pStore, &PKEY_PNPX_SerialNumber, &pvSerial);
	if (SUCCEEDED(hr))
		hr = IPropertyStore_GetValue(pStore, &PKEY_Device_LocationInfo, &pvLocation);

	if (SUCCEEDED(hr) && (pvName.vt == VT_LPWSTR) && (pvSerial.vt == VT_LPWSTR) && (pvLocation.vt == VT_LPWSTR)) {
		if (enumQueryUpdateAction == QUA_ADD) {
			char temp[HEXAGONDISCOVERY_NAME_MAXLEN_DEF];
			char myName[HEXAGONDISCOVERY_NAME_MAXLEN_DEF] = { '\0' };
			char mySerial[HEXAGONDISCOVERY_NAME_MAXLEN_DEF] = { '\0' };
			char myLocation[HEXAGONDISCOVERY_NAME_MAXLEN_DEF] = { '\0' };

			// Convert from wchar_t to char*
			// Does not provide null termination if the max number of characters is written.
			// We use strncat to attach null terminator.
			wcstombs(temp, pvName.pwszVal, sizeof(temp));
			strncat(myName, temp, sizeof(myName) - 1);
			wcstombs(temp, pvSerial.pwszVal, sizeof(temp));
			strncat(mySerial, temp, sizeof(mySerial) - 1);
			wcstombs(temp, pvLocation.pwszVal, sizeof(temp));
			strncat(myLocation, temp, sizeof(myLocation) - 1);

			// Check myLocation. Skip non-Hexagon devices.
			// We can't check myName on Linux on the callback function, so we here we check myLocation only too.
			if (strstr(myLocation, SSDP_HEXAGON_FILENAME) == NULL)
				return hr;

			// Fill the list and increment the counter.
			if (gXmlListCnt < HEXAGONDISCOVERY_LIST_MAXLEN_DEF) {
				c_mutex_lock(&ssdpMutex);
				strcpy(gXmlList[gXmlListCnt++], myLocation);
				c_mutex_unlock(&ssdpMutex);
			}

		}
	}

	PropVariantClear(&pvName);
	IPropertyStore_Release(pStore);

	return hr;
}

// Required for the implementation of IFunctionDiscoveryNotification
static HRESULT STDMETHODCALLTYPE OnError(__RPC__in IFunctionDiscoveryNotification *m_pFunDiscNot, HRESULT hr, FDQUERYCONTEXT fdqcQueryContext, __RPC__in_string const WCHAR *pszProvider) {
	UNREFERENCED_PARAMETER(fdqcQueryContext);
	UNREFERENCED_PARAMETER(m_pFunDiscNot);

	if (pszProvider != NULL) {
		size_t size = wcslen(pszProvider);
		char *temp = c_malloc(size + 1);
		if (temp == NULL)
			return E_OUTOFMEMORY;
		wcstombs(temp, pszProvider, size);

		c_mutex_lock(&ssdpMutex);
		logMsg(c_logger_Severity_DEBUG, "%s encountered 0x%lx.", temp, hr);
		c_mutex_unlock(&ssdpMutex);

		c_free(temp);
	}

	return S_OK;
}

// Required for the implementation of IFunctionDiscoveryNotification
static HRESULT STDMETHODCALLTYPE OnEvent(__RPC__in IFunctionDiscoveryNotification *m_pFunDiscNot, DWORD dwEventID, FDQUERYCONTEXT fdqcQueryContext, __RPC__in_string const WCHAR *pszProvider) {
	UNREFERENCED_PARAMETER(dwEventID);
	UNREFERENCED_PARAMETER(fdqcQueryContext);
	UNREFERENCED_PARAMETER(m_pFunDiscNot);

	if (pszProvider != NULL) {
		size_t size = wcslen(pszProvider);
		char *temp = c_malloc(size + 1);
		if (temp == NULL)
			return E_OUTOFMEMORY;
		wcstombs(temp, pszProvider, size);

		c_mutex_lock(&ssdpMutex);
		logMsg(c_logger_Severity_DEBUG, "%s sent OnEvent notification.", temp);
		c_mutex_unlock(&ssdpMutex);

		c_free(temp);
	}

	return S_OK;
}

static HRESULT STDMETHODCALLTYPE QueryInterface(__RPC__in IFunctionDiscoveryNotification *m_pFunDiscNot, __RPC__in REFIID riid, _COM_Outptr_  void **ppvObject) {
	UNREFERENCED_PARAMETER(m_pFunDiscNot);
	UNREFERENCED_PARAMETER(riid);
	UNREFERENCED_PARAMETER(ppvObject);
	return S_OK;
}

static ULONG STDMETHODCALLTYPE AddRef(__RPC__in IFunctionDiscoveryNotification *m_pFunDiscNot) {
	UNREFERENCED_PARAMETER(m_pFunDiscNot);
	return S_OK;
}

static ULONG STDMETHODCALLTYPE Release(__RPC__in IFunctionDiscoveryNotification *m_pFunDiscNot) {
	UNREFERENCED_PARAMETER(m_pFunDiscNot);
	return S_OK;
}

// DiscoveryThread function should be called on a new thread, as it need to initialize the COM libraries.
static DWORD WINAPI DiscoveryThread(LPVOID lpParam) {
	int32_t ret = CAEN_MCA_RetCode_Success;
	bool isComInitialized = FALSE;
	CMyFDHelper *fd = NULL;
	uint32_t *timeout_ms = (uint32_t*)lpParam;

	// Initializes the COM library to COINIT_MULTITHREADED for use by the calling thread
	HRESULT hr = CoInitializeEx(NULL, COINIT_MULTITHREADED);
	switch (hr) {
	case S_OK:
		isComInitialized = TRUE;
		break;
	case S_FALSE:
		// To close the COM library gracefully on a thread, each successful call to CoInitialize or CoInitializeEx,
		// including any call that returns S_FALSE, must be balanced by a corresponding call to CoUninitialize.
		isComInitialized = TRUE;
		logMsg(c_logger_Severity_ERROR, "The COM library is already initialized on this thread. (CoInitializeEx returned %ld)", hr);
		break;
	case RPC_E_CHANGED_MODE:
		isComInitialized = FALSE;
		logMsg(c_logger_Severity_WARNING, "A previous call to CoInitializeEx specified the concurrency model for this thread as multithread apartment (MTA) (CoInitializeEx returned %ld)", hr);
		ret = CAEN_MCA_RetCode_DiscoveryFunction;
		goto QuitFunction;
	default:
		logMsg(c_logger_Severity_ERROR, "Generic error initializing COM library. (CoInitializeEx returned %ld)", hr);
		ret = CAEN_MCA_RetCode_DiscoveryFunction;
		goto QuitFunction;
	}

	// Create a new Discovery
	fd = NewCMyFDHelper();
	if (fd == NULL) {
		logMsg(c_logger_Severity_ERROR, "Can't create a FunctionDiscover helper.");
		ret = CAEN_MCA_RetCode_DiscoveryFunction;
		goto QuitFunction;
	}

	// Enumerate all SSDP Devices
	hr = ListFunctionInstances(fd, FCTN_CATEGORY_SSDP, *timeout_ms);
	if (FAILED(hr)) {
		logMsg(c_logger_Severity_ERROR, "Error in SSDP devices discovery: %ld", hr);
		ret = CAEN_MCA_RetCode_DiscoveryFunction;
		goto QuitFunction;
	}

QuitFunction:
	// Close stuff
	DeleteCMyFDHelper(fd);

	// Close the COM library on the current thread
	if (isComInitialized)
		CoUninitialize();

	return (DWORD)ret;
}

// Object constructor
CMyFDHelper *NewCMyFDHelper(void) {
	HRESULT hr = S_OK;

	CMyFDHelper *fd = c_malloc(sizeof(*fd));
	if (fd == NULL) {
		hr = E_FAIL;
		goto QuitFunction;
	}

	// Creates a single uninitialized object of the class associated with a specified CLSID.
	hr = CoCreateInstance(&CLSID_FunctionDiscovery,
		NULL,
		CLSCTX_ALL,
		&IID_IFunctionDiscovery,
		(LPVOID*)&fd->m_pFunDisc);

	if (FAILED(hr)) {
		goto QuitFunction;
	}

	// Initialize the notification system
	fd->m_pFunDiscNot = c_malloc(sizeof(*fd->m_pFunDiscNot));
	if (fd->m_pFunDiscNot == NULL) {
		hr = E_FAIL;
		goto QuitFunction;
	}

	fd->m_pFunDiscNot->lpVtbl = c_malloc(sizeof(*fd->m_pFunDiscNot->lpVtbl));
	if (fd->m_pFunDiscNot->lpVtbl == NULL) {
		hr = E_FAIL;
		goto QuitFunction;
	}

	// Set pointers to local functions
	IFunctionDiscoveryNotificationVtbl * const vtbl = fd->m_pFunDiscNot->lpVtbl;
	vtbl->QueryInterface = &QueryInterface;
	vtbl->AddRef = &AddRef;
	vtbl->Release = &Release;
	vtbl->OnUpdate = &OnUpdate;
	vtbl->OnError = &OnError;
	vtbl->OnEvent = &OnEvent;

QuitFunction:
	if (FAILED(hr)) {
		DeleteCMyFDHelper(fd);
		fd = NULL;
	}

	return fd;
}

// Object destructor
void DeleteCMyFDHelper(CMyFDHelper *fd) {
	if (fd == NULL)
		return;

	// Delete the FunDisc
	if (fd->m_pFunDisc != NULL) {
		c_mutex_lock(&ssdpMutex);
		IFunctionDiscovery_Release(fd->m_pFunDisc);
		c_mutex_unlock(&ssdpMutex);
	}

	if (fd->m_pFunDiscNot != NULL) {
		c_free(fd->m_pFunDiscNot->lpVtbl);
		c_free(fd->m_pFunDiscNot);
	}
	c_free(fd);
}

// Outputs a list of all instances in the Category
HRESULT ListFunctionInstances(CMyFDHelper *fd, const WCHAR *pszCategory, uint32_t timeout_ms) {
	HRESULT hr = S_OK;

	IFunctionInstanceCollectionQuery *pQuery = NULL;
	IFunctionInstanceCollection *pCollection = NULL;

	// Create an instance collection query. This CMyFDHelper class
	// implements CFunctionDiscoveryNotificationWrapper and
	// therefore IFunctionDiscoveryNotification. One of the parameters
	// to CreateInstanceCollectionQuery is a IFunctionDiscoveryNotification
	// pointer. This object is sent query events.
	if (SUCCEEDED(hr))
		hr = IFunctionDiscovery_CreateInstanceCollectionQuery(fd->m_pFunDisc,
			pszCategory,
			NULL,
			TRUE,
			fd->m_pFunDiscNot,
			NULL,
			&pQuery);

	if (SUCCEEDED(hr))
		hr = IFunctionInstanceCollectionQuery_AddQueryConstraint(pQuery,
			PROVIDERSSDP_QUERYCONSTRAINT_TYPE,
			SSDP_CONSTRAINTVALUE_TYPE_ROOT);

	// Execute the query. If it's a network query
	// hr will be set to E_PENDING and the collection will be
	// empty. All instances from a network query are returned
	// via notifications.
	if (SUCCEEDED(hr))
		hr = IFunctionInstanceCollectionQuery_Execute(pQuery, &pCollection);

	// If it's a network query, we expected E_PENDING from the above
	// call to pQuery->Execute( &pCollection )
	// Instances will be returned via notifications, return S_OK
	if (hr == E_PENDING)
		hr = S_OK;

	Sleep((DWORD)timeout_ms);

	if (pQuery != NULL)
		IFunctionInstanceCollectionQuery_Release(pQuery);
	if (pCollection != NULL)
		IFunctionInstanceCollection_Release(pCollection);

	return hr;
}


#else // Linux

//Library handles
void *libGSSDP, *libGOBJECT, *libGLIB;

// GSSDP function pointers
GSSDPClient * (*my_gssdp_client_new) (GMainContext *, const char *, GError **);
GSSDPResourceBrowser * (*my_gssdp_resource_browser_new) (GSSDPClient *, const char*);
void (*my_gssdp_resource_browser_set_active) (GSSDPResourceBrowser *, gboolean);

// GLIB function pointers
void (*my_g_error_free) (GError *);
GMainLoop * (*my_g_main_loop_new) (GMainContext *, gboolean);
guint (*my_g_timeout_add_full) (gint, guint, GSourceFunc, gpointer, GDestroyNotify);
void (*my_g_main_loop_run) (GMainLoop*);
void (*my_g_main_loop_quit) (GMainLoop*);
void (*my_g_main_loop_unref) (GMainLoop*);

// GOBJECT function pointers
gulong (*my_g_signal_connect_data) (gpointer, const gchar*, GCallback, gpointer, GClosureNotify, GConnectFlags);
void (*my_g_object_unref) (gpointer);

void MCALIB_LoadGSSDPLibraries(void) {
	char *error;

	// Load libraries needed by GSSDP to perform device discovery
	// There is no need to unload them at exit.
	libGSSDP = dlopen("libgssdp-1.0.so.3", RTLD_LAZY);
	error = dlerror();
	if (error != NULL) {
		logMsg(c_logger_Severity_WARNING, "Can't load libgssdp-1.0.so.3, trying another version...");
		libGSSDP = NULL;
		libGSSDP = dlopen("libgssdp-1.2.so", RTLD_LAZY);
		error = dlerror();
		if (error != NULL) {
			logMsg(c_logger_Severity_WARNING, "Can't load libgssdp-1.2.so.0");
			libGSSDP = NULL;
			error = NULL;
		}
	}

	libGOBJECT = dlopen("libgobject-2.0.so.0", RTLD_LAZY);
	error = dlerror();
	if (error != NULL) {
		logMsg(c_logger_Severity_WARNING, "Can't load libgobject-2.0.so.0");
		libGOBJECT = NULL;
		error = NULL;
 	}
	libGLIB = dlopen("libglib-2.0.so.0", RTLD_LAZY);
	error = dlerror();
	if (error != NULL) {
		logMsg(c_logger_Severity_WARNING, "Can't load libglib-2.0.so.0");
		libGLIB = NULL;
		error = NULL;
	}

}

void MCALIB_CloseGSSDPLibraries(void) {
	char *error = NULL;
	int ret;

	// Unload libraries needed by GSSDP to perform device discovery
	ret = dlclose(libGSSDP);
	if (ret != 0) {
		error = dlerror();
		if (error != NULL) {
			logMsg(c_logger_Severity_WARNING, "Can't close libgssdp-1.0.so.3");
			error = NULL;
		}
	}
	else {
		libGSSDP = NULL;
	}

	ret = dlclose(libGOBJECT);
	if (ret != 0) {
		error = dlerror();
		if (error != NULL) {
			logMsg(c_logger_Severity_WARNING, "Can't close libgobject-2.0.so.0");
			error = NULL;
		}
	}
	else {
		libGOBJECT = NULL;
	}

	ret = dlclose(libGLIB);
	if (ret != 0) {
		error = dlerror();
		if (error != NULL) {
			logMsg(c_logger_Severity_WARNING, "Can't close libglib-2.0.so.0");
			error = NULL;
		}
	}
	else {
		libGLIB = NULL;
	}

}

static void resource_available_cb(G_GNUC_UNUSED GSSDPResourceBrowser *resource_browser, G_GNUC_UNUSED const char *usn, GList *locations) {

	char myLocation[HEXAGONDISCOVERY_NAME_MAXLEN_DEF];

	// Local copy
	if (locations == NULL)
		return;

	myLocation[0] = '\0';
	strncat(myLocation, (char*)locations->data, sizeof(myLocation) - 1);

	// Check myLocation. Skip non-Hexagon devices.
	if (strstr(myLocation, SSDP_HEXAGON_FILENAME) == NULL)
		return;

	// Fill the list and increment the counter.
	if (gXmlListCnt < HEXAGONDISCOVERY_LIST_MAXLEN_DEF) {
		c_mutex_lock(&ssdpMutex);
		strcpy(gXmlList[gXmlListCnt++], myLocation);
		c_mutex_unlock(&ssdpMutex);
	}
}

// Function called when times returns FALSE
static void quit_main_loop(gpointer data) {
	my_g_main_loop_quit((GMainLoop *)data);
}

// Callback function: must return false to stop the timer
gboolean timer(G_GNUC_UNUSED gpointer data) {
	return FALSE;
}

#endif

static int32_t _getSSDPProperties(xmlNode *node, char *friendly_name, char *modelname, uint32_t *sn, uint32_t *nch) {
	if (node == NULL) {
		return CAEN_MCA_RetCode_DiscoveryFunction;
	}

	for (xmlNode *cur_node = node; cur_node; cur_node = cur_node->next) {
		if (cur_node->type == XML_ELEMENT_NODE) {
			if (xmlStrEqual(cur_node->name, BAD_CAST "friendlyName")) {
				xmlChar *content = xmlNodeGetContent(cur_node);
				friendly_name[0] = '\0';
				strncat(friendly_name, (char*)content, HEXAGONDISCOVERY_NAME_MAXLEN_DEF - 1);
				xmlFree(content);
			}
			else if (xmlStrEqual(cur_node->name, BAD_CAST "modelName")) {
				xmlChar *content = xmlNodeGetContent(cur_node);
				modelname[0] = '\0';
				strncat(modelname, (char*)content, HEXAGONDISCOVERY_NAME_MAXLEN_DEF - 1);
				xmlFree(content);
			}
			else if (xmlStrEqual(cur_node->name, BAD_CAST "serialNumber")) {
				xmlChar *content = xmlNodeGetContent(cur_node);
				*sn = (uint32_t)strtoul((char*)content, NULL, 0);
				xmlFree(content);
			}
			else if (xmlStrEqual(cur_node->name, BAD_CAST "inputChannels")) {
				xmlChar *content = xmlNodeGetContent(cur_node);
				*nch = (uint32_t)strtoul((char*)content, NULL, 0);
				xmlFree(content);
			}
		}
	}

	return CAEN_MCA_RetCode_Success;
}

int32_t MCALIB_SetDiscoveredDevices(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	MCALIB_Library_t *lib = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_LIBRARY);
	if (lib == NULL)
		return CAEN_MCA_RetCode_Argument;

	int32_t ret = CAEN_MCA_RetCode_Success;

	// Get the timeout if passed as parameter
	if (dataMask & DATAMASK_DISCOVERY_TIMEOUT_MS) {
		lib->discoverTimeout_ms = va_arg(args, uint32_t);
	}

	return ret;
}

int32_t MCALIB_GetDiscoveredDevices(const MCALIB_Handle_t *handle, uint64_t dataMask, va_list args) {
	MCALIB_Library_t *lib = MCALIB_GetObject(handle, CAEN_MCA_HANDLE_LIBRARY);
	if (lib == NULL)
		return CAEN_MCA_RetCode_Argument;

	int32_t ret = CAEN_MCA_RetCode_Success;

	// Shared list
	char *xmlList[HEXAGONDISCOVERY_LIST_MAXLEN_DEF];

	for (size_t i = 0; i < c_arraysize(xmlList); i++) {
		xmlList[i] = c_malloc(HEXAGONDISCOVERY_NAME_MAXLEN_DEF);
		if (xmlList[i] == NULL)
			return CAEN_MCA_RetCode_OutOfMemory;
	}

	gXmlList = xmlList;
	gXmlListCnt = 0;

	// Define a mutex
	c_mutex_init(&ssdpMutex);

#ifdef _WIN32 // Windows

	// Start the search on a different thread: we need to use COINIT_MULTITHREADED as concurrency model for the COM libraries,
	// but the COM libraries should not be initialized in a thread not created by the current DLL library.
	// This because the caller program could already have set the concurrency model to something different from COINIT_MULTITHREADED,
	// the only one compatible with the functiondiscovery features.
	HANDLE thread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)&DiscoveryThread, &lib->discoverTimeout_ms, 0, NULL);
	if (thread == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "Unable to create thread.");
		ret = CAEN_MCA_RetCode_DiscoveryFunction;
		goto QuitFunction;
	}

	DWORD thread_end_ret = WaitForSingleObject(thread, INFINITE);

	DWORD thread_ret;
	GetExitCodeThread(thread, &thread_ret);

	if (thread_ret != CAEN_MCA_RetCode_Success || thread_end_ret == WAIT_FAILED) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "Error on DiscoveryThread.");
		ret = CAEN_MCA_RetCode_DiscoveryFunction;
		goto QuitFunction;
	}

#else //Linux

	// GLib stuff
	char *ferror;
	GError *error = NULL;
	GSSDPClient * client = NULL;
	GSSDPResourceBrowser *resource_browser = NULL;
	GMainLoop *main_loop = NULL;

	if (libGSSDP == NULL || libGLIB == NULL || libGOBJECT == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "GSSDP library not loaded.");
		ret = CAEN_MCA_RetCode_LibraryLoad;
		goto QuitFunction;
	}

	// *(void**) casts required to avoid warnings about pointer function conversion not allowed
	// when using -pedantic
	// see https://stackoverflow.com/questions/31526876/casting-when-using-dlsym for details

	// Load GSSDP functions
	*(void**)&my_gssdp_client_new = dlsym(libGSSDP, "gssdp_client_new");
	ferror = dlerror();
	if (ferror != NULL) {
		ret = CAEN_MCA_RetCode_LibraryLoad;
		goto QuitFunction;
	}
	*(void**)&my_gssdp_resource_browser_new = dlsym(libGSSDP, "gssdp_resource_browser_new");
	ferror = dlerror();
	if (ferror != NULL) {
		ret = CAEN_MCA_RetCode_LibraryLoad;
		goto QuitFunction;
	}
	*(void**)&my_gssdp_resource_browser_set_active = dlsym(libGSSDP, "gssdp_resource_browser_set_active");
	ferror = dlerror();
	if (ferror != NULL) {
		ret = CAEN_MCA_RetCode_LibraryLoad;
		goto QuitFunction;
	}

	// Load GLIB functions
	*(void**)&my_g_error_free = dlsym(libGLIB, "g_error_free");
	ferror = dlerror();
	if (ferror != NULL) {
		ret = CAEN_MCA_RetCode_LibraryLoad;
		goto QuitFunction;
	}
	*(void**)&my_g_main_loop_new = dlsym(libGLIB, "g_main_loop_new");
	ferror = dlerror();
	if (ferror != NULL) {
		ret = CAEN_MCA_RetCode_LibraryLoad;
		goto QuitFunction;
	}
	*(void**)&my_g_timeout_add_full = dlsym(libGLIB, "g_timeout_add_full");
	ferror = dlerror();
	if (ferror != NULL) {
		ret = CAEN_MCA_RetCode_LibraryLoad;
		goto QuitFunction;
	}
	*(void**)&my_g_main_loop_run = dlsym(libGLIB, "g_main_loop_run");
	ferror = dlerror();
	if (ferror != NULL) {
		ret = CAEN_MCA_RetCode_LibraryLoad;
		goto QuitFunction;
	}
	*(void**)&my_g_main_loop_quit = dlsym(libGLIB, "g_main_loop_quit");
	ferror = dlerror();
	if (ferror != NULL) {
		ret = CAEN_MCA_RetCode_LibraryLoad;
		goto QuitFunction;
	}
	*(void**)&my_g_main_loop_unref = dlsym(libGLIB, "g_main_loop_unref");
	ferror = dlerror();
	if (ferror != NULL) {
		ret = CAEN_MCA_RetCode_LibraryLoad;
		goto QuitFunction;
	}

	// Load GOBJECT functions
	*(void**)&my_g_signal_connect_data = dlsym(libGOBJECT, "g_signal_connect_data");
	ferror = dlerror();
	if (ferror != NULL) {
		ret = CAEN_MCA_RetCode_LibraryLoad;
		goto QuitFunction;
	}
	*(void**)&my_g_object_unref = dlsym(libGOBJECT, "g_object_unref");
	ferror = dlerror();
	if (ferror != NULL) {
		ret = CAEN_MCA_RetCode_LibraryLoad;
		goto QuitFunction;
	}

	// Let's go!
	client = my_gssdp_client_new(NULL, NULL, &error);
	if (error) {
		my_g_error_free(error);
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "Error in gssdp_client_new");
		ret = CAEN_MCA_RetCode_DiscoveryFunction;
		goto QuitFunction;
	}

	/* GSSDPResourceBrowser handles resource discovery. After creating a browser and activating it,
	 * the ::resource-available and ::resource-unavailable signals will be emitted
	 * whenever the availability of a resource matching the specified discovery target changes.
	 * A discovery request is sent out automatically when activating the browser.
	 */

	// Define a new resource browser
	resource_browser = my_gssdp_resource_browser_new(client, "upnp:rootdevice");
	if (resource_browser == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "Error in gssdp_resource_browser_new");
		ret = CAEN_MCA_RetCode_DiscoveryFunction;
		goto QuitFunction;
	}

	// Connect a callback to resource_available_cb when the signal "resource-available" occurs
	my_g_signal_connect_data(resource_browser, "resource-available", G_CALLBACK(resource_available_cb), NULL, NULL, (GConnectFlags) 0);

	// Activate resource_browser
	my_gssdp_resource_browser_set_active(resource_browser, TRUE);

	// Creates a new GMainLoop structure
	main_loop = my_g_main_loop_new(NULL, FALSE);
	if (main_loop == NULL) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "Error in g_main_loop_new");
		ret = CAEN_MCA_RetCode_DiscoveryFunction;
		goto QuitFunction;
	}

	// Add a timeout to call the function "func" after lib->discoverTimeout_ms. If "timer" returns false, "quit_main_loop" is called too.
	my_g_timeout_add_full(G_PRIORITY_DEFAULT, lib->discoverTimeout_ms, timer, main_loop, quit_main_loop);

	// Start the main_loop. The function waits here until g_main_loop_quit is called by "quit_main_loop"
	my_g_main_loop_run(main_loop);

#endif

QuitFunction:

#ifdef _WIN32 // Windows

	if (thread != NULL)
		CloseHandle(thread);

#else // Linux

	// Close stuff
	if (main_loop != NULL)
		my_g_main_loop_unref(main_loop);
	if (resource_browser != NULL)
		my_g_object_unref(resource_browser);
	if (client != NULL)
		my_g_object_unref(client);

#endif

	// Destroy the mutex
	c_mutex_destroy(&ssdpMutex);

	xmlChar xDevicePath[] = "/*[name()='root']/*[name()='device']";

	char friendly_name[HEXAGONDISCOVERY_NAME_MAXLEN_DEF];
	char model_name[HEXAGONDISCOVERY_NAME_MAXLEN_DEF];
	uint32_t sn;
	char ip[HEXAGONDISCOVERY_IP_MAXLEN_DEF];
	uint32_t nch;

	char **friendly_name_list = NULL;
	uint32_t *sn_list = NULL;
	char **ip_list = NULL;
	uint32_t *nch_list = NULL;
	char **model_name_list = NULL;
	char **path_name_list = NULL;

	if (dataMask & DATAMASK_DISCOVERY_TIMEOUT_MS) {
		uint32_t *timeout = va_arg(args, uint32_t*);
		*timeout = lib->discoverTimeout_ms;
	}
	if (dataMask & DATAMASK_DISCOVERY_FOUNDCOUNT) {
		uint32_t *cnt = va_arg(args, uint32_t*);
		*cnt = gXmlListCnt;
	}
	if (dataMask & DATAMASK_DISCOVERY_NAME) {
		friendly_name_list = va_arg(args, char**);
	}
	if (dataMask & DATAMASK_DISCOVERY_SERIALNUMBER) {
		sn_list = va_arg(args, uint32_t*);
	}
	if (dataMask & DATAMASK_DISCOVERY_IP) {
		ip_list = va_arg(args, char**);
	}
	if (dataMask & DATAMASK_DISCOVERY_INPUT_CHANNELS) {
		nch_list = va_arg(args, uint32_t*);
	}
	if (dataMask & DATAMASK_DISCOVERY_MODEL_NAME) {
		model_name_list = va_arg(args, char**);
	}
	if (dataMask & DATAMASK_DISCOVERY_PATH) {
		path_name_list = va_arg(args, char**);
	}

	// Loop on results
	for (uint32_t i = 0; i < gXmlListCnt; i++) {
		// xmlReadFile retrieves both file path and URL
		c_xmlfile_t *xmlFile = c_xml_newfile(xmlList[i]);
		if (xmlFile != NULL) {
			xmlNodeSet *deviceObject = c_xml_getnodeset(xmlFile, xDevicePath);
			if (deviceObject != NULL) {
				// Reset data
				friendly_name[0] = '\0';
				model_name[0] = '\0';
				sn = 0;
				ip[0] = '\0';
				nch = 0;

				// Get properties from XML
				_getSSDPProperties(xmlXPathNodeSetItem(deviceObject, 0)->children, friendly_name, model_name, &sn, &nch);

				// Get hostname
				c_url_field_t *url = c_url_parse(xmlList[i]);
				if (url != NULL && url->host != NULL)
					strncat(ip, url->host, sizeof(ip) - 1);
				c_url_free(url);

				// Copy to client pointer
				if (friendly_name_list != NULL)
					if (friendly_name_list[i] != NULL)
						strcpy(friendly_name_list[i], friendly_name);
				if (model_name_list != NULL)
					if (model_name_list[i] != NULL)
						strcpy(model_name_list[i], model_name);
				if (path_name_list != NULL)
					if (path_name_list[i] != NULL)
						snprintf(path_name_list[i], HEXAGONDISCOVERY_NAME_MAXLEN_DEF, "eth://%s", ip);
				if (sn_list != NULL)
					sn_list[i] = sn;
				if (ip_list != NULL)
					if (ip_list[i] != NULL)
						strcpy(ip_list[i], ip);
				if (nch_list != NULL)
					nch_list[i] = nch;

				// Free the object
				c_xml_freenodeset(deviceObject);
			}
			c_xml_freefile(xmlFile);
		}
	}

	// Free the list
	for (size_t i = 0; i < c_arraysize(xmlList); i++) {
		c_free(xmlList[i]);
	}

	if (ret == CAEN_MCA_RetCode_LibraryLoad) {
		MCALIB_LogMsgExt(handle, c_logger_Severity_ERROR, "Can't load GDDSP library. This feature requires libgssdp. Please install it and retry.");
	}

	return ret;
}
